#include "stockpile_pipeline.h"
#include "metrics.h"

#include <infra/yasm/stockpile_client/common/base_types.h>

#include <library/cpp/unistat/unistat.h>
#include <util/generic/xrange.h>

using namespace NHistDb;
using namespace NHistDb::NStockpile;

TWritingStockpilePipeline::TWritingStockpilePipeline(
    TLog& logger,
    size_t stockpileThreadCount,
    const NStockpile::TSettings& settings,
    const NYasm::NCommon::TFastConfigSettings& fastConfigSettings,
    const TStockpileWriterSettings& writerSettings
)
    : Logger(logger)
    , FastConfig(Logger, fastConfigSettings)
    , WriterSettings(writerSettings)
    , StockpileState(logger, settings, FastConfig)
    , StockpileSplitter(Logger)
    , WriterStats(stockpileThreadCount)
{
    for (const auto idx : xrange(stockpileThreadCount)) {
        CreateStockpileThread(idx);
    }
}

void TWritingStockpilePipeline::OnRecord(const IRecordDescriptor& recordDescriptor) {
    if (!recordDescriptor.GetSignalName().IsOld()) {
        StockpileSplitter.OnRecord(recordDescriptor);
    }
}

TMaybe<TInstant> TWritingStockpilePipeline::GetLastTime() {
    return StockpileSplitter.GetLastTime();
}

void TWritingStockpilePipeline::CreateStockpileThread(size_t index) {
    StockpileWriters.emplace_back(Logger, StockpileState, WriterSettings, index, WriterStats);
    StockpileSplitter.RegisterVisitor(StockpileWriters.back());
}

void TWritingStockpilePipeline::StartFastConfig() {
    FastConfig.Start();
}

void TWritingStockpilePipeline::StopFastConfig() {
    FastConfig.Stop();
}

void TWritingStockpilePipeline::Start() {
    StockpileState.UpdateShards();
    StockpileState.Start();
    StockpileSplitter.Start();
}

void TWritingStockpilePipeline::Interrupt() {
    StockpileSplitter.Stop();
    StockpileState.Stop();
}

void TWritingStockpilePipeline::Finish() {
    StockpileSplitter.Stop();
    StockpileSplitter.Finish();
    for (auto& writer: StockpileWriters) {
        writer.Finish();
    }
    StockpileState.Stop();
}
