#include "stockpile_state.h"

#include <infra/yasm/histdb/components/dumper/common_ut.h>
#include <infra/yasm/stockpile_client/state_ut.h>

#include <library/cpp/testing/unittest/registar.h>

using namespace NHistDb;
using namespace NHistDb::NStockpile;


NHistDb::TWritingStockpilePipeline GetPipeline(TLog& logger) {
    return {
        logger,
        1,
        NTest::CreateTestSettings(),
        NTest::MakeTestingFastConfigSettings(),
        TStockpileWriterSettings()
    };
}

Y_UNIT_TEST_SUITE(TTestStockpilePipelineState) {
    Y_UNIT_TEST(WriteAndRead) {
        TLog logger;
        auto pipeline = GetPipeline(logger);

        const TFsPath path = "test.tmp";
        const auto timestamp = TInstant::Seconds(45);
        {
            NHistDb::TStockpileDumperState(path, pipeline).SetLastTimeDumper(timestamp);
        }
        {
            auto state = NHistDb::TStockpileDumperState(path, pipeline);
            UNIT_ASSERT_VALUES_EQUAL(state.GetLastTimeDumped(), timestamp);
        }
    }
    Y_UNIT_TEST(NoFile) {
        TLog logger;
        auto pipeline = GetPipeline(logger);

        const TFsPath path = "nofile.tmp";
        auto state = NHistDb::TStockpileDumperState(path, pipeline);
        UNIT_ASSERT_VALUES_EQUAL(state.GetLastTimeDumped(), TInstant::Zero());
    }
}

Y_UNIT_TEST_SUITE(TStockpileSharding) {
    Y_UNIT_TEST(MetabaseShardingConfig) {
        TClusterInfo info {EStockpileClusterType::Testing, "sas"};
        TString config {R"#({"shards": [{"itype": "base", "shards_count": 10}]})#"};
        TSettings settings {info, GetEnv("OAUTH_TOKEN"), true, {config}};

        TLog log {NYasm::NCommon::NTest::CreateLog()};
        TFastConfig fastConfig {log, NTest::MakeTestingFastConfigSettings()};
        NHistDb::NStockpile::TStockpileState stockpileState {log, settings, fastConfig};

        TStockpileDumperStats stockpileDumperStats(1);
        TStockpileQueueProcessor processor(log, stockpileState, TStockpileWriterSettings(), 0, stockpileDumperStats);
        TMockedRecordDescriptor record {};
        record.SetInstanceKey(NTags::TInstanceKey::FromNamed("base"));
        auto key1 = processor.ToShardKey(record);
        UNIT_ASSERT_VALUES_EQUAL(key1.GetCluster(), "group_1");

        record.SetSignalName(TStringBuf("some_other_summ"));
        auto key2 = processor.ToShardKey(record);
        UNIT_ASSERT_VALUES_EQUAL(key2.GetCluster(), "group_5");

        //NOTE(rocco66): configless itype
        record.SetInstanceKey(NTags::TInstanceKey::FromNamed("upper"));
        auto key3 = processor.ToShardKey(record);
        UNIT_ASSERT_VALUES_EQUAL(key3.GetCluster(), "group_0");
    }
}
