#include <infra/yasm/histdb/dumper/lib/handlers.h>
#include <infra/yasm/histdb/dumper/lib/metrics.h>
#include <infra/yasm/histdb/dumper/lib/pipeline.h>
#include <infra/yasm/histdb/dumper/lib/snapshot_reader.h>
#include <infra/yasm/histdb/dumper/lib/settings.h>
#include <infra/yasm/histdb/dumper/lib/stockpile_pipeline.h>
#include <infra/yasm/histdb/dumper/lib/stockpile_state.h>
#include <infra/yasm/stockpile_client/metrics.h>

#include <infra/monitoring/common/application.h>

using namespace NHistDb;
using namespace NMonitoring;
using namespace NZoom::NYasmConf;

class TDumper final: public TBaseApplication<TDumperSettings> {
public:
    TDumper(TLog& logger, const TDumperSettings& settings)
        : TBaseApplication(logger, settings)
        , StockpileThreadCount(settings.GetStockpileThreadCount())
        , WritingStockpilePipeline(
              Logger,
              StockpileThreadCount,
              settings.CreateStockpileSettings(),
              NYasm::NCommon::MakeYasmFastConfigSettings(
                  settings.GetSnapshotsDirectory() / "fast_config.json"
              ),
              TStockpileWriterSettings()
          )
        , HttpHandlers(Logger, WritingStockpilePipeline)
    {
        NStockpile::TGrpcSettings::Init("yasmdumper", logger);
        StockpileDumpState.Reset(new TStockpileDumperState(settings.GetDumpFilePath(), WritingStockpilePipeline));
        auto stockpileStartingOffset = StockpileDumpState->GetLastTimeDumped();
        StockpileSnapshotReader.Reset(new TStockpileSnapshotReader{
            Logger,
            settings.GetSnapshotsDirectory(),
            stockpileStartingOffset,
            WritingStockpilePipeline
        });
        Logger << ELogPriority::TLOG_INFO << "Starting from " << stockpileStartingOffset
               << " for stockpile (with  " << StockpileThreadCount << " stockpile threads)";
    }

    void Run() override {
        // start http server
        HttpHandlers.Register(HttpServer);
        HttpServer.StartServing();

        // starting
        if (StockpileThreadCount > 0) {
            WritingStockpilePipeline.StartFastConfig();
            WritingStockpilePipeline.Start();
            StockpileSnapshotReader->Start();
            if (StockpileDumpState) {
                StockpileDumpState->Start();
            }
        }

        Event.Wait();

        // stopping

        if (StockpileThreadCount > 0) {
            StockpileSnapshotReader->Stop();
            WritingStockpilePipeline.Interrupt();
            WritingStockpilePipeline.StopFastConfig();
        }
    }

    static void FillParser(NLastGetopt::TOpts& options) {
        TDumperSettings::FillParser(options);
    }

    static void ProcessParsedOptions(const NLastGetopt::TOptsParseResult& parsed, TDumperSettings& settings) {
        settings.ProcessParsedOptions(parsed);
        settings.SetClientTimeout(TDuration::Seconds(10));
    }

    THolder<TStatsInitializer> GetStatsInitializer() override {
        return MakeHolder<NMetrics::TDumperStatsInitializer>();
    }

private:
    const size_t StockpileThreadCount;

    TWritingStockpilePipeline WritingStockpilePipeline;
    THolder<TStockpileDumperState> StockpileDumpState;
    THolder<TStockpileSnapshotReader> StockpileSnapshotReader;

    TDumperHandlersCollection HttpHandlers;
};

int main(int argc, const char** argv) {
    try {
        return RunFromMain<TDumper, TDumperSettings>(argc, argv);
    } catch (...) {
        Cerr << CurrentExceptionMessage() << Endl;
        return 1;
    }
}
