#pragma once

#include <infra/yasm/histdb/components/streams/snappy.h>
#include <infra/yasm/histdb/components/formats/something.h>
#include <infra/yasm/zoom/components/record/record.h>

namespace NHistDb {
    class TSimpleSomethingIteratorRow {
    public:
        TSimpleSomethingIteratorRow() = default;

        TSimpleSomethingIteratorRow(TSomethingFormat::TIteratorRow row)
            : Timestamp(std::get<0>(row))
            , Key(std::get<1>(row))
            , Record(MakeHolder<NZoom::NRecord::TRecord>(std::move(std::get<2>(row))))
        {
        }

        const TString& GetKey() {
            return Key.ToNamed();
        }

        NZoom::NRecord::TRecord* ReleaseRecord() {
            return Record.Release();
        }

        ui64 Timestamp = 0;
        NTags::TInstanceKey Key;
        TAutoPtr<NZoom::NRecord::TRecord> Record;
    };

    class TSimpleSomethingIterator {
    public:
        TSimpleSomethingIterator() = default;
        TSimpleSomethingIterator(TSimpleSomethingIterator&&) = default;
        TSimpleSomethingIterator& operator=(TSimpleSomethingIterator&&) = default;

        TSimpleSomethingIterator(TSomethingIterator iterator)
            : Iterator_(std::move(iterator))
        {
        }

        TMaybe<TSimpleSomethingIteratorRow>& Next() {
            if (Iterator_->Next()) {
                Current_.ConstructInPlace(std::move(Iterator_->Get()));
            } else {
                Current_.Clear();
            }
            return Current_;
        }

    private:
        TMaybe<TSomethingIterator> Iterator_;
        TMaybe<TSimpleSomethingIteratorRow> Current_;
    };

    class TSimpleSomethingKeyIteratorRow {
    public:
        TSimpleSomethingKeyIteratorRow() = default;

        TSimpleSomethingKeyIteratorRow(TSomethingFormat::TKeyIteratorRow row)
            : Key(row.first)
            , Signals(row.second)
        {
        }

        const TString& GetKey() {
            return Key.ToNamed();
        }

        TVector<TString> GetSignals() {
            TVector<TString> result;
            for (const auto& signal : *Signals) {
                result.emplace_back(signal.GetName());
            }
            return result;
        }

        NTags::TInstanceKey Key;
        const TVector<NZoom::NSignal::TSignalName>* Signals;
    };

    class TSimpleSomethingKeyIterator {
    public:
        TSimpleSomethingKeyIterator() = default;
        TSimpleSomethingKeyIterator(TSimpleSomethingKeyIterator&&) = default;
        TSimpleSomethingKeyIterator& operator=(TSimpleSomethingKeyIterator&&) = default;

        TSimpleSomethingKeyIterator(TSomethingKeyIterator keyIterator)
            : KeyIterator_(std::move(keyIterator))
        {
        }

        TMaybe<TSimpleSomethingKeyIteratorRow>& Next() {
            if (KeyIterator_->Next()) {
                Current_.ConstructInPlace(std::move(KeyIterator_->Get()));
            } else {
                Current_.Clear();
            }
            return Current_;
        }

    private:
        TMaybe<TSomethingKeyIterator> KeyIterator_;
        TMaybe<TSimpleSomethingKeyIteratorRow> Current_;
    };

    class TSimpleSomethingRow {
    public:
        TSimpleSomethingRow() = default;

        TSimpleSomethingRow(TSomethingFormat::TReadRow row)
            : Timestamp(std::get<0>(row))
            , RequestedIndex(std::get<1>(row))
            , Record(MakeHolder<NZoom::NRecord::TRecord>(std::move(std::get<2>(row))))
        {
        }

        NZoom::NRecord::TRecord* ReleaseRecord() {
            return Record.Release();
        }

        ui64 Timestamp = 0;
        ui64 RequestedIndex = 0;
        TAutoPtr<NZoom::NRecord::TRecord> Record;
    };

    class TReadData {
    public:
        TReadData() = default;
        TReadData(ui64 timeStamp, NZoom::NRecord::TRecord& record, const NTags::TRequestKey& requestKey, NTags::TInstanceKey instanceKey)
            : TimeStamp(timeStamp)
            , RequesterName(requestKey.ToNamed())
            , InstanceName(instanceKey.ToNamed())
            , Record(MakeHolder<NZoom::NRecord::TRecord>(std::move(record)))
        {
        }

        NZoom::NRecord::TRecord* ReleaseRecord() {
            return Record.Release();
        }

    public:
        ui64 TimeStamp;
        TString RequesterName;
        TString InstanceName;
        TAutoPtr<NZoom::NRecord::TRecord> Record;
    };

    class TSimpleSomethingFormat {
    public:
        TSimpleSomethingFormat(
            const TString& fileName, ESnappyMode mode, bool useDirect=true
        );
        TSimpleSomethingFormat(
            TStringBuf incoming, const TString& fileName, ESnappyMode mode, bool useNewTags=true, bool useDirect=true
        );

        TString Dump();

        void WriteRecord(
            ui64 timestamp,
            const TString& key,
            const NZoom::NRecord::TRecord& record
        );

        TVector<bool> HasRecords(
            const TVector<ui64>& timestamps,
            const TVector<TString>& keys
        );

        TVector<TSimpleSomethingRow> ReadRecords(
            const TVector<ui64>& timestamps,
            const TVector<std::pair<TString, TVector<TString>>>& keysAndSignals
        );

        TSimpleSomethingIterator IterateRecords(const TVector<ui64>& timestamps);
        TSimpleSomethingKeyIterator IterateKeys();

        TMaybe<ui64> FirstRecordTime();
        TMaybe<ui64> LastRecordTime();

        void Finish();

        TVector<TReadData> Read(TVector<ui64>& times, TVector<std::pair<TString, TVector<TString>>> &);

    private:
        TAtomicSharedPtr<TSomethingFormat> Format_;
        TSnappyFile File_;
    };
}
