#include "data_requesters.h"

#include <infra/monitoring/common/perf.h>
#include <infra/yasm/zoom/components/containers/accumulator_mapping.h>

using namespace NHistDb;
using namespace NTags;
using namespace NZoom::NProtobuf;
using namespace NZoom::NHost;
using namespace NZoom::NAccumulators;
using namespace NZoom::NYasmConf;
using namespace NZoom::NContainers;
using namespace NZoom::NSignal;
using namespace NZoom::NSubscription;
using namespace NZoom::NValue;
using namespace NYasm::NCommon;

namespace {
    using EStatusCode = NYasm::NInterfaces::NInternal::THistoryAggregatedSeries::EStatusCode;

    class THistDbExecutor : public TBaseThreadedExecutor {
        Y_DECLARE_SINGLETON_FRIEND()
    public:
        static THistDbExecutor& Get() {
            return *SingletonWithPriority<THistDbExecutor, 100001>();
        }

    protected:
        THistDbExecutor()
            : TBaseThreadedExecutor("THistDbExecutor", 8)
        {
        }
    };

    class TSeriesMerger {
    public:
        static TSeriesMerger Create(const THistoryRequest& request, TAccumulatorMapping& mapping) {
            try {
                auto accumulator(mapping.GetAccumulatorType(request.RequestKey.GetRequestKey().GetItype(), request.SignalName));
                if (accumulator.Defined()) {
                    return TSeriesMerger(request, *accumulator);
                } else {
                    return TSeriesMerger(request, EStatusCode::THistoryAggregatedSeries_EStatusCode_NOT_FOUND);
                }
            } catch(...) {
                return TSeriesMerger(request, EStatusCode::THistoryAggregatedSeries_EStatusCode_INTERNAL_ERROR);
            }
        }

        bool IsValid() const {
            return StatusCode == EStatusCode::THistoryAggregatedSeries_EStatusCode_OK;
        }

        const TRecordPeriod& GetPeriod() const {
            return Period;
        }

        const THistoryRequest& GetRequest() const {
            return Request;
        }

        void ComputeOffsets(TInstant chunkTime, size_t& start, size_t& end) const {
            if (Request.Start <= chunkTime) {
                start = 0;
            } else {
                start = Min(start, Period.GetOffset(Request.Start));
            }
            if (Request.End >= chunkTime + Period.GetInterval()) {
                end = Period.GetRecordsPerFile();
            } else {
                end = Max(end, Period.GetOffset(Request.End) + 1);
            }
        }

        void Mul(TInstant timestamp, TValueRef value) {
            if (timestamp < Request.Start || timestamp > Request.End) {
                return;
            }
            const size_t offset = (timestamp - Request.Start).GetValue() / Period.GetResolution().GetValue();

            TGuard<TAdaptiveLock> guard(Lock);
            Accumulators.Mul(value, offset);
        }

        THistoryResponse CreateResponse() {
            TGuard<TAdaptiveLock> guard(Lock);
            TVector<NZoom::NValue::TValue> values(Reserve(Accumulators.Len()));
            for (const auto idx : xrange(Accumulators.Len())) {
                values.emplace_back(Accumulators.GetValue(idx));
            }
            if (values.empty() && StatusCode == EStatusCode::THistoryAggregatedSeries_EStatusCode_OK) {
                StatusCode = EStatusCode::THistoryAggregatedSeries_EStatusCode_NOT_FOUND;
            }
            Accumulators.Clean();
            return THistoryResponse(Request, Request.Start, std::move(values), StatusCode);
        }

        void SetStatusCode(EStatusCode statusCode) {
            TGuard<TAdaptiveLock> guard(Lock);
            StatusCode = statusCode;
        }

    private:
        TSeriesMerger(const THistoryRequest& request, EAccumulatorType accumulatorType)
            : Request(request)
            , Period(TRecordPeriod::Get(Request.Period))
            , Accumulators(accumulatorType, Request.GetPoints())
            , StatusCode(EStatusCode::THistoryAggregatedSeries_EStatusCode_OK)
        {
        }

        TSeriesMerger(const THistoryRequest& request, EStatusCode statusCode)
            : Request(request)
            , Period(TRecordPeriod::Get("s5"))
            , Accumulators(EAccumulatorType::Summ, Request.GetPoints())
            , StatusCode(statusCode)
        {
        }

        const THistoryRequest& Request;
        const TRecordPeriod& Period;
        TCompactAccumulatorsArray Accumulators;
        EStatusCode StatusCode;
        TAdaptiveLock Lock;
    };

    struct THeaderAccumulator {

        void SetStatusCode(EStatusCode statusCode) {
            for (auto& [requestKey, signals] : Mergers) {
                for (auto& [signal, mergers] : signals) {
                    for (auto* merger : mergers) {
                        merger->SetStatusCode(statusCode);
                    }
                }
            }
        }

        void MakeRequest(TSecondHeaderCache& headerCache, const TString& root, TInstant deadline, NMonitoring::TRequestLog& logger) {
            if (TInstant::Now() > deadline) {
                SetStatusCode(EStatusCode::THistoryAggregatedSeries_EStatusCode_TIME_EXCEEDED);
                return;
            }

            size_t signalsSize = 0;

            size_t start = Period.GetRecordsPerFile();
            size_t end = 0;
            TSomethingFormat::TTagSignals tagSignals;
            TVector<TInternedRequestKey> requestKeys(Reserve(tagSignals.size()));
            for (auto& [requestKey, signals] : Mergers) {
                tagSignals.emplace_back(TRequestKey::FromString(requestKey.GetName()), TVector<TSignalName>{});
                requestKeys.emplace_back(requestKey);
                signalsSize += signals.size();
                for (auto& [signal, mergers] : signals) {
                    tagSignals.back().second.emplace_back(signal);
                    for (auto* merger : mergers) {
                        merger->ComputeOffsets(ChunkTime, start, end);
                    }
                }
            }

            TVector<TSomethingFormat::TTimestamp> times(Reserve(end - start));
            for (const auto offset : xrange(start, end)) {
                times.emplace_back(offset);
            }

            logger << TLOG_INFO << "signals size: " << signalsSize << "; tagSignals size: " << tagSignals.size() << "; times size: " << times.size();

            THolder<TSecondPlacementReader> reader(headerCache.Create(root, HostName.GetName(), Period, ChunkTime));
            if (!reader) {
                return;
            }

            TVector<TSomethingFormat::TReadData> records(reader->Read(times, tagSignals));
            for (const auto& record : records) {
                TInstant timestamp = Period.FromOffset(ChunkTime, record.Timestamp);
                auto& mergersBySignal(Mergers.at(requestKeys.at(record.RequestKeyIndex)));
                for (const auto& [signal, value] : record.Record.GetValues()) {
                    for (auto* merger : mergersBySignal.at(signal)) {
                        TUnistat::Instance().PushSignalUnsafe(
                            NMetrics::REQUESTER_READ_HISTDB_TOTAL_POINTS,
                            merger->GetRequest().GetPoints());
                        merger->Mul(timestamp, value.GetValue());
                    }
                }
            }
        }

        THostName HostName;
        TRecordPeriod Period;
        TInstant ChunkTime;
        THashMap<TInternedRequestKey, THashMap<TSignalName, TVector<TSeriesMerger*>>> Mergers;
    };

    class THeaderRegistry {
    public:
        THeaderAccumulator& Get(THostName hostName, const TRecordPeriod& period, TInstant chunkTime) {
            THeaderKey key{hostName, period.GetResolution(), chunkTime};
            THeaderMap::insert_ctx ctx;
            auto it = Headers.find(key, ctx);
            if (it == Headers.end()) {
                it = Headers.emplace_direct(ctx, key, THeaderAccumulator{
                    .HostName = hostName,
                    .Period = period,
                    .ChunkTime = chunkTime
                });
            }
            return it->second;
        }

        TVector<THeaderAccumulator> GetResult() {
            TVector<THeaderAccumulator> result(Reserve(Headers.size()));
            for (auto& [key, accumulator] : Headers) {
                result.emplace_back(std::move(accumulator));
            }
            return result;
        }

    private:
        // host name, resolution, chunk start time
        using THeaderKey = std::tuple<THostName, TDuration, TInstant>;
        using THeaderMap = THashMap<THeaderKey, THeaderAccumulator>;

        THeaderMap Headers;
    };

    TVector<TInstant> FindChunks(const TRecordPeriod& period, TInstant start, TInstant end) {
        TVector<TInstant> chunks;
        start = period.GetStartTime(start);
        end = period.GetStartTime(end);
        TDuration interval(period.GetInterval());
        for (const auto position : xrange((end + interval - start).GetValue() / interval.GetValue())) {
            chunks.emplace_back(start + interval * position);
        }
        return chunks;
    }
}

TReplicaIndex THistDbRequester::GetReplicaIndex() const {
    return ReplicaIndex;
}

TReplicaIndex TStockpileRequester::GetReplicaIndex() const {
    return ReplicaIndex;
}

TVector<THistoryResponse> THistDbRequester::ReadHistoryData(const TVector<THistoryRequest>& historyRequests, TInstant deadline,
                                                            const TString& requestId) {
    NMonitoring::TMeasuredMethod scopedTimer(Logger, "", NMetrics::REQUESTER_READ_HISTDB_TIME);
    NMonitoring::TRequestLog logger(Logger, requestId);
    TAccumulatorMapping accumulatorMapping(YasmConf, EAggregationMethod::MetaGroup);
    TVector<TSeriesMerger> seriesMergers(Reserve(historyRequests.size()));
    TVector<THeaderAccumulator> requestsByHeaders;

    TVector<THistoryRequest> requests = NZoom::NProtobuf::THistoryResponse::FilterRequests(ReplicaIndex, historyRequests);

    // find which requests should go into which chunks
    {
        THeaderRegistry headers;
        for (const auto& historyRequest : requests) {
            auto& merger(seriesMergers.emplace_back(TSeriesMerger::Create(historyRequest, accumulatorMapping)));
            if (!merger.IsValid()) {
                continue;
            }

            for (const auto chunkTime : FindChunks(merger.GetPeriod(), historyRequest.Start, historyRequest.End)) {
                auto& accumulator(headers.Get(historyRequest.HostName, merger.GetPeriod(), chunkTime));
                accumulator.Mergers[historyRequest.RequestKey][historyRequest.SignalName].emplace_back(&merger);
            }
        }
        requestsByHeaders = headers.GetResult();
    }

    // spawn queries
    TVector<THistDbExecutor::TFuture> futures(Reserve(requestsByHeaders.size()));
    for (auto& headerRequests : requestsByHeaders) {
        futures.emplace_back(THistDbExecutor::Get().Add([&logger, &headerRequests, deadline, this]() {
            try {
                headerRequests.MakeRequest(HeaderCache, Root, deadline, logger);
            } catch (const NHistDb::TSignalLimitExceeded&) {
                headerRequests.SetStatusCode(EStatusCode::THistoryAggregatedSeries_EStatusCode_LIMIT_EXCEEDED);
                logger << TLOG_ERR << "Limits exceeded at chunk " << headerRequests.ChunkTime << ": " << CurrentExceptionMessage();
            } catch (const NHistDb::TTimeLimitExceeded&) {
                headerRequests.SetStatusCode(EStatusCode::THistoryAggregatedSeries_EStatusCode_TIME_EXCEEDED);
                logger << TLOG_ERR << "Time left at chunk " << headerRequests.ChunkTime << ": " << CurrentExceptionMessage();
            } catch (...) {
                headerRequests.SetStatusCode(EStatusCode::THistoryAggregatedSeries_EStatusCode_INTERNAL_ERROR);
                logger << TLOG_ERR << "Unexpected failure at chunk " << headerRequests.ChunkTime << ": " << CurrentExceptionMessage();
            }
        }));
    }
    NThreading::WaitExceptionOrAll(futures).GetValueSync();

    TVector<THistoryResponse> result(Reserve(seriesMergers.size()));
    for (auto& merger : seriesMergers) {
        result.push_back(merger.CreateResponse());
    }
    return result;
}

TVector<THistoryResponse> TStockpileRequester::ReadHistoryData(const TVector<THistoryRequest>& requests, TInstant deadline,
                                                               const TString& requestId) {
    NMonitoring::TMeasuredMethod scopedTimer(Logger, "", NMetrics::REQUESTER_READ_STOCKPILE_TIME);
    NMonitoring::TRequestLog logger(Logger, requestId);
    TStockpileReader reader(StockpileState, MetabaseTimeout, StockpileTimeout, NZoom::NProtobuf::THistoryResponse::FilterRequests(ReplicaIndex, requests), logger);
    reader.Execute(deadline);

    TVector<THistoryResponse> result(Reserve(reader.ResponseCount()));
    for (const auto idx : xrange(reader.ResponseCount())) {
        auto response = reader.GetResponse(idx);
        switch (response.StatusCode) {
            case EStatusCode::THistoryAggregatedSeries_EStatusCode_OK:
                TUnistat::Instance().PushSignalUnsafe(NMetrics::REQUESTER_STOCKPILE_STATUS_CODE_OK, 1);
                break;
            case EStatusCode::THistoryAggregatedSeries_EStatusCode_LIMIT_EXCEEDED:
                TUnistat::Instance().PushSignalUnsafe(NMetrics::REQUESTER_STOCKPILE_STATUS_CODE_LIMIT_EXCEEDED, 1);
                break;
            case EStatusCode::THistoryAggregatedSeries_EStatusCode_TIME_EXCEEDED:
                TUnistat::Instance().PushSignalUnsafe(NMetrics::REQUESTER_STOCKPILE_STATUS_CODE_TIME_EXCEEDED, 1);
                break;
            case EStatusCode::THistoryAggregatedSeries_EStatusCode_INTERNAL_ERROR:
                TUnistat::Instance().PushSignalUnsafe(NMetrics::REQUESTER_STOCKPILE_STATUS_CODE_INTERNAL_ERROR, 1);
                break;
            case EStatusCode::THistoryAggregatedSeries_EStatusCode_NOT_FOUND:
                TUnistat::Instance().PushSignalUnsafe(NMetrics::REQUESTER_STOCKPILE_STATUS_CODE_NOT_FOUND, 1);
                break;
            case EStatusCode::THistoryAggregatedSeries_EStatusCode_PARTIAL:
                TUnistat::Instance().PushSignalUnsafe(NMetrics::REQUESTER_STOCKPILE_STATUS_CODE_PARTIAL, 1);
                break;
            default:
                break;
        }
        result.push_back(std::move(response));
    }
    return result;
}
