#include "handlers.h"

#include <infra/yasm/common/points/hgram/ugram/compress/compress.h>
#include <infra/yasm/zoom/components/serialization/common/msgpack_utils.h>
#include <infra/yasm/histdb/components/placements/second.h>

#include <infra/monitoring/common/perf.h>
#include <infra/monitoring/common/msgpack.h>

using namespace NHistDb;
using namespace NTags;
using namespace NMonitoring;
using namespace NZoom::NHost;
using namespace NZoom::NSignal;
using namespace NZoom::NRecord;
using namespace NZoom::NPython;

namespace {
    TVector<TSignalName> ParseSignals(msgpack::object root) {
        TVector<TSignalName> result;
        EnsureIs(root, msgpack::type::ARRAY);
        result.reserve(root.via.array.size);
        for (const auto& element : TArrayIterator(root.via.array)) {
            result.emplace_back(element.as<TStringBuf>());
        }
        return result;
    }

    TSomethingFormat::TTagSignals ParseTagSignals(msgpack::object root) {
        TSomethingFormat::TTagSignals result;
        EnsureIs(root, msgpack::type::MAP);
        result.reserve(root.via.map.size);
        for (const auto& pair : TMapIterator(root.via.map)) {
            result.emplace_back(
                TRequestKey::FromString(pair.key.as<TStringBuf>()),
                ParseSignals(pair.val)
            );
        }
        return result;
    }

    class TRecordSerializer final: public ISignalValueCallback {
    private:
        msgpack::packer<msgpack::sbuffer>& Packer;
        TValueRefSerializer<msgpack::sbuffer, NZoom::NHgram::TUgramCompressor> ValueSerializer;

    public:
        TRecordSerializer(msgpack::packer<msgpack::sbuffer>& packer)
            : Packer(packer)
            , ValueSerializer(Packer)
        {
        }

        void SetObjectsCount(const size_t count) final {
            Packer.pack_map(count);
        }

        void OnSignalValue(const NZoom::NSignal::TSignalName& name, const NZoom::NValue::TValueRef& value) final {
            PackString(Packer, name.GetName());
            value.Update(ValueSerializer);
        }
    };

    TString EncodeResponse(const TVector<TSomethingFormat::TReadAggregatedData>& response) {
        msgpack::sbuffer buf;
        msgpack::packer<msgpack::sbuffer> packer(buf);
        TRecordSerializer recordSerializer(packer);

        for (const auto& row : response) {
            packer.pack_array(4);
            packer.pack_uint64(row.Timestamp);
            packer.pack_uint64(row.MatchedInstances);
            PackString(packer, row.RequestKey.ToNamed());
            row.Record.Process(recordSerializer);
        }

        return TString(buf.data(), buf.size());
    }

    TString EncodeResponse(const TVector<TCompatResponse>& response) {
        msgpack::sbuffer buf;
        msgpack::packer<msgpack::sbuffer> packer(buf);
        TRecordSerializer recordSerializer(packer);

        for (const auto& row : response) {
            packer.pack_array(3);
            packer.pack_uint64(row.Timestamp.Seconds());
            PackString(packer, row.RequestKey.ToNamed());
            row.Record.Process(recordSerializer);
        }

        return TString(buf.data(), buf.size());
    }
}

void TPingHandler::DoReply(const NMonitoring::TServiceRequest::TRef request, const TParsedHttpFull&) {
    TMeasuredMethod perf(Logger, "", NMetrics::HANDLERS_PING_TIME);
    request->Finish(THttpResponse(HttpCodes::HTTP_OK).SetContent("OK"));
}

TCompatRequest TReadStockpileHandler::DeserializeRequest(const TString& body) {
    TCompatRequest request;

    auto message = msgpack::unpack(body.data(), body.size());
    EnsureIs(message.get(), msgpack::type::MAP);
    for (const auto& pair : TMapIterator(message.get().via.map)) {
        TStringBuf key(pair.key.as<TStringBuf>());
        if (key == TStringBuf("start")) {
            request.Start = TInstant::Seconds(pair.val.as<ui64>());
        } else if (key == TStringBuf("end")) {
            request.End = TInstant::Seconds(pair.val.as<ui64>());
        } else if (key == TStringBuf("period")) {
            request.Period = TDuration::Seconds(pair.val.as<ui64>());
        } else if (key == TStringBuf("tag_signals")) {
            request.TagSignals = ParseTagSignals(pair.val);
        } else if (key == TStringBuf("host")) {
            request.HostName = THostName(pair.val.as<TStringBuf>());
        }
    }

    if (request.HostName.Empty() || !request.Start || !request.End || request.TagSignals.empty()) {
        ythrow yexception() << "invalid request";
    }
    if (request.Period == TDuration::Zero()) {
        request.Period = TDuration::Seconds(5);
    }

    return request;
}

void TReadStockpileHandler::DoReply(const NMonitoring::TServiceRequest::TRef request, const TParsedHttpFull&) {
    TMeasuredMethod perf(Logger, "", NMetrics::HANDLERS_STOCKPILE_READ_TIME);

    const auto parsedRequest = DeserializeRequest(request->GetInput().ReadAll());
    TStockpileReader reader(StockpileState, MetabaseTimeout, StockpileTimeout, NZoom::NProtobuf::THistoryResponse::FilterRequests(ReplicaIndex, GetRequestsFromCompat(parsedRequest)), Logger);
    try {
        const auto readerResponse = reader.ReadCompat();
        request->Finish(THttpResponse(HTTP_OK).SetContent(EncodeResponse(readerResponse)));
    } catch (const yexception& e) {
        request->Finish(THttpResponse(HttpCodes::HTTP_INTERNAL_SERVER_ERROR).SetContent(e.what()));
    }

}

TReadAggregatedHandler::TRequest TReadAggregatedHandler::DeserializeRequest(const TString& body) {
    TReadAggregatedHandler::TRequest request;

    auto message = msgpack::unpack(body.data(), body.size());
    EnsureIs(message.get(), msgpack::type::MAP);
    for (const auto& pair : TMapIterator(message.get().via.map)) {
        TStringBuf key(pair.key.as<TStringBuf>());
        if (key == TStringBuf("header_path")) {
            request.HeaderPath = pair.val.as<TString>();
        } else if (key == TStringBuf("data_path")) {
            request.DataPath = pair.val.as<TString>();
        } else if (key == TStringBuf("offset")) {
            request.Offset = pair.val.as<size_t>();
        } else if (key == TStringBuf("limit")) {
            request.Limit = pair.val.as<size_t>();
        } else if (key == TStringBuf("tag_signals")) {
            request.TagSignals = ParseTagSignals(pair.val);
        }
    }

    return request;
}

THolder<TSecondPlacementReader> TReadAggregatedHandler::GetReader(const TString& headerPath, const TString& dataPath) {
    TMeasuredMethod perf(Logger, "", NMetrics::HANDLERS_HEADER_LOADING_TIME);
    return HeaderCache.Create(headerPath, dataPath);
}

void TReadAggregatedHandler::DoReply(const NMonitoring::TServiceRequest::TRef request, const TParsedHttpFull&) {
    TMeasuredMethod perf(Logger, "", NMetrics::HANDLERS_AGGREGATED_READ_TIME);

    const auto parsedRequest = DeserializeRequest(request->GetInput().ReadAll());
    const auto reader(GetReader(parsedRequest.HeaderPath, parsedRequest.DataPath));
    if (!reader) {
        request->Finish(THttpResponse(HttpCodes::HTTP_OK));
        return;
    }

    const auto response(reader->ReadAggregated(parsedRequest.Offset, parsedRequest.Limit, parsedRequest.TagSignals, YasmConf));

    request->Finish(THttpResponse(HttpCodes::HTTP_OK).SetContent(EncodeResponse(response)));
}
