#pragma once

#include <infra/yasm/histdb/server/lib/data_requesters.h>
#include <infra/yasm/histdb/server/lib/metrics.h>
#include <infra/yasm/histdb/server/lib/settings.h>
#include <infra/yasm/histdb/server/lib/proto_handlers.h>

#include <infra/yasm/histdb/components/stockpile/stockpile_reader.h>
#include <infra/yasm/histdb/components/placements/second.h>

#include <infra/monitoring/common/application.h>

namespace NHistDb {
    static constexpr size_t CACHE_SIZE = 25;

    class TPingHandler: public NMonitoring::IServiceReplier {
    public:
        TPingHandler(TLog& logger)
            : Logger(logger)
        {
        }

        void DoReply(const NMonitoring::TServiceRequest::TRef request, const TParsedHttpFull& meta) override;

    private:
        TLog& Logger;
    };

    class TReadStockpileHandler: public NMonitoring::IServiceReplier {
    public:
        TReadStockpileHandler(NStockpile::TStockpileState& stockpileState, const TServerSettings& settings, TLog& logger, const TReplicaIndex replicaIndex = TReplicaIndex())
            : StockpileState(stockpileState)
            , MetabaseTimeout(settings.GetReaderMetabaseTimeout())
            , StockpileTimeout(settings.GetReaderStockpileTimeout())
            , Logger(logger)
            , ReplicaIndex(replicaIndex)
        {
        }

    private:
        TCompatRequest DeserializeRequest(const TString& body);

        void DoReply(const NMonitoring::TServiceRequest::TRef request, const TParsedHttpFull& meta) override;

        NStockpile::TStockpileState& StockpileState;
        const TDuration MetabaseTimeout;
        const TDuration StockpileTimeout;
        TLog& Logger;
        TReplicaIndex ReplicaIndex;
    };

    class TReadAggregatedHandler: public NMonitoring::IServiceReplier {
    public:
        TReadAggregatedHandler(TSecondHeaderCache& headerCache, const NZoom::NYasmConf::TYasmConf& yasmConf, TLog& logger)
            : HeaderCache(headerCache)
            , YasmConf(yasmConf)
            , Logger(logger)
        {
        }

    private:
        struct TRequest {
            TString HeaderPath;
            TString DataPath;
            size_t Offset;
            size_t Limit;
            TSomethingFormat::TTagSignals TagSignals;
        };

        TRequest DeserializeRequest(const TString& body);

        THolder<TSecondPlacementReader> GetReader(const TString& headerPath, const TString& dataPath);

        void DoReply(const NMonitoring::TServiceRequest::TRef request, const TParsedHttpFull& meta) override;

        TSecondHeaderCache& HeaderCache;
        const NZoom::NYasmConf::TYasmConf& YasmConf;
        TLog& Logger;
    };

    class THistDbHandlersCollection {
    public:
        THistDbHandlersCollection(
            TLog& logger,
            const TServerSettings& settings,
            const NTags::NPrivate::TInstanceKeyTags tags = NTags::TInstanceKey::FromRtcEnviron().GetTags()
        ) : YasmConf(NZoom::NYasmConf::TYasmConf::FromFileName(settings.GetYasmConfPath()))
            , ReplicaIndex(ConstructReplicaIndex(tags))
            , HeaderCache(CACHE_SIZE)
            , StockpileSettings {settings.CreateStockpileSettings()}
            , FastConfig(logger, NYasm::NCommon::MakeYasmFastConfigSettings("fast_config.json"))
            , StockpileState(logger, StockpileSettings, FastConfig)
            , StockpileRequester(StockpileState, settings.GetReaderMetabaseTimeout(),
                settings.GetReaderStockpileTimeout(), logger, ReplicaIndex)
            , IsStockpileEnabled(settings.IsStockpileEnabled())
            , Logger(logger)
        {
            assert(ReplicaIndex == StockpileRequester.GetReplicaIndex());

            // Handler to check that histdb-cacher is running. Called by histfront.
            Handlers.emplace_back("/ping", MakeHolder<TPingHandler>(logger));

            // Unused handler. Referenced by placement logic in histdb code in bb though.
            Handlers.emplace_back("/read-stockpile", MakeHolder<TReadStockpileHandler>(StockpileState, settings, logger, ReplicaIndex));

            // Unused handler. Referenced by placement logic in histdb code in bb though.
            Handlers.emplace_back("/read-aggregated", MakeHolder<TReadAggregatedHandler>(HeaderCache, YasmConf, logger));

            // Main handler to read history data.
            Handlers.emplace_back("/read_stockpile_history", MakeHolder<TReadHistoryHandler>(StockpileRequester,
                NMetrics::HANDLERS_READ_STOCKPILE_HISTORY_TIME, logger));
        }

        ~THistDbHandlersCollection() {
            Stop();
        }

        void Start() {
            FastConfig.Start();
            if (IsStockpileEnabled) {
                Logger << TLOG_INFO << "Starting stockpile client...";
                StockpileState.UpdateShards();
                StockpileState.Start();
            }
        }

        void Stop() {
            if (IsStockpileEnabled) {
                StockpileState.Stop();
            }
            FastConfig.Stop();
        }

        void Register(NMonitoring::TWebServer& server) {
            for (const auto& pair : Handlers) {
                server.Add(pair.first, *pair.second);
            }
        }

    private:
        NZoom::NYasmConf::TYasmConf YasmConf;

        const TReplicaIndex ReplicaIndex;

        TSecondHeaderCache HeaderCache;

        NStockpile::TSettings StockpileSettings;
        NYasm::NCommon::TFastConfig FastConfig;
        NStockpile::TStockpileState StockpileState;

        TStockpileRequester StockpileRequester;
        const bool IsStockpileEnabled;
        TLog& Logger;

        TVector<std::pair<TString, THolder<NMonitoring::IServiceReplier>>> Handlers;

        static TReplicaIndex ConstructReplicaIndex(const NTags::NPrivate::TInstanceKeyTags tags) {
            TReplicaIndex result;
            for (auto tag = tags.begin(); tag != tags.end(); ++tag) {
                if (tag->Name == "replica") {
                    try {
                        result.ConstructInPlace(FromString(tag->Value));
                    } catch (const TFromStringException& e) {
                    }
                    return result;
                }
            };
            return result; // may be did not marked
        }
    };
}
