#include "proto_handlers.h"

#include <infra/monitoring/common/proto_replier_ut.h>
#include <infra/yasm/common/tests.h>
#include <infra/yasm/common/config/fast_config.h>
#include <infra/yasm/histdb/components/dumper/common_ut.h>
#include <infra/yasm/histdb/components/placements/second.h>
#include <infra/yasm/histdb/dumper/lib/five_seconds_writer.h>
#include <infra/yasm/zoom/components/serialization/history/history.h>

#include <library/cpp/testing/unittest/registar.h>

#include <util/folder/tempdir.h>

using namespace NHistDb;
using namespace NZoom::NProtobuf;
using namespace NZoom::NYasmConf;
using namespace NZoom::NValue;
using namespace NYasm::NCommon;
using namespace NTags;
using NYasm::NInterfaces::NInternal::THistoryReadAggregatedRequest;
using NYasm::NInterfaces::NInternal::THistoryReadAggregatedResponse;
using EStatusCode = NYasm::NInterfaces::NInternal::THistoryAggregatedSeries::EStatusCode;

namespace {
    class THelper {
    public:
        THelper(const TReplicaIndex replicaIndex)
            : Logger(NYasm::NCommon::NTest::CreateLog())
            , HeaderCache(1)
            , YasmConf(TYasmConf::FromString(TStringBuf("{\"conflist\": {\"common\": {\"signals\": {}, \"patterns\": {}, \"periods\": {}}}}")))
            , HistDbRequester(Root.Name(), HeaderCache, YasmConf, Logger, replicaIndex)
            , ReadHandler(HistDbRequester, "test_read_handler", Logger)
            , Writer(Logger, Root.Name(), TInstant::Zero())
            , Period(TRecordPeriod::Get("s5"))
            , FiveMinutesPeriod(TRecordPeriod::Get("m5"))
        {
        }

        THelper() : THelper(TReplicaIndex()) {}

        TVector<TValue> FillValues(TInstant startTime, TInstant endTime) {
            TVector<TValue> values;
            for (const auto idx : xrange(Period.GetOffset(endTime) + 1 - Period.GetOffset(startTime))) {
                values.emplace_back(TValue(1));
                Y_UNUSED(idx);
            }
            return values;
        }

        void WriteChunk(TMockedRecordDescriptor& record, TInstant startTime, TVector<TValue> values) {
            UNIT_ASSERT(!values.empty());
            record
                .SetValues(values)
                .SetStartTime(startTime)
                .SetEndTime(startTime + Period.GetResolution() * (values.size() - 1))
                .SetFlushOffset(FiveMinutesPeriod.GetPointStartTime(startTime));
            Writer.OnRecord(record);
        }

        void WriteChunk(TVector<TMockedRecordDescriptor*> records, TInstant startTime, TInstant endTime) {
            // N.B. last five minutes will be lost
            TVector<std::pair<TInstant, TInstant>> timePairs;
            TInstant normalizedStartTime(FiveMinutesPeriod.GetPointStartTime(startTime));
            TInstant normalizedEndTime(FiveMinutesPeriod.GetPointStartTime(endTime) + FiveMinutesPeriod.GetResolution());

            size_t chunks = (normalizedEndTime - normalizedStartTime).GetValue() / FiveMinutesPeriod.GetResolution().GetValue();
            if (chunks) {
                for (const auto chunk : xrange(chunks)) {
                    TInstant left = Max(normalizedStartTime + FiveMinutesPeriod.GetResolution() * chunk, startTime);
                    TInstant right = Min(normalizedStartTime + FiveMinutesPeriod.GetResolution() * (chunk + 1), endTime);
                    timePairs.emplace_back(left, right);
                }
            } else{
                timePairs.emplace_back(startTime, endTime);
            }

            for (const auto& [left, right] : timePairs) {
                TDuration duration = right - left - Period.GetResolution();
                for (auto* record : records) {
                    WriteChunk(*record, left, FillValues(left, left + duration));
                }
            }

            Writer.Finish();
        }

        void RawRead(const THistoryReadAggregatedRequest& request, THistoryReadAggregatedResponse& response) {
            auto httpResponse = NMonitoring::THttpRequestBuilder()
                .FromProtoRequest(request)
                .Execute(ReadHandler);
            UNIT_ASSERT_VALUES_EQUAL(httpResponse.Code, 200);
            httpResponse.ToProto(response);
        }

        TVector<THistoryResponse> ReadMany(const TVector<THistoryRequest>& historyRequests) {
            THistoryReadAggregatedRequest request;
            request.SetRequestId("test");
            THistoryReadAggregatedResponse response;

            {
                THistoryRequestWriter requestWriter(request);
                requestWriter.Reserve(historyRequests.size());
                for (const auto& historyRequest : historyRequests) {
                    requestWriter.Add(historyRequest);
                }
            }

            RawRead(request, response);
            return THistoryResponse::FromProto(response);
        }

        THistoryResponse ReadOne(const THistoryRequest& historyRequest) {
            auto responses(ReadMany(TVector<THistoryRequest>{historyRequest}));
            UNIT_ASSERT_VALUES_EQUAL(responses.size(), 1);
            return std::move(responses.back());
        }

        const TRecordPeriod& GetPeriod() const {
            return Period;
        }

    private:
        TLog Logger;
        TTempDir Root;
        TSecondHeaderCache HeaderCache;
        NZoom::NYasmConf::TYasmConf YasmConf;
        THistDbRequester HistDbRequester;
        TReadHistoryHandler ReadHandler;
        TGroupFiveSecondsWriter Writer;
        TRecordPeriod Period;
        TRecordPeriod FiveMinutesPeriod;
    };
}

Y_UNIT_TEST_SUITE(TestReadHistoryHandler) {
    Y_UNIT_TEST(TestSimpleReading) {
        THelper helper;

        TMockedRecordDescriptor record;
        helper.WriteChunk(TVector<TMockedRecordDescriptor*>{&record}, TInstant::Seconds(15005), TInstant::Seconds(18005));

        THistoryRequest request{
            .Start = TInstant::Seconds(15005),
            .End = TInstant::Seconds(17995),
            .Period = TDuration::Seconds(5),
            .HostName = TStringBuf("SAS.000"),
            .RequestKey = TStringBuf("itype=testing"),
            .SignalName = TStringBuf("disk_usage_summ")
        };
        auto response(helper.ReadOne(request));
        UNIT_ASSERT_VALUES_EQUAL(response.StatusCode, EStatusCode::THistoryAggregatedSeries_EStatusCode_OK);

        TVector<TValue> values;
        for (const auto idx : xrange(599)) {
            values.emplace_back(TValue(1));
            Y_UNUSED(idx);
        }
        UNIT_ASSERT_VALUES_EQUAL(response.Series.GetValues(), values);
    }

    Y_UNIT_TEST(TestAggregatedReading) {
        THelper helper;

        TMockedRecordDescriptor firstRecord;
        firstRecord.SetInstanceKey(NTags::TInstanceKey::FromNamed("testing|prj=first"));
        TMockedRecordDescriptor secondRecord;
        secondRecord.SetInstanceKey(NTags::TInstanceKey::FromNamed("testing|prj=second"));
        helper.WriteChunk(TVector<TMockedRecordDescriptor*>{&firstRecord, &secondRecord}, TInstant::Seconds(15005), TInstant::Seconds(18005));

        THistoryRequest request{
            .Start = TInstant::Seconds(15005),
            .End = TInstant::Seconds(17995),
            .Period = TDuration::Seconds(5),
            .HostName = TStringBuf("SAS.000"),
            .RequestKey = TStringBuf("itype=testing"),
            .SignalName = TStringBuf("disk_usage_summ")
        };
        auto response(helper.ReadOne(request));
        UNIT_ASSERT_VALUES_EQUAL(response.StatusCode, EStatusCode::THistoryAggregatedSeries_EStatusCode_OK);

        TVector<TValue> values;
        for (const auto idx : xrange(599)) {
            values.emplace_back(TValue(2));
            Y_UNUSED(idx);
        }
        UNIT_ASSERT_VALUES_EQUAL(response.Series.GetValues(), values);
    }

    Y_UNIT_TEST(TestNotFound) {
        THelper helper;
        THistoryRequest request{
            .Start = TInstant::Seconds(15005),
            .End = TInstant::Seconds(17995),
            .Period = TDuration::Seconds(5),
            .HostName = TStringBuf("SAS.000"),
            .RequestKey = TStringBuf("itype=testing"),
            .SignalName = TStringBuf("disk_usage_summ")
        };
        auto response(helper.ReadOne(request));
        UNIT_ASSERT_VALUES_EQUAL(response.StatusCode, EStatusCode::THistoryAggregatedSeries_EStatusCode_NOT_FOUND);
    }
}

Y_UNIT_TEST_SUITE(TestReplicaIndexMatching) {
    TString StringDescription(const TReplicaIndex replicaIndex, const TString& name = "") {
        TString indexString = "";
        if (replicaIndex.Defined()) {
            indexString = ToString(replicaIndex.GetRef());
        }
        return name + (name == "" ? "" : ": ") + "TReplicaIndex (" + indexString + ")";
    }

    Y_UNIT_TEST(TestDifferentReplicaIndices) {
        const TVector<const TReplicaIndex> replicaIndices = {
            TReplicaIndex(),
            TReplicaIndex(0),
            TReplicaIndex(1)
        };

        for (auto envReplicaIndex : replicaIndices) {
            THelper helper(envReplicaIndex);

            TMockedRecordDescriptor record;
            helper.WriteChunk(TVector<TMockedRecordDescriptor *>{&record}, TInstant::Seconds(15005),
                              TInstant::Seconds(18005));

            for (auto requestReplicaIndex : replicaIndices) {
                if (MatchReplicas(envReplicaIndex, requestReplicaIndex)) {
                    Cout << "Checking matching replicas. " << StringDescription(envReplicaIndex, "env replica") << "; " << StringDescription(requestReplicaIndex, "request replica") << Endl;
                } else {
                    Cout << "Checking mismatching replicas. " << StringDescription(envReplicaIndex, "env replica") << "; " << StringDescription(requestReplicaIndex, "request replica") << Endl;
                }

                THistoryRequest request{
                    .Start = TInstant::Seconds(15005),
                    .End = TInstant::Seconds(17995),
                    .Period = TDuration::Seconds(5),
                    .HostName = TStringBuf("SAS.000"),
                    .RequestKey = TStringBuf("itype=testing"),
                    .SignalName = TStringBuf("disk_usage_summ"),
                    .ReplicaIndex = requestReplicaIndex
                };

                auto responses(helper.ReadMany(TVector<THistoryRequest>{request}));

                if (MatchReplicas(envReplicaIndex, requestReplicaIndex)) {
                    UNIT_ASSERT_VALUES_EQUAL(responses.size(), 1);
                    UNIT_ASSERT_VALUES_EQUAL(responses[0].StatusCode, EStatusCode::THistoryAggregatedSeries_EStatusCode_OK);
                } else {
                    UNIT_ASSERT_VALUES_EQUAL(responses.size(), 0);
                }
                Cout << "OK" << Endl;
            }
        }
    }
}
