#pragma once

#include <infra/yasm/stockpile_client/common/base_types.h>
#include <infra/monitoring/common/settings.h>
#include <util/datetime/base.h>

#include <util/folder/path.h>

namespace NHistDb {
    static const TString YASM_CONF(TStringBuf("yasm.conf"));
    static const TString HISTDB_ROOT(TStringBuf("/place/db/bsconfig/webstate/yasm_histdb/histdb"));

    class TServerSettings final: public NMonitoring::TSettings {
    public:
        inline const TString& GetYasmConfPath() const {
            return YasmConfPath;
        }

        inline void SetYasmConfPath(const TString& value) {
            YasmConfPath = value;
        }

        inline const TString& GetHistdbRootPath() const {
            return HistdbRootPath;
        }

        inline void SetHistdbRootPath(const TString& value) {
            HistdbRootPath = value;
        }

        inline void EnableStockpile() {
            StockpileEnabled = true;
        }

        inline bool IsStockpileEnabled() const {
            return StockpileEnabled;
        }

        inline TDuration GetReaderMetabaseTimeout() const {
            return ReaderMetabaseTimeout;
        }

        inline void SetReaderMetabaseTimeout(size_t duration) {
            ReaderMetabaseTimeout = TDuration::Seconds(duration);
        }

        inline TDuration GetReaderStockpileTimeout() const {
            return ReaderStockpileTimeout;
        }

        inline void SetReaderStockpileTimeout(size_t duration) {
            ReaderStockpileTimeout = TDuration::Seconds(duration);
        }

        NStockpile::TSettings CreateStockpileSettings() const {
            return StockpileSettingsFactory.CreateStockpileSettings(false);
        }

        void ProcessParsedOptions(const NLastGetopt::TOptsParseResult& parsed) {
            SetYasmConfPath(parsed.Get("yasm-conf"));
            SetHistdbRootPath(parsed.Get("histdb-root"));
            if (parsed.Has("enable-stockpile")) {
                EnableStockpile();
            }

            SetReaderStockpileTimeout(parsed.Get<size_t>("stockpile-timeout"));
            SetReaderMetabaseTimeout(parsed.Get<size_t>("metabase-timeout"));

            StockpileSettingsFactory.ProcessParsedOptions(parsed);
        }

        static void FillParser(NLastGetopt::TOpts& opts) {
            opts.AddLongOption("yasm-conf", "path to yasm.conf").Required().RequiredArgument().DefaultValue(YASM_CONF);
            opts.AddLongOption("histdb-root", "patch to histdb root").Optional().RequiredArgument().DefaultValue(HISTDB_ROOT);
            opts.AddLongOption("enable-stockpile", "enable stockpile").Optional().NoArgument();

            opts.AddLongOption("metabase-timeout", "time limit for metabase").Optional().RequiredArgument().DefaultValue("10");
            opts.AddLongOption("stockpile-timeout", "time limit for stockpile").Optional().RequiredArgument().DefaultValue("10");
            NStockpile::TStockpileSetingsFactory::FillParser(opts);
        }

    protected:
        void FillFromFileConfig(const NConfig::TConfig& config) override {
            TSettings::FillFromFileConfig(config);

            SetYasmConfPath(config["yasm_conf"].As<TString>(YASM_CONF));
            SetHistdbRootPath(config["histdb_root"].As<TString>(HISTDB_ROOT));

            SetReaderMetabaseTimeout(config["reader_metabase_timeout"].As<size_t>(10));
            SetReaderStockpileTimeout(config["reader_stockpile_timeout"].As<size_t>(10));

            StockpileSettingsFactory.FillFromFileConfig(config);
        }

    private:
        TString YasmConfPath;
        TString HistdbRootPath;

        TDuration ReaderMetabaseTimeout;
        TDuration ReaderStockpileTimeout;
        bool StockpileEnabled = false;

        NStockpile::TStockpileSetingsFactory StockpileSettingsFactory;
    };
}
