from libcpp cimport bool
from libcpp.utility cimport pair
from cpython cimport PyObject
from util.generic.ptr cimport THolder
from util.generic.string cimport TString, TStringBuf
from util.generic.vector cimport TVector
from util.system.types cimport i64


cdef extern from "infra/yasm/neh_bindings/wrapper/wrapper.h" namespace "NNehWrapper":
    cdef cppclass TResponse:
        TResponse() except +
        bool Empty() except +
        bool IsReady() except +
        bool IsError() except +
        bool IsCancelled() except +
        int GetErrorCode() except +
        int GetSystemErrorCode() except +
        TString GetErrorText() except +
        const TString& GetAddr() except +
        const TString& GetData() except +
        double GetDuration() except +
        TVector[pair[TStringBuf, TStringBuf]] GetHeaders() except +
        PyObject* GetPayload() except +

    cdef cppclass TRequesterIterator:
        TRequesterIterator() except +
        TResponse Next() nogil except +

    cdef cppclass THttpRequest:
        THttpRequest() except +
        THttpRequest& SetAddr(const TString&) except +
        THttpRequest& SetContent(const TString&, bool) except +
        THttpRequest& SetContentType(const TString&) except +
        THttpRequest& AddHeader(const TString&, const TString&) except +

    cdef cppclass TRequester:
        TRequester() nogil except +
        void SetOption(const TString&, const TString&) except +
        void Add(const TString&, const TString&, double, PyObject*) except +
        void Add(const THttpRequest&, double, PyObject*) except +
        void AddToGroup(const TString&, const TString&, i64, double, PyObject*) except +
        void AddToGroup(const THttpRequest&, i64, double, PyObject*) except +
        TRequesterIterator Iterate(double) except +
        TResponse Wait(double) nogil except +
        i64 ReserveGroupId() nogil
        void CancelGroup(i64 groupId) nogil except +

    void LockAllMemory(bool future) except +


cdef class _Response:
    cdef TResponse _c_response

    def __repr__(self):
        return "<Response: addr={!r} payload={!r}>".format(self.addr, self.payload)

    def __str__(self):
        return repr(self)

    @staticmethod
    cdef _create(TResponse& c_response):
        if c_response.Empty():
            return None
        else:
            response = _Response()
            response._c_response = c_response
            return response

    property ready:
        def __get__(self):
            return self._c_response.IsReady()

    property failed:
        def __get__(self):
            return self._c_response.IsError()

    property cancelled:
        def __get__(self):
            return self._c_response.IsCancelled()

    property error_code:
        def __get__(self):
            return self._c_response.GetSystemErrorCode()

    property status_code:
        def __get__(self):
            return self._c_response.GetErrorCode()

    property error_text:
        def __get__(self):
            return self._c_response.GetErrorText()

    property data:
        def __get__(self):
            return self._c_response.GetData()

    property addr:
        def __get__(self):
            return self._c_response.GetAddr()

    property duration:
        def __get__(self):
            return self._c_response.GetDuration()

    property headers:
        def __get__(self):
            return self._c_response.GetHeaders()

    property payload:
        def __get__(self):
            return <object>self._c_response.GetPayload()


cdef class _RequesterIterator:
    cdef TRequesterIterator _c_iterator

    @staticmethod
    cdef _create(TRequesterIterator& c_iterator):
        iterator = _RequesterIterator()
        iterator._c_iterator = c_iterator
        return iterator

    def __iter__(self):
        return self

    def __next__(self):
        cdef TResponse c_response
        with nogil:
            c_response = self._c_iterator.Next()
        response = _Response._create(c_response)
        if response is None:
            raise StopIteration()
        else:
            return response


cdef class HttpRequest:
    cdef THttpRequest _c_request

    def __cinit__(self, addr):
        self._c_request.SetAddr(addr)

    def set_content(self, content, compress_body=False):
        self._c_request.SetContent(content, compress_body)
        return self

    def set_content_type(self, content_type):
        self._c_request.SetContentType(content_type)
        return self

    def add_header(self, name, value):
        self._c_request.AddHeader(name, value)
        return self


cdef class Requester:
    cdef THolder[TRequester] _c_requester

    def __cinit__(self):
        with nogil:
            self._c_requester.Reset(new TRequester())

    def add(self, addr, data=None, timeout=None, payload=None):
        cdef TString c_addr = addr
        cdef TString c_data = data or ""
        self._c_requester.Get().Add(c_addr, c_data, timeout or 0.0, <PyObject*>payload)

    def add_to_group(self, addr, group_id, data=None, timeout=None, payload=None):
        cdef TString c_addr = addr
        cdef TString c_data = data or ""
        self._c_requester.Get().AddToGroup(c_addr, c_data, group_id, timeout or 0.0, <PyObject*>payload)

    def add_request(self, HttpRequest request, timeout=None, payload=None):
        self._c_requester.Get().Add(request._c_request, timeout or 0.0, <PyObject*>payload)

    def add_request_to_group(self, HttpRequest request, group_id, timeout=None, payload=None):
        self._c_requester.Get().AddToGroup(request._c_request, group_id, timeout or 0.0, <PyObject*>payload)

    def reserve_group_id(self):
        return self._c_requester.Get().ReserveGroupId()

    def set_connect_timeout(self, timeout=None):
        if timeout:
            self._c_requester.Get().SetOption("http2/ConnectTimeout", timeout)

    def set_slow_connect(self, delay=None):
        if delay:
            self._c_requester.Get().SetOption("http2/SymptomSlowConnect", delay)

    def iterate(self, timeout=None):
        return _RequesterIterator._create(self._c_requester.Get().Iterate(timeout or 0.0))

    def wait(self, timeout=None):
        cdef TResponse c_response
        cdef double c_timeout
        c_timeout = timeout or 0.0
        with nogil:
            c_response = self._c_requester.Get().Wait(c_timeout)
        return _Response._create(c_response)

    def cancel_group(self, group_id):
        cdef i64 c_group_id = group_id
        with nogil:
            self._c_requester.Get().CancelGroup(c_group_id)


def lock_all_memory(future=True):
    LockAllMemory(future)
