#include "wrapper.h"

#include <library/cpp/neh/http_common.h>
#include <library/cpp/blockcodecs/stream.h>
#include <library/cpp/blockcodecs/codecs.h>

#include <util/stream/str.h>
#include <util/system/mlock.h>

namespace NNehWrapper {
    inline TInstant ToDeadline(double timeout) {
        return timeout ? TDuration::Seconds(timeout).ToDeadLine() : TInstant::Max();
    }

    TResponse::TResponse(THolder<TRequestInFlight> inFlight)
        : InFlight(std::move(inFlight))
    {
    }

    void TResponse::Assign(TResponse& other) {
        InFlight.Swap(other.InFlight);
    }

    TVector<std::pair<TStringBuf, TStringBuf>> TResponse::GetHeaders() const {
        TVector<std::pair<TStringBuf, TStringBuf>> result(Reserve(Response().Headers.Count()));
        for (const auto& header : Response().Headers) {
            result.emplace_back(header.Name(), header.Value());
        }
        return result;
    }

    const TRequestInFlight& TResponse::Request() const {
        if (Empty()) {
            ythrow yexception() << "response not ready";
        } else {
            return *InFlight;
        }
    }

    const NNeh::TResponse& TResponse::Response() const {
        return *Request().Response();
    }

    TRequesterIterator::TRequesterIterator()
        : Requester(nullptr)
        , Deadline()
    {
    }

    TRequesterIterator::TRequesterIterator(TRequester* requester, TInstant deadline)
        : Requester(requester)
        , Deadline(deadline)
    {
    }

    TResponse TRequesterIterator::Next() {
        return Requester->Wait(Deadline);
    }

    THttpRequest& THttpRequest::SetContent(const TString& content, bool compressBody) {
        if (compressBody) {
            static auto snappyCodec = NBlockCodecs::Codec("snappy");
            TStringStream ss;
            NBlockCodecs::TCodedOutput out(&ss, snappyCodec, 32 * 1024 /* like in library/cpp/http */);
            out << content;
            out.Finish();
            Content = ss.Str();
            AddHeader("Content-Encoding", "z-snappy");
        } else {
            Content = content;
        }
        return *this;
    }

    NNeh::TMessage THttpRequest::Create() const {
        NNeh::TMessage msg = NNeh::TMessage::FromString(Addr);
        TString headerBuf;
        TStringOutput headerStream(headerBuf);
        Headers.OutTo(&headerStream);
        headerStream.Finish();
        if (!NNeh::NHttp::MakeFullRequest(msg, headerBuf, Content, ContentType)) {
            ythrow yexception() << "can't create request properly";
        }
        return msg;
    }

    TRequester::TRequester()
        : MultiClient(NNeh::CreateMultiClient())
    {
    }

    TRequester::~TRequester()
    {
        MultiClient->Interrupt();
    }

    void TRequester::SetOption(const TString& key, const TString& value) {
        if (!NNeh::SetProtocolOption(key, value)) {
            ythrow yexception() << "can't set option " << key << " to " << value;
        }
    }

    i64 TRequester::ReserveGroupId() noexcept {
        return AtomicGetAndIncrement(NextRequestGroupId);
    }

    void TRequester::AddToGroup(const TString& addr, const TString& data, i64 groupId, double timeout, PyObject* payload) {
        NNeh::TMessage msg;
        msg.Addr = addr;
        msg.Data = data;
        AddToGroup(msg, groupId, ToDeadline(timeout), payload);
    }

    void TRequester::AddToGroup(const THttpRequest& request, i64 groupId, double timeout, PyObject* payload) {
        const auto msg(request.Create());
        AddToGroup(msg, groupId, ToDeadline(timeout), payload);
    }

    void TRequester::Add(const TString& addr, const TString& data, double timeout, PyObject* payload) {
        NNeh::TMessage msg;
        msg.Addr = addr;
        msg.Data = data;
        Add(msg, ToDeadline(timeout), payload);
    }

    void TRequester::Add(const THttpRequest& request, double timeout, PyObject* payload) {
        const auto msg(request.Create());
        Add(msg, ToDeadline(timeout), payload);
    }

    void TRequester::Add(const NNeh::TMessage& message, TInstant deadline, PyObject* payload) {
        THolder<TRequestInFlight> inFlight(MakeHolder<TRequestInFlight>(message, payload));
        NNeh::IMultiClient::TRequest request(message, deadline, inFlight.Get());
        {
            TGuard<TAdaptiveLock> guard(RequestsLock);
            inFlight->Handle = MultiClient->Request(request);
            UngroupedRequests.PushBack(inFlight.Release());
        }
    }

    void TRequester::AddToGroup(const NNeh::TMessage& message, i64 groupId, TInstant deadline, PyObject* payload) {
        THolder<TRequestInFlight> inFlight(MakeHolder<TRequestInFlight>(message, groupId, payload));
        NNeh::IMultiClient::TRequest request(message, deadline, inFlight.Get());
        {
            TGuard<TAdaptiveLock> guard(RequestsLock);
            inFlight->Handle = MultiClient->Request(request);
            GroupedRequests[groupId].PushBack(inFlight.Release());
        }
    }

    TRequesterIterator TRequester::Iterate(double timeout) {
        return TRequesterIterator(this, ToDeadline(timeout));
    }

    TResponse TRequester::Wait(double timeout) {
        return Wait(ToDeadline(timeout));
    }

    TResponse TRequester::Wait(TInstant deadline) {
        return TResponse(WaitInternal(deadline));
    }

    THolder<TRequestInFlight> TRequester::WaitInternal(TInstant deadline) {
        NNeh::IMultiClient::TEvent event;
        THolder<TRequestInFlight> inFlight;
        if (MultiClient->Wait(event, deadline)) {
            {
                TGuard<TAdaptiveLock> guard(RequestsLock);
                auto inFlightRaw = static_cast<TRequestInFlight*>(event.UserData);
                inFlightRaw->Unlink();
                inFlight.Reset(inFlightRaw);
                if (inFlight->GroupId.Defined()) {
                    auto groupIt = GroupedRequests.find(inFlight->GroupId.GetRef());
                    if (groupIt != GroupedRequests.end() && groupIt->second.Empty()) {
                        GroupedRequests.erase(groupIt);
                    }
                }
            }
            if (event.Type == NNeh::IMultiClient::TEvent::Response) {
                inFlight->Ready = true;
            } else if (event.Type == NNeh::IMultiClient::TEvent::Timeout) {
                inFlight->Ready = true;
                inFlight->Handle->Cancel();
            } else {
                inFlight->Ready = false;
            }
        }
        return inFlight;
    }

    void TRequester::CancelGroup(i64 groupId) {
        TGuard<TAdaptiveLock> guard(RequestsLock);
        auto groupIt = GroupedRequests.find(groupId);
        if (groupIt != GroupedRequests.end()) {
            for (auto& requestInFlight: groupIt->second) {
                requestInFlight.Handle->Cancel();
            }
        }
    }

    void LockAllMemory(bool future) {
        if (future) {
            ::LockAllMemory(ELockAllMemoryFlag::LockCurrentMemory | ELockAllMemoryFlag::LockFutureMemory);
        } else {
            ::LockAllMemory(ELockAllMemoryFlag::LockCurrentMemory);
        }
    }
}
