#pragma once

#include <library/cpp/neh/neh.h>
#include <library/cpp/neh/multiclient.h>
#include <library/cpp/pybind/ptr.h>
#include <util/generic/hash.h>
#include <util/generic/maybe.h>

namespace NNehWrapper {
    // Move assignment operator isn't supported, check https://github.com/cython/cython/issues/1612
    class TRequester;

    class TRequestInFlight : public TIntrusiveListItem<TRequestInFlight> {
    public:
        friend TRequester;

        TRequestInFlight(const NNeh::TMessage& message, PyObject* payload = nullptr)
            : Message(message)
            , Payload(payload)
            , Ready(false)
            , GroupId()
        {
        }

        TRequestInFlight(const NNeh::TMessage& message, i64 groupId, PyObject* payload = nullptr)
            : Message(message)
            , Payload(payload)
            , Ready(false)
            , GroupId(groupId)
        {
        }

        const NNeh::TMessage& GetMessage() const {
            return Message;
        }

        PyObject* GetPayload() const {
            return Payload.Get();
        }

        const NNeh::TResponse* Response() const {
            if (IsReady()) {
                return Handle->Response();
            } else {
                ythrow yexception() << "response not ready";
            }
        }

        bool IsReady() const {
            return Ready;
        }

    private:
        const NNeh::TMessage Message;
        const NPyBind::TPyObjectPtr Payload;
        NNeh::THandleRef Handle;
        bool Ready;
        TMaybe<i64> GroupId;
    };

    class TResponse : public TMoveOnly {
    public:
        TResponse() = default;
        explicit TResponse(THolder<TRequestInFlight> inFlight);
        ~TResponse() = default;

        TResponse& operator=(TResponse& other) {
            Assign(other);
            return *this;
        }

        TResponse& operator=(TResponse&& other) {
            Assign(other);
            return *this;
        }

        bool Empty() const {
            return !InFlight;
        }

        bool IsReady() const {
            return InFlight && InFlight->IsReady();
        }

        bool IsError() const {
            return Response().IsError();
        }

        bool IsCancelled() const {
            return Response().GetErrorType() == NNeh::TError::Cancelled;
        }

        inline i32 GetErrorCode() const {
            return Response().GetErrorCode();
        }

        inline i32 GetSystemErrorCode() const {
            return Response().GetSystemErrorCode();
        }

        inline TString GetErrorText() const {
            return Response().GetErrorText();
        }

        const TString& GetAddr() const {
            return Request().GetMessage().Addr;
        }

        const TString& GetData() const {
            return Response().Data;
        }

        double GetDuration() const {
            return Response().Duration.SecondsFloat();
        }

        TVector<std::pair<TStringBuf, TStringBuf>> GetHeaders() const;

        PyObject* GetPayload() const {
            return Request().GetPayload();
        }

    private:
        const TRequestInFlight& Request() const;

        const NNeh::TResponse& Response() const;

        void Assign(TResponse& other);

        THolder<TRequestInFlight> InFlight;
    };

    class TRequesterIterator {
    public:
        TRequesterIterator();
        TRequesterIterator(TRequester* state, TInstant deadline);
        TRequesterIterator& operator=(const TRequesterIterator& other) = default;

        TResponse Next();

    private:
        TRequester* Requester;
        TInstant Deadline;
    };

    class THttpRequest {
    public:
        THttpRequest& SetAddr(const TString& addr) {
            Addr = addr;
            return *this;
        }

        THttpRequest& SetContent(const TString& content, bool compressBody);

        THttpRequest& SetContentType(const TString& contentType) {
            ContentType = contentType;
            return *this;
        }

        THttpRequest& AddHeader(const TString& name, const TString& value) {
            Headers.AddHeader(name, value);
            return *this;
        }

        NNeh::TMessage Create() const;

    private:
        TString Addr;
        TString Content;
        TString ContentType;
        THttpHeaders Headers;
    };

    class TRequester : public TNonCopyable {
    public:
        TRequester();
        ~TRequester();
        void SetOption(const TString& key, const TString& value);

        /** Thread-safe functions to add requests to the requester */
        void Add(const TString& addr, const TString& data, double timeout = 0.0, PyObject* payload = nullptr);
        void Add(const THttpRequest& request, double timeout = 0.0, PyObject* payload = nullptr);
        void AddToGroup(const TString& addr, const TString& data, i64 groupId, double timeout = 0.0, PyObject* payload = nullptr);
        void AddToGroup(const THttpRequest& request, i64 groupId, double timeout = 0.0, PyObject* payload = nullptr);

        /** Wait should be called from a SINGLE thread **/
        TResponse Wait(double timeout = 0.0);
        TResponse Wait(TInstant deadline);

        /** Iterator should be iterated in a SINGLE thread (the same one that calls Wait()) **/
        TRequesterIterator Iterate(double timeout = 0.0);

        /** Reserve group id to group several requests into a cancellable group **/
        i64 ReserveGroupId() noexcept;

        /** Cancels all requests in a group **/
        void CancelGroup(i64 groupId);

    private:
        using TRequestsList = TIntrusiveListWithAutoDelete<TRequestInFlight, TDelete>;

        void Add(const NNeh::TMessage& message, TInstant deadline, PyObject* payload);
        void AddToGroup(const NNeh::TMessage& message, i64 groupId, TInstant deadline, PyObject* payload);
        THolder<TRequestInFlight> WaitInternal(TInstant deadline);

        NNeh::TMultiClientPtr MultiClient;
        THashMap<i64, TRequestsList> GroupedRequests;
        TRequestsList UngroupedRequests;
        TAdaptiveLock RequestsLock;

        TAtomic NextRequestGroupId;
    };

    void LockAllMemory(bool future=true);
}
