#include <infra/yasm/server/lib/storage/fresh.h>
#include <infra/yasm/server/common/const.h>

#include <library/cpp/testing/benchmark/bench.h>

#include <util/datetime/base.h>
#include <util/generic/vector.h>
#include <util/generic/xrange.h>

using namespace NYasmServer;
using NZoom::NValue::TValue;
using NZoom::NSignal::TSignalName;
using NZoom::NHost::THostName;
using NTags::TInstanceKey;

template <class T>
inline static TVector<T> GenerateNames(size_t count, const std::function<T(size_t i)>& callback) {
    TVector<T> result;
    result.reserve(count);
    for (size_t idx : xrange(count)) {
        result.emplace_back(callback(idx));
    }
    return result;
}

Y_CPU_BENCHMARK(Fresh_SeriesSearch, iface) {
    TFreshStorage storage;
    auto signals = GenerateNames<TSignalName>(10, [](size_t i) { return TSignalName(ToString(i)); });
    auto hosts = GenerateNames<TString>(10, [](size_t i) { return ToString(i); });
    ;

    for (auto tagIdx : xrange(10)) {
        auto instanceKey = TInstanceKey::FromNamed(ToString(tagIdx));
        for (const auto& signal : signals) {
            for (const auto& host : hosts) {
                storage.EmplaceSeries(instanceKey, signal, host, ESeriesKind::Double);
            }
        }
    }

    auto instanceKey = TInstanceKey::FromNamed("5");
    for (size_t i = 0; i < iface.Iterations(); i++) {
        auto ptr = storage.FindSeries(instanceKey, signals[6], hosts[8]);
        Y_DO_NOT_OPTIMIZE_AWAY(ptr != nullptr);
    }
}

Y_CPU_BENCHMARK(Fresh_BatchWrite, iface) {
    TFreshStorage storage;
    TVector<TValue> values;
    for (auto i : xrange(5)) {
        values.emplace_back((double)i);
    }
    auto start = TInstant::Seconds(10000);
    auto interval = TDuration::Seconds(5) * values.size();
    auto instanceKey = TInstanceKey::FromNamed("itype=base");
    TSignalName signal{TStringBuf("random_float_axxx")};
    THostName host(TStringBuf("random.host"));

    // create buffers
    for (auto i : xrange(FRESH_SAMPLES_COUNT / values.size())) {
        storage.PushSignal(instanceKey, signal, host, ESeriesKind::Double, start + interval * i, values);
    }

    // actual benchmark
    for (size_t i = 0; i < iface.Iterations(); i++) {
        auto accepted = storage.PushSignal(instanceKey, signal, host, ESeriesKind::Double,
                                           start + interval * i, values);
        Y_DO_NOT_OPTIMIZE_AWAY(accepted);
    }
}
