#include <infra/yasm/server/lib/storage/histogram_list.h>

#include <infra/yasm/common/interval.h>

#include <library/cpp/testing/benchmark/bench.h>

#include <util/datetime/base.h>

using namespace NYasmServer;
using namespace NYasm::NCommon::NInterval;

Y_CPU_BENCHMARK(HistogramList_WritingSimple_Empty, iface) {
    THistogram hist{TSimpleHistogram()};

    THistogramList storage;
    auto start = GetCurrentIterationStart();

    for (size_t i = 0; i < iface.Iterations(); i++) {
        storage.PushValue(start + ITERATION_SIZE * i, hist);
    }
}

template <bool changing>
inline static void BenchWritingZeroCountSimple(NBench::NCpu::TParams& iface) {
    TSimpleHistogram simple;
    simple.SetZeroCount(10);
    THistogram hist(std::move(simple));

    THistogramList storage;
    auto start = GetCurrentIterationStart();

    for (size_t i = 0; i < iface.Iterations(); i++) {
        storage.PushValue(start + ITERATION_SIZE * i, hist);
        if (changing) {
            hist.MutableSimpleHistogram().SetZeroCount(i + 1); // avoid setting null
        }
    }
}

Y_CPU_BENCHMARK(HistogramList_WritingSimple_ZeroCount_Changing, iface) {
    BenchWritingZeroCountSimple<true>(iface);
}

Y_CPU_BENCHMARK(HistogramList_WritingSimple_ZeroCount_Unchanging, iface) {
    BenchWritingZeroCountSimple<false>(iface);
}

template <bool changing>
inline static void BenchWritingSingleValueSimple(NBench::NCpu::TParams& iface) {
    TSimpleHistogram simple;
    simple.MutableValues().push_back(10.5);
    THistogram hist(std::move(simple));

    THistogramList storage;
    auto start = GetCurrentIterationStart();

    for (size_t i = 0; i < iface.Iterations(); i++) {
        storage.PushValue(start + ITERATION_SIZE * i, hist);
        if (changing) {
            hist.MutableSimpleHistogram().MutableValues()[0] = (double)i;
        }
    }
}

Y_CPU_BENCHMARK(HistogramList_WritingSimple_SingleValue_Changing, iface) {
    BenchWritingSingleValueSimple<true>(iface);
}

Y_CPU_BENCHMARK(HistogramList_WritingSimple_SingleValue_Unchanging, iface) {
    BenchWritingSingleValueSimple<false>(iface);
}

template <bool changing>
inline static void BenchWritingNormalSimple(NBench::NCpu::TParams& iface) {
    TSimpleHistogram simple;
    simple.SetZeroCount(15);
    simple.MutableValues().push_back(10.5);
    simple.MutableValues().push_back(2.5);
    simple.MutableValues().push_back(17.3);
    simple.MutableValues().push_back(16);
    THistogram hist(std::move(simple));

    THistogramList storage;
    auto start = GetCurrentIterationStart();

    for (size_t i = 0; i < iface.Iterations(); i++) {
        storage.PushValue(start + ITERATION_SIZE * i, hist);
        if (changing) {
            hist.MutableSimpleHistogram().MutableValues()[0] = (double)i;
        }
    }
}

Y_CPU_BENCHMARK(HistogramList_WritingSimple_Normal_Changing, iface) {
    BenchWritingNormalSimple<true>(iface);
}

Y_CPU_BENCHMARK(HistogramList_WritingSimple_Normal_Unchanging, iface) {
    BenchWritingNormalSimple<false>(iface);
}

template <bool changing>
inline static void BenchWritingUser(NBench::NCpu::TParams& iface) {
    TUserHistogram user;
    for (size_t i = 0; i < 10; i++) {
        user.MutableBuckets().push_back(TUserHistogram::TBucket{.LowerBound = (double)i, .Weight = (i + 1) * 50});
    }
    THistogram hist(std::move(user));

    THistogramList storage;
    auto start = GetCurrentIterationStart();

    for (size_t i = 0; i < iface.Iterations(); i++) {
        storage.PushValue(start + ITERATION_SIZE * i, hist);
        if (changing) {
            hist.MutableUserHistogram().MutableBuckets()[5].Weight++;
        }
    }
}

Y_CPU_BENCHMARK(HistogramList_WritingUser_Changing, iface) {
    BenchWritingUser<true>(iface);
}

Y_CPU_BENCHMARK(HistogramList_WritingUser_Unchanging, iface) {
    BenchWritingUser<false>(iface);
}

template <bool changing>
inline static void BenchWritingLog(NBench::NCpu::TParams& iface) {
    TLogHistogram log;
    log.SetZeroCount(15);
    log.SetStartPower(-4);
    for (size_t i = 0; i < 10; i++) {
        log.MutableWeights().push_back((i + 1) * 50);
    }
    THistogram hist(std::move(log));

    THistogramList storage;
    auto start = GetCurrentIterationStart();

    for (size_t i = 0; i < iface.Iterations(); i++) {
        storage.PushValue(start + ITERATION_SIZE * i, hist);
        if (changing) {
            hist.MutableLogHistogram().MutableWeights()[5]++;
        }
    }
}

Y_CPU_BENCHMARK(HistogramList_WritingLog_Changing, iface) {
    BenchWritingLog<true>(iface);
}

Y_CPU_BENCHMARK(HistogramList_WritingLog_Unchanging, iface) {
    BenchWritingLog<false>(iface);
}

inline static void BenchReadingSimpleEmpty(TDuration offset, NBench::NCpu::TParams& iface) {
    THistogramList storage;
    auto start = GetCurrentIterationStart();

    THistogram hist{TSimpleHistogram()};
    for (size_t i = 0; i < FRESH_DURATION / ITERATION_SIZE; i++) {
        storage.PushValue(start + ITERATION_SIZE * i, hist);
    }

    for (size_t i = 0; i < iface.Iterations(); i++) {
        Y_DO_NOT_OPTIMIZE_AWAY(storage.GetValueAt(start + offset).Defined());
    }
}

Y_CPU_BENCHMARK(HistogramList_ReadingSimple_Empty_Beginning, iface) {
    BenchReadingSimpleEmpty(TDuration::Zero(), iface);
}

Y_CPU_BENCHMARK(HistogramList_ReadingSimple_Empty_Middle, iface) {
    BenchReadingSimpleEmpty(FRESH_DURATION / 2, iface);
}

Y_CPU_BENCHMARK(HistogramList_ReadingSimple_Empty_End, iface) {
    BenchReadingSimpleEmpty(FRESH_DURATION - ITERATION_SIZE, iface);
}

template <bool changing>
inline static void BenchReadingSimpleZeroCount(TDuration offset, NBench::NCpu::TParams& iface) {
    THistogramList storage;
    auto start = GetCurrentIterationStart();

    THistogram hist{TSimpleHistogram()};
    hist.MutableSimpleHistogram().SetZeroCount(15);
    Y_VERIFY(hist.AsSimpleHistogram().GetSimpleKind() == TSimpleHistogram::EKind::ZeroCount);

    for (size_t i = 0; i < FRESH_DURATION / ITERATION_SIZE; i++) {
        storage.PushValue(start + ITERATION_SIZE * i, hist);
        if (changing) {
            hist.MutableSimpleHistogram().SetZeroCount(i + 1);
        }
    }

    for (size_t i = 0; i < iface.Iterations(); i++) {
        Y_DO_NOT_OPTIMIZE_AWAY(storage.GetValueAt(start + offset).Defined());
    }
}

Y_CPU_BENCHMARK(HistogramList_ReadingSimple_ZeroCount_Unchanging_Beginning, iface) {
    BenchReadingSimpleZeroCount<false>(TDuration::Zero(), iface);
}

Y_CPU_BENCHMARK(HistogramList_ReadingSimple_ZeroCount_Unchanging_Middle, iface) {
    BenchReadingSimpleZeroCount<false>(FRESH_DURATION / 2, iface);
}

Y_CPU_BENCHMARK(HistogramList_ReadingSimple_ZeroCount_Unchanging_End, iface) {
    BenchReadingSimpleZeroCount<false>(FRESH_DURATION - ITERATION_SIZE, iface);
}

Y_CPU_BENCHMARK(HistogramList_ReadingSimple_ZeroCount_Changing_Beginning, iface) {
    BenchReadingSimpleZeroCount<true>(TDuration::Zero(), iface);
}

Y_CPU_BENCHMARK(HistogramList_ReadingSimple_ZeroCount_Changing_Middle, iface) {
    BenchReadingSimpleZeroCount<true>(FRESH_DURATION / 2, iface);
}

Y_CPU_BENCHMARK(HistogramList_ReadingSimple_ZeroCount_Changing_End, iface) {
    BenchReadingSimpleZeroCount<true>(FRESH_DURATION - ITERATION_SIZE, iface);
}

template <bool changing>
inline static void BenchReadingSimpleSingleValue(TDuration offset, NBench::NCpu::TParams& iface) {
    THistogramList storage;
    auto start = GetCurrentIterationStart();

    THistogram hist{TSimpleHistogram()};
    hist.MutableSimpleHistogram().MutableValues().push_back(15.5);
    Y_VERIFY(hist.AsSimpleHistogram().GetSimpleKind() == TSimpleHistogram::EKind::SingleValue);

    for (size_t i = 0; i < FRESH_DURATION / ITERATION_SIZE; i++) {
        storage.PushValue(start + ITERATION_SIZE * i, hist);
        if (changing) {
            hist.MutableSimpleHistogram().MutableValues()[0]++;
        }
    }

    for (size_t i = 0; i < iface.Iterations(); i++) {
        Y_DO_NOT_OPTIMIZE_AWAY(storage.GetValueAt(start + offset).Defined());
    }
}

Y_CPU_BENCHMARK(HistogramList_ReadingSimple_SingleValue_Unchanging_Beginning, iface) {
    BenchReadingSimpleSingleValue<false>(TDuration::Zero(), iface);
}

Y_CPU_BENCHMARK(HistogramList_ReadingSimple_SingleValue_Unchanging_Middle, iface) {
    BenchReadingSimpleSingleValue<false>(FRESH_DURATION / 2, iface);
}

Y_CPU_BENCHMARK(HistogramList_ReadingSimple_SingleValue_Unchanging_End, iface) {
    BenchReadingSimpleSingleValue<false>(FRESH_DURATION - ITERATION_SIZE, iface);
}

Y_CPU_BENCHMARK(HistogramList_ReadingSimple_SingleValue_Changing_Beginning, iface) {
    BenchReadingSimpleSingleValue<true>(TDuration::Zero(), iface);
}

Y_CPU_BENCHMARK(HistogramList_ReadingSimple_SingleValue_Changing_Middle, iface) {
    BenchReadingSimpleSingleValue<true>(FRESH_DURATION / 2, iface);
}

Y_CPU_BENCHMARK(HistogramList_ReadingSimple_SingleValue_Changing_End, iface) {
    BenchReadingSimpleSingleValue<true>(FRESH_DURATION - ITERATION_SIZE, iface);
}

template <bool changing>
inline static void BenchReadingSimpleNormal(TDuration offset, NBench::NCpu::TParams& iface) {
    THistogramList storage;
    auto start = GetCurrentIterationStart();

    THistogram hist{TSimpleHistogram()};
    hist.MutableSimpleHistogram().SetZeroCount(45);
    hist.MutableSimpleHistogram().MutableValues() = {{15.3, 12.7, 76.3, 123.8, 12.54, 13.56, 1365.3, 3123.7, 123.7}};
    Y_VERIFY(hist.AsSimpleHistogram().GetSimpleKind() == TSimpleHistogram::EKind::Normal);

    for (size_t i = 0; i < FRESH_DURATION / ITERATION_SIZE; i++) {
        storage.PushValue(start + ITERATION_SIZE * i, hist);
        if (changing) {
            hist.MutableSimpleHistogram().MutableValues()[5]++;
        }
    }

    for (size_t i = 0; i < iface.Iterations(); i++) {
        Y_DO_NOT_OPTIMIZE_AWAY(storage.GetValueAt(start + offset).Defined());
    }
}

Y_CPU_BENCHMARK(HistogramList_ReadingSimple_Normal_Unchanging_Beginning, iface) {
    BenchReadingSimpleNormal<false>(TDuration::Zero(), iface);
}

Y_CPU_BENCHMARK(HistogramList_ReadingSimple_Normal_Unchanging_Middle, iface) {
    BenchReadingSimpleNormal<false>(FRESH_DURATION / 2, iface);
}

Y_CPU_BENCHMARK(HistogramList_ReadingSimple_Normal_Unchanging_End, iface) {
    BenchReadingSimpleNormal<false>(FRESH_DURATION - ITERATION_SIZE, iface);
}

Y_CPU_BENCHMARK(HistogramList_ReadingSimple_Normal_Changing_Beginning, iface) {
    BenchReadingSimpleNormal<true>(TDuration::Zero(), iface);
}

Y_CPU_BENCHMARK(HistogramList_ReadingSimple_Normal_Changing_Middle, iface) {
    BenchReadingSimpleNormal<true>(FRESH_DURATION / 2, iface);
}

Y_CPU_BENCHMARK(HistogramList_ReadingSimple_Normal_Changing_End, iface) {
    BenchReadingSimpleNormal<true>(FRESH_DURATION - ITERATION_SIZE, iface);
}

template <bool changing>
inline static void BenchReadingUser(TDuration offset, NBench::NCpu::TParams& iface) {
    THistogramList storage;
    auto start = GetCurrentIterationStart();

    THistogram hist{TUserHistogram()};
    for (size_t i = 0; i < 10; i++) {
        hist.MutableUserHistogram().MutableBuckets().push_back(TUserHistogram::TBucket{.LowerBound = (double)i, .Weight = (i + 1) * 50});
    }

    for (size_t i = 0; i < FRESH_DURATION / ITERATION_SIZE; i++) {
        storage.PushValue(start + ITERATION_SIZE * i, hist);
        if (changing) {
            hist.MutableUserHistogram().MutableBuckets()[5].Weight++;
        }
    }

    for (size_t i = 0; i < iface.Iterations(); i++) {
        Y_DO_NOT_OPTIMIZE_AWAY(storage.GetValueAt(start + offset).Defined());
    }
}

Y_CPU_BENCHMARK(HistogramList_ReadingUser_Unchanging_Beginning, iface) {
    BenchReadingUser<false>(TDuration::Zero(), iface);
}

Y_CPU_BENCHMARK(HistogramList_ReadingUser_Unchanging_Middle, iface) {
    BenchReadingUser<false>(FRESH_DURATION / 2, iface);
}

Y_CPU_BENCHMARK(HistogramList_ReadingUser_Unchanging_End, iface) {
    BenchReadingUser<false>(FRESH_DURATION - ITERATION_SIZE, iface);
}

Y_CPU_BENCHMARK(HistogramList_ReadingUser_Changing_Beginning, iface) {
    BenchReadingUser<true>(TDuration::Zero(), iface);
}

Y_CPU_BENCHMARK(HistogramList_ReadingUser_Changing_Middle, iface) {
    BenchReadingUser<true>(FRESH_DURATION / 2, iface);
}

Y_CPU_BENCHMARK(HistogramList_ReadingUser_Changing_End, iface) {
    BenchReadingUser<true>(FRESH_DURATION - ITERATION_SIZE, iface);
}

template <bool changing>
inline static void BenchReadingLog(TDuration offset, NBench::NCpu::TParams& iface) {
    THistogramList storage;
    auto start = GetCurrentIterationStart();

    THistogram hist{TLogHistogram()};

    TLogHistogram log;
    hist.MutableLogHistogram().SetZeroCount(15);
    hist.MutableLogHistogram().SetStartPower(-4);
    for (size_t i = 0; i < 10; i++) {
        hist.MutableLogHistogram().MutableWeights().push_back((i + 1) * 50);
    }

    for (size_t i = 0; i < FRESH_DURATION / ITERATION_SIZE; i++) {
        storage.PushValue(start + ITERATION_SIZE * i, hist);
        if (changing) {
            hist.MutableLogHistogram().MutableWeights()[5]++;
        }
    }

    for (size_t i = 0; i < iface.Iterations(); i++) {
        Y_DO_NOT_OPTIMIZE_AWAY(storage.GetValueAt(start + offset).Defined());
    }
}

Y_CPU_BENCHMARK(HistogramList_ReadingLog_Unchanging_Beginning, iface) {
    BenchReadingLog<false>(TDuration::Zero(), iface);
}

Y_CPU_BENCHMARK(HistogramList_ReadingLog_Unchanging_Middle, iface) {
    BenchReadingLog<false>(FRESH_DURATION / 2, iface);
}

Y_CPU_BENCHMARK(HistogramList_ReadingLog_Unchanging_End, iface) {
    BenchReadingLog<false>(FRESH_DURATION - ITERATION_SIZE, iface);
}

Y_CPU_BENCHMARK(HistogramList_ReadingLog_Changing_Beginning, iface) {
    BenchReadingLog<true>(TDuration::Zero(), iface);
}

Y_CPU_BENCHMARK(HistogramList_ReadingLog_Changing_Middle, iface) {
    BenchReadingLog<true>(FRESH_DURATION / 2, iface);
}

Y_CPU_BENCHMARK(HistogramList_ReadingLog_Changing_End, iface) {
    BenchReadingLog<true>(FRESH_DURATION - ITERATION_SIZE, iface);
}
