#include "handlers.h"

#include "metrics.h"
#include "msgpack_tools.h"
#include "zoom_writers.h"

#include <infra/yasm/common/interval.h>
#include <infra/yasm/server/common/helpers.h>
#include <infra/yasm/zoom/components/compression/zoom_types.h>
#include <infra/yasm/common/points/accumulators/accumulators.h>
#include <infra/yasm/common/points/hgram/ugram/compress/compress.h>
#include <infra/yasm/zoom/components/serialization/common/msgpack_utils.h>

#include <infra/monitoring/common/perf.h>

#include "library/cpp/monlib/metrics/metric_registry.h"

#include <util/generic/hash_set.h>

using NMonitoring::TMeasuredMethod;
using NZoom::NSignal::TSignalName;
using NZoom::NHost::THostName;
using NZoom::NValue::TValue;
using NZoom::NValue::TValueRef;
using NTags::TInstanceKey;

namespace NYasmServer {
    using namespace NYasm::NCommon::NInterval;

    template <class T>
    inline static TVector<T> DeserializeArray(const msgpack::object& val, const std::function<T(TStringBuf value)>& callback) {
        auto& arr = SafeArray(val);
        TVector<T> out;
        out.reserve(arr.size);
        for (const auto& item : TArrayIterator(arr)) {
            out.emplace_back(callback(item.as<TStringBuf>()));
        }
        return out;
    }

    template <class T>
    class TArrayWriter: public ISeriesVisitor {
    public:
        TArrayWriter(msgpack::packer<T>& packer)
            : Packer(packer)
            , Serializer(Packer)
        {
        }

        void OnHeader(TInstant timestamp, size_t count) override {
            if (count == 0) {
                Packer.pack_array(0);
            } else {
                Packer.pack_array(2);
                Packer.pack_uint64(timestamp.Seconds());
                Packer.pack_array(count);
                ExpectedSize = count;
            }
        }

        void OnValue(TValueRef value) override {
            Y_VERIFY(ValuesWritten < ExpectedSize);
            ValuesWritten++;
            value.Update(Serializer);
        }

        ~TArrayWriter() override {
            Y_VERIFY(ValuesWritten == ExpectedSize);
        }

    private:
        size_t ExpectedSize = 0;
        size_t ValuesWritten = 0;
        msgpack::packer<T>& Packer;
        NZoom::NPython::TValueRefSerializer<T, NZoom::NHgram::TUgramCompressor> Serializer;
    };

    inline TMaybe<ESeriesKind> DetermineType(const TVector<TValue>& values) {
        for (const auto& value : values) {
            TValueTyper typer;
            value.Update(typer);

            if (typer.GetType().Defined()) {
                return typer.GetType();
            }
        }
        return Nothing();
    }

    struct TItypeStats {
        ui32 points{0};
        ui32 aggregates{0};
        ui32 commons{0};
    };

    size_t TPushSignalsHandler::ProcessReply(const NMonitoring::TServiceRequest::TRef request) {
        size_t acceptedValues = 0;
        size_t ignoredValues = 0;
        size_t uselessWrites = 0;
        size_t emptyTags = 0;

        auto body = request->GetInput().ReadAll();
        TUnistat::Instance().PushSignalUnsafe(NMetrics::PUSH_SIGNALS_REQUEST_SIZE, body.size());

        auto message = msgpack::unpack(body.data(), body.size());

        EnsureIs(message.get(), msgpack::type::MAP);

        const auto& deserialized = message.get().via.map;

        TInstant timestamp;
        TString host;
        TMaybe<TString> group;
        TMaybe<msgpack::object> valuesRaw;

        for (const auto& pair : TMapIterator(deserialized)) {
            auto key = pair.key.as<TStringBuf>();
            if (key == "group") {
                group = pair.val.as<TString>();
            } else if (key == "host") {
                host = pair.val.as<TString>();
            } else if (key == "time") {
                timestamp = TInstant::Seconds(pair.val.via.u64);
            } else if (key == "values") {
                valuesRaw = pair.val;
            }
        }

        EnsureIs(valuesRaw.GetRef(), msgpack::type::MAP);
        THashMap<TString, TItypeStats> statsByItype;

        for (const auto& keyValues : TMapIterator(valuesRaw.GetRef().via.map)) {
            if (keyValues.val.is_nil()) {
                ++emptyTags;
                continue;
            }
            auto instanceKey = TInstanceKey::FromNamed(keyValues.key.as<TStringBuf>());

            EnsureIs(keyValues.val, msgpack::type::MAP);
            const auto& signalValuesRaw = keyValues.val.via.map;

            ui32 points = 0;
            ui32 commons = 0;
            for (const auto& signalValue : TMapIterator(signalValuesRaw)) {
                TSignalName signalName = signalValue.key.as<TStringBuf>();
                auto value = NZoom::NPython::TMsgPackValueHierarchy::DeserializeValue(signalValue.val, true);
                if (Fresh.PushSignal(instanceKey, signalName, host, timestamp, value)) {
                    ++acceptedValues;
                    ++points;

                    const auto& name = signalName.GetName();
                    if (name.StartsWith(TStringBuf("cpu-"))
                        || name.StartsWith(TStringBuf("cgroup-"))
                        || name.StartsWith(TStringBuf("disk-"))
                        || name.StartsWith(TStringBuf("fdstat-"))
                        || name.StartsWith(TStringBuf("instances-"))
                        || name.StartsWith(TStringBuf("iostat-"))
                        || name.StartsWith(TStringBuf("loadavg-"))
                        || name.StartsWith(TStringBuf("mem-"))
                        || name.StartsWith(TStringBuf("netstat-"))
                        || name.StartsWith(TStringBuf("sockstat-"))
                        || name.StartsWith(TStringBuf("hbf4-"))
                        || name.StartsWith(TStringBuf("hbf6-")))
                    {
                        ++commons;
                    }
                } else {
                    ++ignoredValues;
                }
            }

            auto& stats = statsByItype[instanceKey.GetItype()];
            if (instanceKey.GetAggregated().empty()) {
                stats.aggregates += points;
            } else {
                stats.points += points;
            }
            stats.commons += commons;
        }

        if (group.Defined()) {
            Fresh.EmplaceHostIndex(group.GetRef(), host, timestamp);
        }

        auto lag = Now() - timestamp;
        auto* registry = NMonitoring::TMetricRegistry::Instance();
        for (auto& [itype, stats] : statsByItype) {
            registry->Rate({{"sensor", "tsdb.accepted.points"}, {"itype", itype}})->Add(stats.points);
            registry->Rate({{"sensor", "tsdb.accepted.aggregates"}, {"itype", itype}})->Add(stats.aggregates);
            registry->Rate({{"sensor", "tsdb.accepted.commons"}, {"itype", itype}})->Add(stats.commons);
            registry->HistogramRate({{"sensor", "tsdb.accepted.lag_seconds"}, {"itype", itype}}, []() {
               return NMonitoring::ExponentialHistogram(13, 2, 1);
            })->Record(lag.SecondsFloat(), stats.points);
            registry->Rate({{"sensor", "tsdb.accepted.points"}, {"itype", "total"}})->Add(stats.points);
            registry->Rate({{"sensor", "tsdb.accepted.aggregates"}, {"itype", "total"}})->Add(stats.aggregates);
            registry->Rate({{"sensor", "tsdb.accepted.commons"}, {"itype", "total"}})->Add(stats.commons);
            registry->HistogramRate({{"sensor", "tsdb.accepted.lag_seconds"}, {"itype", "total"}}, []() {
                return NMonitoring::ExponentialHistogram(13, 2, 1);
            })->Record(lag.SecondsFloat(), stats.points);
        }

        TUnistat::Instance().PushSignalUnsafe(NMetrics::PUSH_SIGNALS_EMPTY_TAGS_COUNT, emptyTags);
        TUnistat::Instance().PushSignalUnsafe(NMetrics::PUSH_SIGNALS_USELESS_WRITES_COUNT, uselessWrites);
        TUnistat::Instance().PushSignalUnsafe(NMetrics::PUSH_SIGNALS_ACCEPTED_VALUES_COUNT, acceptedValues);
        TUnistat::Instance().PushSignalUnsafe(NMetrics::PUSH_SIGNALS_IGNORED_VALUES_COUNT, ignoredValues);
        return acceptedValues;
    }

    void TPushSignalsHandler::DoReply(const NMonitoring::TServiceRequest::TRef request, const TParsedHttpFull&) {
        TMeasuredMethod perf(Logger, "", NMetrics::PUSH_SIGNALS_RESPONSE_TIME);

        if (Fresh.IsStopped()) {
            request->Finish(THttpResponse(HttpCodes::HTTP_SERVICE_UNAVAILABLE)
                .SetContent("Storage stopped, writing disabled"));
            return;
        }

        size_t acceptedValues;
        try {
            acceptedValues = ProcessReply(request);
        } catch (...) {
            THttpResponse response(HttpCodes::HTTP_BAD_REQUEST);
            response.SetContent(CurrentExceptionMessage());
            request->Finish(response);
            return;
        }

        request->Finish(THttpResponse(HttpCodes::HTTP_OK).SetContent("Accepted values: " + ToString(acceptedValues)));
    }

    TListAllSignalsHandler::TRequest TListAllSignalsHandler::DeserializeRequest(const TString& body) {
        auto oh = msgpack::unpack(body.data(), body.size());
        TRequest result;
        EnsureIs(oh.get(), msgpack::type::MAP);
        const auto& deserialized = oh.get().via.map;
        for (const auto& pair : TMapIterator(deserialized)) {
            TStringBuf key(pair.key.as<TStringBuf>());
            if (key == TStringBuf("start")) {
                result.Start = NormalizeToIntervalDown(TInstant::Seconds(pair.val.via.u64));
            } else if (key == TStringBuf("end")) {
                result.End = NormalizeToIntervalUp(TInstant::Seconds(pair.val.via.u64));
            }
        }

        if (result.Start > result.End) {
            ythrow yexception() << "Invalid time range, ends must be greater than start";
        }

        return result;
    }

    void TListAllSignalsHandler::DoReply(const NMonitoring::TServiceRequest::TRef request, const TParsedHttpFull&) {
        TMeasuredMethod perf(Logger, "", NMetrics::LIST_ALL_SIGNALS_RESPONSE_TIME);

        auto requestBody = DeserializeRequest(request->GetInput().ReadAll());

        THashMap<THostName, THashMap<TInstanceKey, TVector<TSignalName>>> knownSeries;
        Fresh.IterAllSignals(requestBody.Start, requestBody.End, [&knownSeries](const THostName& host, TInstanceKey instanceKey, TSignalName signal) {
            knownSeries[host][instanceKey].emplace_back(signal);
        });

        msgpack::sbuffer buffer;
        msgpack::packer<msgpack::sbuffer> packer(&buffer);

        packer.pack_map(1);
        PackString(packer, "series");

        packer.pack_map(knownSeries.size());
        for (const auto& hostData : knownSeries) {
            PackString(packer, hostData.first.GetName());

            packer.pack_map(hostData.second.size());
            for (const auto& tagData : hostData.second) {
                PackString(packer, tagData.first.ToNamed());

                packer.pack_array(tagData.second.size());
                for (const auto signal : tagData.second) {
                    PackString(packer, signal.GetName());
                }
            }
        }

        request->Finish(THttpResponse(HttpCodes::HTTP_OK).SetContent(TString(buffer.data(), buffer.size())));
    }

    void TFlushDbHandler::DoReply(const NMonitoring::TServiceRequest::TRef request, const TParsedHttpFull&) {
        Fresh.Clear();
        request->Finish(THttpResponse(HttpCodes::HTTP_OK).SetContent("OK"));
    }

    TFetchHostsHandler::TRequest TFetchHostsHandler::DeserializeRequest(const TString& body) const {
        auto oh = msgpack::unpack(body.data(), body.size());
        TRequest result;
        Y_VERIFY(oh.get().type == msgpack::type::MAP);
        const auto& deserialized = oh.get().via.map;
        for (const auto& pair : TMapIterator(deserialized)) {
            TStringBuf key(pair.key.as<TStringBuf>());
            if (key == TStringBuf("start")) {
                result.Start = TInstant::Seconds(pair.val.via.u64);
            } else if (key == TStringBuf("end")) {
                result.End = TInstant::Seconds(pair.val.via.u64);
            } else if (key == TStringBuf("groups")) {
                result.Groups = DeserializeArray<THostName>(pair.val, [](TStringBuf value) { return THostName(value); });
            } else if (key == TStringBuf("itype")) {
                result.Itype = pair.val.as<TString>();
            }
        }

        return result;
    }

    void TFetchHostsHandler::DoReply(const NMonitoring::TServiceRequest::TRef request, const TParsedHttpFull&) {
        TMeasuredMethod perf(Logger, "", NMetrics::FETCH_HOSTS_RESPONSE_TIME);
        auto requestBody = DeserializeRequest(request->GetInput().ReadAll());
        THashSet<THostName> hosts;
        for (const auto& group : requestBody.Groups) {
            Fresh.IterHostsInGroup(group, requestBody.Start, requestBody.End, [&hosts](THostName host) {
                hosts.emplace(host);
            });
        }

        if (requestBody.Itype) {
            THashSet<THostName> itypeHosts;
            Fresh.IterHostsInItype(requestBody.Itype, [&itypeHosts](THostName host) {
                itypeHosts.emplace(host);
            });

            THashSet<THostName> filteredHosts;
            for (const auto& host : hosts) {
                if (itypeHosts.contains(host)) {
                    filteredHosts.emplace(host);
                }
            }
            hosts.swap(filteredHosts);
        }

        //serialize
        msgpack::sbuffer buffer;
        msgpack::packer<msgpack::sbuffer> packer(&buffer);

        packer.pack_map(1);
        PackString(packer, "hosts");
        packer.pack_array(hosts.size());
        for (const auto& host : hosts) {
            PackString(packer, host.GetName());
        }
        request->Finish(THttpResponse(HttpCodes::HTTP_OK).SetContent(TString(buffer.data(), buffer.size())));
    }

    inline static void DeserializeTags(const msgpack::object& val, TVector<TTagSignal>& out) {
        EnsureIs(val, msgpack::type::MAP);
        const auto& deserialized = val.via.map;

        for (const auto& pair : TMapIterator(deserialized)) {
            auto key = NTags::TRequestKey::FromString(pair.key.as<TString>());
            auto signals = DeserializeArray<TSignalName>(pair.val, [](TStringBuf value) { return TSignalName(value); });
            out.emplace_back(std::move(key), std::move(signals));
        }
    }

    TFetchAggregatedDataHandler::TRequest TFetchAggregatedDataHandler::DeserializeRequest(const TString& body) {
        auto oh = msgpack::unpack(body.data(), body.size());
        TRequest result;
        EnsureIs(oh.get(), msgpack::type::MAP);
        const auto& deserialized = oh.get().via.map;
        for (const auto& pair : TMapIterator(deserialized)) {
            TStringBuf key(pair.key.as<TStringBuf>());
            if (key == TStringBuf("start")) {
                result.Start = NormalizeToIntervalDown(TInstant::Seconds(pair.val.via.u64));
            } else if (key == TStringBuf("end")) {
                result.End = NormalizeToIntervalUp(TInstant::Seconds(pair.val.via.u64));
            } else if (key == TStringBuf("hosts")) {
                result.Hosts = DeserializeArray<THostName>(pair.val, [](TStringBuf value) { return THostName(value); });
            } else if (key == TStringBuf("tags")) {
                DeserializeTags(pair.val, result.Tags);
            }
        }

        if (result.Start > result.End) {
            ythrow yexception() << "Invalid time range, ends must be greater than start";
        }

        return result;
    }

    void TFetchAggregatedDataHandler::DoReply(const NMonitoring::TServiceRequest::TRef request, const TParsedHttpFull&) {
        TMeasuredMethod perf(Logger, "", NMetrics::FETCH_DATA_RESPONSE_TIME);
        auto requestBody = DeserializeRequest(request->GetInput().ReadAll());

        auto [data, seriesRead] = LoadAggregatedData(Fresh, requestBody.Hosts, requestBody.Tags, requestBody.Start, requestBody.End);
        TUnistat::Instance().PushSignalUnsafe(NMetrics::FETCH_DATA_SERIES_READ, seriesRead);

        //serialize
        msgpack::sbuffer buffer;
        msgpack::packer<msgpack::sbuffer> packer(&buffer);

        NZoom::NPython::TValueRefSerializer<msgpack::sbuffer, NZoom::NHgram::TUgramCompressor> serializer(packer);

        packer.pack_array(data.size());
        for (const auto& record : data) {
            packer.pack_array(5);

            PackString(packer, record.Host.GetName());
            PackString(packer, record.RequestKey.ToNamed());
            packer.pack_uint64(record.Timestamp.Seconds());
            packer.pack_uint64(record.MatchedInstances);

            packer.pack_map(record.SignalsData.size());
            for (const auto& signalData : record.SignalsData) {
                PackString(packer, signalData.first.GetName());
                signalData.second.Update(serializer);
            }
        }
        // return list [[hostname, tagname, timestamp, matchedInstances, {signal:value, ...}] ...]
        request->Finish(THttpResponse(HttpCodes::HTTP_OK).SetContent(TString(buffer.data(), buffer.size())));
    }

    void TShutdownHandler::DoReply(const NMonitoring::TServiceRequest::TRef request, const TParsedHttpFull&) {
        Fresh.Shutdown();
        request->Finish(THttpResponse(HttpCodes::HTTP_OK).SetContent("OK"));
    }
} // namespace NYasmServer
