#pragma once

#include "load_signals.h"
#include "proto_handlers.h"

#include "storage/fresh.h"

#include <infra/yasm/interfaces/internal/tsdb.pb.h>
#include <infra/yasm/common/labels/tags/instance_key.h>
#include <infra/yasm/common/points/value/types.h>

#include <infra/monitoring/common/web_handlers.h>
#include <infra/monitoring/common/web_server.h>

#include <contrib/libs/msgpack/include/msgpack.hpp>

namespace NYasmServer {
    using NZoom::NSignal::TSignalName;

    class TFetchAggregatedDataHandler: public NMonitoring::IServiceReplier {
    public:
        TFetchAggregatedDataHandler(TFreshStorage& fresh, TLog& logger)
            : Fresh(fresh)
            , Logger(logger)
        {
        }

        virtual void DoReply(const NMonitoring::TServiceRequest::TRef request, const TParsedHttpFull& meta) override;

    private:
        struct TRequest {
            TInstant Start;
            TInstant End;
            TVector<NZoom::NHost::THostName> Hosts;
            TVector<TTagSignal> Tags;
        };

    private:
        static TRequest DeserializeRequest(const TString& body);

    private:
        TFreshStorage& Fresh;
        TLog& Logger;
    };

    class TPushSignalsHandler: public NMonitoring::IServiceReplier {
    public:
        TPushSignalsHandler(TFreshStorage& fresh, TLog& logger)
            : Fresh(fresh)
            , Logger(logger)
        {
        }

        void DoReply(const NMonitoring::TServiceRequest::TRef request, const TParsedHttpFull& meta) override;

    private:
        size_t ProcessReply(const NMonitoring::TServiceRequest::TRef request);
        TFreshStorage& Fresh;
        TLog& Logger;
    };

    class TListAllSignalsHandler: public NMonitoring::IServiceReplier {
    public:
        TListAllSignalsHandler(TFreshStorage& fresh, TLog& logger)
            : Fresh(fresh)
            , Logger(logger)
        {
        }

        virtual void DoReply(const NMonitoring::TServiceRequest::TRef request, const TParsedHttpFull& meta) override;

    private:
        struct TRequest {
            TInstant Start;
            TInstant End;
        };

    private:
        static TRequest DeserializeRequest(const TString& body);

    private:
        TFreshStorage& Fresh;
        TLog& Logger;
    };

    class TFlushDbHandler: public NMonitoring::IServiceReplier {
    public:
        TFlushDbHandler(TFreshStorage& fresh)
            : Fresh(fresh)
        {
        }

        virtual void DoReply(const NMonitoring::TServiceRequest::TRef request, const TParsedHttpFull& meta) override;

    private:
        TFreshStorage& Fresh;
    };

    class TFetchHostsHandler: public NMonitoring::IServiceReplier {
    public:
        TFetchHostsHandler(TFreshStorage& fresh, TLog& logger)
            : Fresh(fresh)
            , Logger(logger)
        {
        }

        virtual void DoReply(const NMonitoring::TServiceRequest::TRef request, const TParsedHttpFull& meta) override;

    private:
        struct TRequest {
            TInstant Start;
            TInstant End;
            TString Itype;
            TVector<NZoom::NHost::THostName> Groups;
        };

        TRequest DeserializeRequest(const TString& body) const;

    private:
        TFreshStorage& Fresh;
        TLog& Logger;
    };

    class TShutdownHandler: public NMonitoring::IServiceReplier {
    public:
        TShutdownHandler(TFreshStorage& fresh)
            : Fresh(fresh)
        {
        }

        virtual void DoReply(const NMonitoring::TServiceRequest::TRef request, const TParsedHttpFull& meta) override;

    private:
        TFreshStorage& Fresh;
    };

    class TFreshHandlersCollection {
    public:
        TFreshHandlersCollection(TFreshStorage& fresh, TLog& logger)
        {
            // fetch_hosts is called by hserver's crawlers. The collected data is then collected by yasmcollector.
            Handlers.emplace_back("/fetch_hosts", MakeHolder<TFetchHostsHandler>(fresh, logger));

            // most likely unused. new version is push_signals_protobuf.
            Handlers.emplace_back("/push_signals", MakeHolder<TPushSignalsHandler>(fresh, logger));

            // list_all_signals is called by hserver's crawlers. The collected data is then collected by yasmcollector.
            Handlers.emplace_back("/list_all_signals", MakeHolder<TListAllSignalsHandler>(fresh, logger));

            // flush_db called from tests only to cleanup histdb on test setups
            Handlers.emplace_back("/flush_db", MakeHolder<TFlushDbHandler>(fresh));

            // fetch_aggregated is called by hserver when calculating results for top
            Handlers.emplace_back("/fetch_aggregated", MakeHolder<TFetchAggregatedDataHandler>(fresh, logger));

            // most likely unused
            Handlers.emplace_back("/shutdown", MakeHolder<TShutdownHandler>(fresh));

            // push_signals_protobuf is called by yasmserver's requester and middle workers
            Handlers.emplace_back("/push_signals_protobuf", MakeHolder<TPushSignalsProtobufHandler>(fresh, logger));

            // read_aggregated_protobuf is called by histfront to read data
            Handlers.emplace_back("/read_aggregated_protobuf", MakeHolder<TReadAggregatedProtobufHandler>(fresh, logger));
        }

        void Register(NMonitoring::TWebServer& server) {
            for (const auto& pair : Handlers) {
                server.Add(pair.first, *pair.second);
            }
        }

    private:
        TVector<std::pair<TString, THolder<NMonitoring::IServiceReplier>>> Handlers;
    };
} // namespace NYasmServer
