#include "handlers.h"
#include "load_signals.h"
#include "msgpack_tools.h"
#include "storage/counted_sum_list.h"
#include "storage/float_list.h"
#include "storage/histogram_list.h"

#include <infra/yasm/common/labels/tags/instance_key.h>
#include <infra/yasm/zoom/components/record/record.h>
#include <infra/yasm/zoom/python/pipelines/tsdb.h>
#include <infra/yasm/common/tests.h>

#include <library/cpp/http/client/client.h>
#include <library/cpp/json/json_reader.h>
#include <library/cpp/json/json_writer.h>
#include <library/cpp/testing/unittest/gtest.h>
#include <library/cpp/testing/unittest/registar.h>
#include <library/cpp/testing/unittest/tests_data.h>

#include <util/generic/xrange.h>

using namespace NYasmServer;
using namespace NMonitoring;
using NTags::TInstanceKey;
using NZoom::NRecord::TRecord;
using NZoom::NValue::TValue;
using NZoom::NHost::THostName;

namespace {
    class TTestHttpServer {
    public:
        TTestHttpServer(TFreshStorage& fresh)
            : Logger(NYasm::NCommon::NTest::CreateLog())
            , Port(PortManager.GetPort())
            , Server(THttpServerOptions().SetHost("localhost").SetPort(Port).SetThreads(1))
            , Handlers(fresh, Logger)
        {
            Handlers.Register(Server);
            Server.StartServing();
        }

        ~TTestHttpServer() {
            Server.Stop();
        }

        NHttpFetcher::TResultRef SendPostRequest(TStringBuf path, const TString& body, bool protobuf=false) const {
            NHttp::TFetchOptions options;
            options.SetTimeout(TDuration::Seconds(5)).SetPostData(body);
            if (protobuf) {
                options.SetContentType("application/x-protobuf");
            }
            return NHttp::Fetch(NHttp::TFetchQuery(TStringBuilder() << "http://localhost:" << Port << path, options));
        }

        void PushSignals(TInstant time, THostName host, THostName group, const THashMap<TInstanceKey, TRecord>& values) {
            NZoom::NPython::TTsdbRequestState state;
            auto packer(state.CreatePacker(time, host, group));

            packer.SetCount(values.size());
            for (const auto& [ key, record ] : values) {
                packer.AddRecord(key, record);
            }

            auto reply = SendPostRequest("/push_signals_protobuf", state.Serialize(), true);
            UNIT_ASSERT_VALUES_EQUAL(reply->Code, 200);
        }

    private:
        TLog Logger;
        TPortManager PortManager;
        ui16 Port;
        TWebServer Server;
        TFreshHandlersCollection Handlers;
    };

    inline TVector<TString> SortedStringList(const NJson::TJsonValue& result) {
        TVector<TString> output;
        for (const auto& item : result.GetArraySafe()) {
            output.emplace_back(item.GetStringSafe());
        }
        Sort(output);
        return output;
    }
}

Y_UNIT_TEST_SUITE(TestPushSignalsHandler) {
    Y_UNIT_TEST(TestSimplePush) {
        TFreshStorage fresh;
        TTestHttpServer server(fresh);

        TVector<TInstanceKey> keys = {
            TInstanceKey::FromNamed("newsd|ctype=prod"),
            TInstanceKey::FromNamed("newsd|ctype=prestable"),
        };
        auto start = TInstant::Seconds(100000);

        TSignalName signal1(TStringBuf("signal1_axxx"));
        TSignalName signal2(TStringBuf("signal2_hgram"));
        THostName host(TStringBuf("host1"));

        for (size_t i : xrange(5)) {
            for (size_t tagId : xrange(keys.size())) {
                THashMap<TInstanceKey, TRecord> values;
                auto& keyValues = values[keys[tagId]].GetValues();

                keyValues.emplace_back(signal1, TValue(i + 5 * tagId + 1.));

                if (tagId == 0) {
                    auto hgram = NZoom::NHgram::THgram::Small({48, 6, 48, 6}, 0);
                    keyValues.emplace_back(signal2, TValue(std::move(hgram)));
                }

                server.PushSignals(start + ITERATION_SIZE * i, host, TStringBuf(""), values);
            }
        }

        UNIT_ASSERT_VALUES_EQUAL(fresh.GetFloatValue(keys[0], TStringBuf("signal1_axxx"), TStringBuf("host1"), start), 1);
        UNIT_ASSERT_VALUES_EQUAL(fresh.GetFloatValue(keys[1], TStringBuf("signal1_axxx"), TStringBuf("host1"), start + ITERATION_SIZE * 2), 8);

        auto hist = fresh.GetHistogramValue(keys[0], TStringBuf("signal2_hgram"), TStringBuf("host1"), start);
        UNIT_ASSERT(hist.Defined());
        UNIT_ASSERT_EQUAL(hist->GetKind(), EHistogramKind::Simple);
        UNIT_ASSERT_VALUES_EQUAL(hist->AsSimpleHistogram().GetValues(), (TVector<double>{48, 6, 48, 6}));

        UNIT_ASSERT(!fresh.GetFloatValue(keys[1], TStringBuf("signal2_hgram"), TStringBuf("host1"), start).Defined());

        UNIT_ASSERT(!fresh.GetFloatValue(keys[0], TStringBuf("missing_signal_axxx"), TStringBuf("host1"), start).Defined());
        UNIT_ASSERT(!fresh.GetFloatValue(keys[1], TStringBuf("missing_signal_axxx"), TStringBuf("host1"), start).Defined());
    }
}

Y_UNIT_TEST_SUITE(TestPushRequesterSignalsHandler) {
    Y_UNIT_TEST(TestSimplePush) {
        TFreshStorage fresh;
        TTestHttpServer server(fresh);

        auto start = TInstant::Seconds(100000);
        THostName group(TStringBuf("GROUP"));
        THostName host(TStringBuf("host1"));
        TSignalName signal(TStringBuf("some_axxx"));
        auto instanceKey = TInstanceKey::FromNamed("newsd|ctype=prod");

        for (size_t i : xrange(2)) {
            THashMap<TInstanceKey, TRecord> values;
            auto& keyValues = values[instanceKey].GetValues();
            keyValues.emplace_back(signal, TValue(i + 1.));

            server.PushSignals(start + ITERATION_SIZE * i, host, group, values);
        }

        UNIT_ASSERT_VALUES_EQUAL(fresh.GetFloatValue(instanceKey, signal, host, start), 1);
        UNIT_ASSERT_VALUES_EQUAL(fresh.GetFloatValue(instanceKey, signal, host, start + ITERATION_SIZE), 2);

        TVector<THostName> hosts;
        fresh.IterHostsInGroup(group, start, start + ITERATION_SIZE, [&hosts](THostName host) { hosts.emplace_back(host); });
        UNIT_ASSERT_VALUES_EQUAL(hosts.size(), 1);
        UNIT_ASSERT_VALUES_EQUAL(hosts[0], host);
    }
}

Y_UNIT_TEST_SUITE(TestPushRequesterMiddleHandler) {
    Y_UNIT_TEST(TestSimplePush) {
        TFreshStorage fresh;
        TTestHttpServer server(fresh);

        auto start = TInstant::Seconds(100000);
        THostName group(TStringBuf("GROUP"));
        TSignalName signal(TStringBuf("some_axxx"));
        auto instanceKey = TInstanceKey::FromNamed("newsd|ctype=prod");

        for (size_t i : xrange(2)) {
            THashMap<TInstanceKey, TRecord> values;
            values[instanceKey].GetValues().emplace_back(signal, TValue(i + 1.));
            server.PushSignals(start + ITERATION_SIZE * i, group, TStringBuf(""), values);
        }

        UNIT_ASSERT_VALUES_EQUAL(fresh.GetFloatValue(instanceKey, signal, group, start), 1);
        UNIT_ASSERT_VALUES_EQUAL(fresh.GetFloatValue(instanceKey, signal, group, start + ITERATION_SIZE), 2);
    }
}

Y_UNIT_TEST_SUITE(TestFetchHostsHandler) {
    Y_UNIT_TEST(TestNormalFetch) {
        TFreshStorage fresh;
        TTestHttpServer server(fresh);

        TSignalName signal(TStringBuf("signal1_axxx"));
        THostName group(TStringBuf("ASEARCH"));
        auto start = TInstant::Seconds(100000);
        TVector<TInstanceKey> keys = {
            TInstanceKey::FromNamed("newsd|ctype=prod"),
            TInstanceKey::FromNamed("base")};

        {
            THostName host(TStringBuf("host1"));
            THashMap<TInstanceKey, TRecord> values;
            values[keys[0]].GetValues().emplace_back(signal, TValue(1.));
            values[keys[1]].GetValues().emplace_back(signal, TValue(1.));
            server.PushSignals(start, host, group, values);
        }

        {
            THostName host(TStringBuf("host2"));
            THashMap<TInstanceKey, TRecord> values;
            values[keys[0]].GetValues().emplace_back(signal, TValue(1.));
            server.PushSignals(start, host, group, values);
        }

        THashSet<THostName> hosts;
        for(const auto& host : fresh.GetHostsInGroup(TStringBuf("ASEARCH"))) {
            hosts.emplace(host);
        }
        UNIT_ASSERT_VALUES_EQUAL(hosts, (THashSet<THostName>{TStringBuf("host1"), TStringBuf("host2")}));

        // getting without itype
        auto reply = server.SendPostRequest("/fetch_hosts", JsonToMsgpack(R"({
            "start": 100000, "end": 1000000, "groups": ["ASEARCH"]
        })"));
        auto body = MsgpackToJson(reply->Data);
        UNIT_ASSERT_VALUES_EQUAL(SortedStringList(body["hosts"]), (TVector<TString>{"host1", "host2"}));

        // now filtering by itype
        reply = server.SendPostRequest("/fetch_hosts", JsonToMsgpack(R"({
            "start": 100000, "end": 1000000, "groups": ["ASEARCH"], "itype": "base"
        })"));
        body = MsgpackToJson(reply->Data);
        UNIT_ASSERT_VALUES_EQUAL(SortedStringList(body["hosts"]), (TVector<TString>{"host1"}));
    }
}

Y_UNIT_TEST_SUITE(TestFetchDataHandler) {
    Y_UNIT_TEST(TestNormalRead) {
        TFreshStorage fresh;
        TTestHttpServer server(fresh);

        auto instanceKey = TInstanceKey::FromNamed("newsd|ctype=prod");
        auto start = TInstant::Seconds(10000);
        TVector<THostName> hosts = {TStringBuf("ASEARCH"), TStringBuf("random.host")};
        TSignalName signal1(TStringBuf("signal1_axxx"));
        TSignalName signal2(TStringBuf("signal2_hgram"));
        for (size_t i : xrange(3)) {
            for (size_t hostId : xrange(hosts.size())) {
                THashMap<TInstanceKey, TRecord> values;
                auto& keyValues = values[instanceKey].GetValues();
                keyValues.emplace_back(signal1, TValue(1. + i + 3 * hostId));
                if (hostId == 0 || i == 2) {
                    auto hgram = NZoom::NHgram::THgram::Small({48, 6, 48, 6}, 0);
                    keyValues.emplace_back(signal2, TValue(std::move(hgram)));
                }
                server.PushSignals(start + ITERATION_SIZE * i, hosts[hostId], TStringBuf(""), values);
            }
        }

        auto readBody = JsonToMsgpack(R"({
            "start": 10000,
            "end": 10005,
            "tags": {"itype=newsd": ["signal1_axxx", "signal2_hgram", "missing_signal_axxx"]},
            "hosts": ["ASEARCH", "random.host"],
        })");

        auto reply = server.SendPostRequest("/fetch_aggregated", readBody);
        UNIT_ASSERT_VALUES_EQUAL(reply->Code, 200);
        auto json = MsgpackToJson(reply->Data);

        NJson::TJsonValue expected;
        NJson::ReadJsonTree(R"([
        [
            "ASEARCH",
            "itype=newsd",
            10000,
            2,
            {
                "signal1_axxx": 1,
                "signal2_hgram": [[48, 6, 48, 6], 0, null]
            }
        ],
        [
            "ASEARCH",
            "itype=newsd",
            10005,
            2,
            {
                "signal1_axxx": 2,
                "signal2_hgram": [[48, 6, 48, 6], 0, null]
            }
        ],
        [
            "random.host",
            "itype=newsd",
            10000,
            1,
            {
                "signal1_axxx": 4,
                "signal2_hgram": null
            }
        ],
        [
            "random.host",
            "itype=newsd",
            10005,
            1,
            {
                "signal1_axxx": 5,
                "signal2_hgram": null
            }
        ]])", &expected);
        UNIT_ASSERT_VALUES_EQUAL(expected, json);
    }

    Y_UNIT_TEST(TestAgregateData) {
        TFreshStorage fresh;
        TTestHttpServer server(fresh);

        auto start = TInstant::Seconds(10000);
        THostName host(TStringBuf("random.host"));
        TSignalName signal(TStringBuf("signal1_ammm"));

        TVector<TInstanceKey> keys = {
            TInstanceKey::FromNamed("newsd|ctype=prod;geo=none;prj=none;tier=none"),
            TInstanceKey::FromNamed("newsd|ctype=prod;geo=man;prj=none|tier"),
            TInstanceKey::FromNamed("newsd|ctype=prod;geo=ams;prj=none|tier"),
        };

        TVector<double> rawValues = {1, 2, 4};

        for (size_t i : xrange(3)) {
            for (size_t tagId : xrange(keys.size())) {
                THashMap<TInstanceKey, TRecord> values;
                auto& keyValues = values[keys[tagId]].GetValues();

                keyValues.emplace_back(signal, TValue(rawValues[tagId]));

                server.PushSignals(start + ITERATION_SIZE * i, host, TStringBuf(""), values);
            }
        }

        auto readBody = JsonToMsgpack(R"({
            "start": 10000,
            "end": 10015,
            "tags": {"itype=newsd": ["signal1_ammm"]},
            "hosts": ["random.host"],
        })");

        auto reply = server.SendPostRequest("/fetch_aggregated", readBody);
        UNIT_ASSERT_VALUES_EQUAL(reply->Code, 200);
        auto json = MsgpackToJson(reply->Data);

        NJson::TJsonValue expected;
        NJson::ReadJsonTree(R"([
                                ["random.host","itype=newsd",10000,2,{"signal1_ammm":6}],
                                ["random.host","itype=newsd",10005,2,{"signal1_ammm":6}],
                                ["random.host","itype=newsd",10010,2,{"signal1_ammm":6}],
                                ["random.host","itype=newsd",10015,0,{"signal1_ammm":null}]]
                                )",
                            &expected);

        UNIT_ASSERT_VALUES_EQUAL(expected, json);
    }

    Y_UNIT_TEST(CommonOnGroup) {
        TFreshStorage fresh;
        TTestHttpServer server(fresh);

        auto instanceKey = TInstanceKey::FromNamed("newsd|ctype=prod");
        THostName host(TStringBuf("CON"));
        TSignalName signal(TStringBuf("netstat-ibytes_summ"));

        TVector<double> signalValues = {5, 4, 3};
        auto start = TInstant::Seconds(10000);

        for (size_t i : xrange(signalValues.size())) {
            THashMap<TInstanceKey, TRecord> values;
            auto& keyValues = values[instanceKey].GetValues();
            keyValues.emplace_back(signal, TValue(signalValues[i]));
            server.PushSignals(start + ITERATION_SIZE * i, host, TStringBuf(""), values);
        }

        auto readBody = JsonToMsgpack(R"({
            "start": 10000,
            "end": 10015,
            "tags": {"itype=newsd": ["netstat-ibytes_summ"]},
            "hosts": ["CON"],
        })");

        auto reply = server.SendPostRequest("/fetch_aggregated", readBody);
        UNIT_ASSERT_VALUES_EQUAL(reply->Code, 200);
        auto json = MsgpackToJson(reply->Data);

        NJson::TJsonValue expected;
        NJson::ReadJsonTree(R"([
                                ["CON","itype=newsd",10000,1,{"netstat-ibytes_summ":5}],
                                ["CON","itype=newsd",10005,1,{"netstat-ibytes_summ":4}],
                                ["CON","itype=newsd",10010,1,{"netstat-ibytes_summ":3}],
                                ["CON","itype=newsd",10015,0,{"netstat-ibytes_summ":null}]]
                                )",
                            &expected);
        UNIT_ASSERT_VALUES_EQUAL(expected, json);
    }

    Y_UNIT_TEST(TestEmptySumm) {
        TFreshStorage fresh;
        TTestHttpServer server(fresh);

        THostName host(TStringBuf("random.host"));
        auto start = TInstant::Seconds(10000);

        THashMap<TInstanceKey, TRecord> values;
        {
            auto& keyValues = values[TInstanceKey::FromNamed("newsd|geo=man")].GetValues();
            keyValues.emplace_back(TStringBuf("some_mmmm"), TValue(1.));
            keyValues.emplace_back(TStringBuf("some_eeee"), TValue(2.));
        }
        {
            auto& keyValues = values[TInstanceKey::FromNamed("newsd|geo=sas")].GetValues();
            keyValues.emplace_back(TStringBuf("some_mmmm"), TValue(4.));
            keyValues.emplace_back(TStringBuf("some_eeee"), TValue(8.));
        }
        server.PushSignals(start, host, TStringBuf(""), values);

        auto readBody = JsonToMsgpack(R"({
            "start": 10000,
            "end": 10005,
            "tags": {"itype=newsd": ["some_mmmm", "some_eeee"]},
            "hosts": ["random.host"],
        })");

        auto reply = server.SendPostRequest("/fetch_aggregated", readBody);
        UNIT_ASSERT_VALUES_EQUAL(reply->Code, 200);

        auto json = MsgpackToJson(reply->Data);
        NJson::TJsonValue expected;
        NJson::ReadJsonTree(R"([
                                ["random.host","itype=newsd",10000,4,{"some_mmmm":5, "some_eeee":10}],
                                ["random.host","itype=newsd",10005,0,{"some_mmmm":null, "some_eeee":null}]
                                )",
                            &expected);
        UNIT_ASSERT_VALUES_EQUAL(expected, json);
    }

    Y_UNIT_TEST(TestChangedAggregated) {
        TFreshStorage fresh;
        TTestHttpServer server(fresh);

        {
            THashMap<TInstanceKey, TRecord> values;
            values[TInstanceKey::FromNamed("newsd|ctype=prestable;geo=man;prj=none|tier")].GetValues().emplace_back(TStringBuf("some_summ"), TValue(3.));
            values[TInstanceKey::FromNamed("newsd|ctype=prod;geo=man;prj=none|tier")].GetValues().emplace_back(TStringBuf("some_summ"), TValue(5.));
            values[TInstanceKey::FromNamed("newsd|ctype=prod;geo=sas;prj=none|tier")].GetValues().emplace_back(TStringBuf("some_summ"), TValue(7.));
            server.PushSignals(TInstant::Seconds(10000), TStringBuf("random.host"), TStringBuf(""), values);
            server.PushSignals(TInstant::Seconds(10010), TStringBuf("random.host"), TStringBuf(""), values);
        }

        {
            THashMap<TInstanceKey, TRecord> values;
            values[TInstanceKey::FromNamed("newsd|ctype=prestable;prj=none|tier,geo")].GetValues().emplace_back(TStringBuf("some_summ"), TValue(3.));
            values[TInstanceKey::FromNamed("newsd|ctype=prod;prj=none|tier,geo")].GetValues().emplace_back(TStringBuf("some_summ"), TValue(5.));
            server.PushSignals(TInstant::Seconds(10010), TStringBuf("random.host"), TStringBuf(""), values);
        }

        auto readBody = JsonToMsgpack(R"({
            "start": 10000,
            "end": 10010,
            "tags": {"itype=newsd;ctype=prestable,prod": ["some_summ"]},
            "hosts": ["random.host"],
        })");

        auto reply = server.SendPostRequest("/fetch_aggregated", readBody);
        UNIT_ASSERT_VALUES_EQUAL(reply->Code, 200);

        auto json = MsgpackToJson(reply->Data);
        NJson::TJsonValue expected;
        // we do not support reading history before pre-aggregate was added
        NJson::ReadJsonTree(R"([
                                ["random.host","itype=newsd;ctype=prestable,prod",10000,0,{"some_summ":null}],
                                ["random.host","itype=newsd;ctype=prestable,prod",10005,0,{"some_summ":null}],
                                ["random.host","itype=newsd;ctype=prestable,prod",10010,2,{"some_summ":8}]
                                )",
                            &expected);
        UNIT_ASSERT_VALUES_EQUAL(expected, json);
    }
}
