#include "load_signals.h"

#include <infra/yasm/common/const.h>
#include <infra/yasm/common/points/accumulators/impl/average.h>
#include <infra/yasm/common/points/accumulators/accumulators.h>

#include <util/generic/xrange.h>

using namespace NYasmServer;
using namespace NYasm::NCommon;

using NTags::TInstanceKey;
using NTags::TRequestKey;
using NZoom::NAccumulators::EAggregationMethod;
using NZoom::NAccumulators::TAccumulator;
using NZoom::NAccumulators::TAggregationRules;
using NZoom::NSignal::TSignalName;
using NZoom::NValue::TValue;
using NZoom::NValue::TValueRef;
using NZoom::NHost::THostName;

using THostSignalData = THashMap<TInstanceKey, TSeriesPtr>;
using THostData = THashMap<TSignalName, THostSignalData>;

class ISeriesProcessor: public ISeriesVisitor {
public:
    virtual TValueRef GetValue(size_t idx) const = 0;
    virtual size_t GetCount(size_t idx) const = 0;
    virtual bool Has(size_t idx) const = 0;
    virtual void Finish() = 0;
};

class TManySeriesVisitor final : public ISeriesProcessor {
public:
    TManySeriesVisitor(const TAggregationRules* rule, THashSet<size_t>& coveredTimeline, TInstant start, size_t size)
        : CoveredTimeline(coveredTimeline)
        , Start(start)
        , IterPos(0)
        , ProcessedSeries(size, 0)
    {
        auto type = rule->GetAccumulatorType(EAggregationMethod::MetaGroup);

        Accumulators.reserve(size);

        for (size_t t : xrange(size)) {
            Y_UNUSED(t);
            Accumulators.emplace_back(type);
        }

        SeenPositions.reserve(size);
    }

    TValueRef GetValue(size_t idx) const override {
        return Accumulators[idx].GetValue();
    }

    size_t GetCount(size_t idx) const override {
        return ProcessedSeries[idx];
    }

    bool Has(size_t idx) const override {
        return SeenPositions.contains(idx);
    }

    void OnHeader(TInstant start, size_t) override {
        IterPos = (start - Start) / ITERATION_SIZE;
    }

    void OnValue(TValueRef value) override {
        if (!CoveredTimeline.contains(IterPos)) {
            if (value != TValueRef()) {
                Accumulators[IterPos].Mul(value);
                ++ProcessedSeries[IterPos];
                SeenPositions.emplace(IterPos);
            }
        }
        ++IterPos;
    }

    void Finish() override {
        CoveredTimeline.insert(SeenPositions.begin(), SeenPositions.end());
    }

private:
    THashSet<size_t>& CoveredTimeline;
    TInstant Start;
    size_t IterPos;
    TVector<TAccumulator> Accumulators;
    TVector<size_t> ProcessedSeries;
    THashSet<size_t> SeenPositions;
};

class TOneSeriesVisitor final : public ISeriesProcessor {
public:
    TOneSeriesVisitor(THashSet<size_t>& coveredTimeline, TInstant start, size_t size)
        : CoveredTimeline(coveredTimeline)
        , Start(start)
        , IterPos(0)
        , Values(size)
    {
        SeenPositions.reserve(size);
    }

    TValueRef GetValue(size_t idx) const override {
        return Values[idx].GetValue();
    }

    size_t GetCount(size_t idx) const override {
        return Values[idx].GetValue() == TValueRef() ? 0 : 1;
    }

    bool Has(size_t idx) const override {
        return SeenPositions.contains(idx);
    }

    void OnHeader(TInstant start, size_t) override {
        IterPos = (start - Start) / ITERATION_SIZE;
    }

    void OnValue(TValueRef value) override {
        if (!CoveredTimeline.contains(IterPos)) {
            Values[IterPos] = value;
            SeenPositions.emplace(IterPos);
        }
        ++IterPos;
    }

    void Finish() override {
        CoveredTimeline.insert(SeenPositions.begin(), SeenPositions.end());
    }

private:
    THashSet<size_t>& CoveredTimeline;
    TInstant Start;
    size_t IterPos;
    TVector<TValue> Values;
    THashSet<size_t> SeenPositions;
};

class TCollector {
public:
    TCollector(const TFreshStorage& fresh,
               const TVector<THostName>& hosts,
               const TVector<TTagSignal>& tags,
               const TInstant start,
               const TInstant end)
        : Fresh(fresh)
        , Hosts(hosts)
        , Tags(tags)
        , Start(start)
        , End(end)
    {
    }

    std::pair<TVector<TAggregatedSeries>, size_t> Load() const {
        std::pair<TVector<TAggregatedSeries>, size_t> result;
        for (const auto& tagSignals : Tags) {
            result.second += ExtractTagValues(tagSignals.RequestKey, tagSignals.Signals, result.first);
        }
        return result;
    }

private:
    TVector<TSeriesPtr> ExtractSeries(const THostName& host,
                                      const TSignalName& signal,
                                      const TVector<TInstanceKey>& matchingTag) const {
        TVector<TSeriesPtr> values;
        for (auto tag : matchingTag) {
            auto series = Fresh.FindSeries(tag, signal, host);
            if (series != nullptr) {
                values.emplace_back(series);
            }
        }
        return values;
    }

    size_t ExtractTagValues(const TRequestKey& key,
                          const TVector<TSignalName>& signals,
                          TVector<TAggregatedSeries>& result) const {
        size_t timelineSize = (End - Start) / ITERATION_SIZE + 1;
        size_t seriesExtracted = 0;

        for (const auto& host : Hosts) {
            TVector<std::pair<TSignalName, THolder<ISeriesProcessor>>> signalsValues;
            for (auto sig : signals) {
                const TAggregationRules* rule = sig.GetAggregationRules();
                if (rule == nullptr) {
                    continue;
                }

                auto matchingInstanceKeys = GetInstances(host, sig, key);
                if (matchingInstanceKeys.empty()) {
                    continue;
                }

                auto series = ExtractSeries(host, sig, matchingInstanceKeys);
                seriesExtracted += series.size();

                THashSet<size_t> coveredTimeline;
                coveredTimeline.reserve(timelineSize);
                ISeriesProcessor* visitor;
                if (series.size() == 1) {
                    visitor = signalsValues.emplace_back(sig, MakeHolder<TOneSeriesVisitor>(coveredTimeline, Start, timelineSize)).second.Get();
                } else {
                    visitor = signalsValues.emplace_back(sig, MakeHolder<TManySeriesVisitor>(rule, coveredTimeline, Start, timelineSize)).second.Get();
                }

                for (const auto& oneSeries : series) {
                    oneSeries->IterValues(Start, End, *visitor);
                }
                visitor->Finish();
            }

            for (size_t idx : xrange(timelineSize)) {
                THashMap<TSignalName, TValue> signalsData;
                signalsData.reserve(signals.size());

                size_t matchedInstances = 0;
                TInstant timestamp = Start + ITERATION_SIZE * idx;
                for (const auto& [signalName, visitor] : signalsValues) {
                    if (visitor->Has(idx)) {
                        signalsData[signalName] = visitor->GetValue(idx);
                        matchedInstances += visitor->GetCount(idx);
                    } else {
                        signalsData.emplace(signalName, TValue());
                    }
                }

                result.emplace_back(host, timestamp, key, std::move(signalsData), matchedInstances);
            }
        }

        return seriesExtracted;
    }

    TVector<TInstanceKey> GetInstances(const THostName& host,
                                       const TSignalName& signal,
                                       const TRequestKey& key) const {
        NTags::TDynamicFilter filter(key);
        Fresh.IterKnownTags(key.GetItype(), signal, host, [&filter](TInstanceKey ikey) {
            filter.Feed(ikey);
        });
        return filter.Resolve();
    };

private:
    const TFreshStorage& Fresh;
    const TVector<THostName>& Hosts;
    const TVector<TTagSignal>& Tags;
    const TInstant Start;
    const TInstant End;
};

std::pair<TVector<TAggregatedSeries>, size_t> NYasmServer::LoadAggregatedData(const TFreshStorage& fresh,
                                                                              const TVector<THostName>& hosts,
                                                                              const TVector<TTagSignal>& tags,
                                                                              const TInstant start,
                                                                              const TInstant end) {
    TCollector collector(fresh, hosts, tags, start, end);
    return collector.Load();
}
