#include "msgpack_tools.h"

#include <library/cpp/json/json_reader.h>

namespace NYasmServer {

    using NJson::EJsonValueType;
    using NJson::TJsonValue;

    template <class T>
    inline void JsonToMsgpack(msgpack::packer<T>& packer, const TJsonValue& obj) {
        switch (obj.GetType()) {
            case EJsonValueType::JSON_NULL: {
                packer.pack_nil();
                break;
            }
            case EJsonValueType::JSON_BOOLEAN: {
                if (obj.GetBoolean()) {
                    packer.pack_true();
                } else {
                    packer.pack_false();
                }
                break;
            }
            case EJsonValueType::JSON_UINTEGER: {
                packer.pack_uint64(obj.GetUInteger());
                break;
            }
            case EJsonValueType::JSON_INTEGER: {
                packer.pack_int64(obj.GetInteger());
                break;
            }
            case EJsonValueType::JSON_DOUBLE: {
                packer.pack_double(obj.GetDouble());
                break;
            }
            case EJsonValueType::JSON_STRING: {
                PackString(packer, obj.GetString());
                break;
            }
            case EJsonValueType::JSON_MAP: {
                auto& map = obj.GetMap();
                packer.pack_map(map.size());
                for (const auto& pair : map) {
                    PackString(packer, pair.first);
                    JsonToMsgpack(packer, pair.second);
                }
                break;
            }
            case EJsonValueType::JSON_ARRAY: {
                auto& arr = obj.GetArray();
                packer.pack_array(arr.size());
                for (const auto& element : arr) {
                    JsonToMsgpack(packer, element);
                }
                break;
            }
            default: {
                ythrow yexception() << "wrong object given";
            }
        }
    }

    TString JsonToMsgpack(const TString& json) {
        TJsonValue parsed;
        NJson::ReadJsonTree(json, &parsed);
        msgpack::sbuffer buf;
        msgpack::packer<msgpack::sbuffer> packer(&buf);
        JsonToMsgpack(packer, parsed);
        return TString(buf.data(), buf.size());
    }

    TJsonValue MsgpackToJson(const msgpack::object& val) {
        switch (val.type) {
            case msgpack::type::NIL:
                return TJsonValue(EJsonValueType::JSON_NULL);
            case msgpack::type::BOOLEAN:
                return TJsonValue(val.via.boolean);
            case msgpack::type::POSITIVE_INTEGER:
                return TJsonValue(val.via.u64);
            case msgpack::type::NEGATIVE_INTEGER:
                return TJsonValue(val.via.i64);
            case msgpack::type::FLOAT:
                return TJsonValue(val.via.f64);
            case msgpack::type::STR:
                return TJsonValue(val.as<TStringBuf>());
            case msgpack::type::ARRAY: {
                TJsonValue arr(EJsonValueType::JSON_ARRAY);
                for (const auto& src : TArrayIterator(val.via.array)) {
                    arr.AppendValue(MsgpackToJson(src));
                }
                return arr;
            }
            case msgpack::type::MAP: {
                TJsonValue map(EJsonValueType::JSON_MAP);
                for (const auto& src : TMapIterator(val.via.map)) {
                    map.InsertValue(src.key.as<TStringBuf>(), MsgpackToJson(src.val));
                }
                return map;
            }
            default:
                ythrow yexception() << "Not supported";
        }
    }

    TJsonValue MsgpackToJson(const TString& data) {
        auto oh = msgpack::unpack(data.data(), data.size());
        return MsgpackToJson(oh.get());
    }
} // namespace NYasmServer
