#include "proto_handlers.h"

#include <infra/yasm/common/const.h>
#include <infra/yasm/common/interval.h>
#include <infra/yasm/common/labels/tags/dynamic_filter.h>
#include <infra/yasm/common/points/accumulators/accumulators.h>
#include <infra/yasm/zoom/components/serialization/zoom_to_protobuf/record.h>
#include <infra/yasm/zoom/components/serialization/zoom_to_protobuf/instance_key.h>
#include <infra/yasm/zoom/components/serialization/history/history.h>
#include <infra/monitoring/common/perf.h>

#include "library/cpp/monlib/metrics/metric_registry.h"

using namespace NYasmServer;
using namespace NYasm::NCommon;
using namespace NYasm::NCommon::NInterval;
using namespace NZoom::NProtobuf;
using namespace NZoom::NAccumulators;
using namespace NTags;

namespace {
    using EStatusCode = NYasm::NInterfaces::NInternal::THistoryAggregatedSeries::EStatusCode;

    static constexpr i64 MAX_SERIES_COUNT = 100000;
    static constexpr i64 MAX_SERIES_PER_REQUEST = 1000;
    static constexpr TDuration READ_TIMEOUT = TDuration::Seconds(5);

    size_t GetOffset(TInstant timestamp, const THistoryRequest& request) {
        return (timestamp - request.Start).GetValue() / request.Period.GetValue();
    }

    size_t GetNormalizedOffset(TInstant timestamp, const THistoryRequest& request) {
        return (timestamp - NormalizeToIntervalDown(request.Start, request.Period)).GetValue() / request.Period.GetValue();
    }

    size_t GetNormalizedPoints(const THistoryRequest& request) {
        TInstant start = NormalizeToIntervalDown(request.Start, request.Period);
        TInstant end = NormalizeToIntervalUp(request.End, request.Period);
        return (end + request.Period - start).GetValue() / request.Period.GetValue();
    }

    struct TRollupMerger final : public ISeriesVisitor {
        TRollupMerger(const THistoryRequest& request)
            : Request(request)
            , AccumulatorType(Request.SignalName.GetAggregationType(EAggregationMethod::Rollup).GetRef())
            , Accumulators(
                AccumulatorType,
                GetNormalizedPoints(Request),
                Request.DontAggregate ? ENullPolicy::Nullable : ENullPolicy::NonNullable)
        {
        }

        void OnHeader(TInstant start, size_t) override {
            Timestamp = start;
        }

        void OnValue(NZoom::NValue::TValueRef value) override {
            auto offset = GetNormalizedOffset(Timestamp, Request);
            EndOffset = offset + 1;

            Accumulators.Mul(value, offset);
            Timestamp += ITERATION_SIZE;
        }

        const THistoryRequest& Request;
        NZoom::NAccumulators::EAccumulatorType AccumulatorType;
        NZoom::NAccumulators::TCompactAccumulatorsArray Accumulators;
        TInstant Timestamp{};
        size_t EndOffset{0};
    };

    struct TMetagroupMerger final : public ISeriesVisitor {
        TMetagroupMerger(const THistoryRequest& request)
            : Request(request)
            , AccumulatorType(Request.SignalName.GetAggregationType(EAggregationMethod::MetaGroup).GetRef())
            , Accumulators(
                AccumulatorType,
                GetNormalizedPoints(Request),
                Request.DontAggregate ? ENullPolicy::Nullable : ENullPolicy::NonNullable)
        {
        }

        void OnHeader(TInstant start, size_t) override {
            Timestamp = start;
        }

        void OnValue(NZoom::NValue::TValueRef value) override {
            auto offset = GetOffset(Timestamp, Request);
            EndOffset = Max(EndOffset, offset + 1);

            Accumulators.Mul(value, offset);
            Timestamp += Request.Period;
        }

        TInstant GetStart() const {
            return NormalizeToIntervalDown(Request.Start, Request.Period);
        }

        void Merge(const TRollupMerger& merger) {
            OnHeader(GetStart(), /*not used*/0);
            auto endOffset = Request.DontAggregate ? merger.EndOffset : merger.Accumulators.Len();

            for (const auto offset : xrange(merger.Accumulators.Len())) {
                if (offset >= endOffset) {
                    break;
                }

                OnValue(merger.Accumulators.GetValue(offset));
            }
        }

        size_t Aggregate(TFreshStorage& fresh) {
            TDynamicFilter filter(Request.RequestKey.GetRequestKey());

            bool exactMatch = Request.DontAggregate;
            fresh.IterKnownTags(Request.RequestKey.GetRequestKey().GetItype(), Request.SignalName, Request.HostName, [&filter, exactMatch](TInstanceKey key) {
                filter.Feed(key, exactMatch);
            });

            size_t seriesCount = 0;
            auto instanceKeys = filter.Resolve();
            for (const auto& instanceKey : instanceKeys) {
                if (seriesCount >= MAX_SERIES_PER_REQUEST) {
                    StatusCode = EStatusCode::THistoryAggregatedSeries_EStatusCode_LIMIT_EXCEEDED;
                    break;
                }
                auto series = fresh.FindSeries(instanceKey, Request.SignalName, Request.HostName);
                if (series) {
                    ++seriesCount;
                    if (Request.Period > ITERATION_SIZE) {
                        TRollupMerger rollupMerger(Request);
                        series->IterValues(Request.Start, Request.End, rollupMerger);
                        Merge(rollupMerger);
                    } else {
                        series->IterValues(Request.Start, Request.End, *this);
                    }
                }
            }

            if (0 == seriesCount) {
                StatusCode = EStatusCode::THistoryAggregatedSeries_EStatusCode_NOT_FOUND;
            }
            return seriesCount;
        }

        THistoryResponse CreateResponse() const {
            auto endOffset = Request.DontAggregate ? EndOffset : Accumulators.Len();
            TVector<TValue> values(::Reserve(endOffset));

            for (const auto offset : xrange(Accumulators.Len())) {
                if (offset >= endOffset) {
                    break;
                }

                values.emplace_back(Accumulators.GetValue(offset));
            }
            return THistoryResponse(Request, GetStart(), std::move(values), StatusCode);
        }

        const THistoryRequest& Request;
        NZoom::NAccumulators::EAccumulatorType AccumulatorType;
        NZoom::NAccumulators::TCompactAccumulatorsArray Accumulators;
        TInstant Timestamp{};
        size_t EndOffset{0};

        EStatusCode StatusCode = EStatusCode::THistoryAggregatedSeries_EStatusCode_OK;
    };

    struct TItypeStats {
        ui32 points{0};
        ui32 aggregates{0};
        ui32 commons{0};
    };
}

void TPushSignalsProtobufHandler::Handle(TProtoContext& context) {
    if (Fresh.IsStopped()) {
        context.Response.SetStatus(NYasm::NInterfaces::NInternal::TSDB_STOPPED);
        context.Response.SetMessage("Storage stopped, writing disabled");
        context.ReplyingContext.ResponceCode = HttpCodes::HTTP_SERVICE_UNAVAILABLE;
        return;
    }

    NZoom::NProtobuf::THostNameDeserializer hostNameDeserializer(context.Request.GetHostNameTable());
    NZoom::NProtobuf::TInstanceKeyDeserializer instanceKeyDeserializer(context.Request.GetInstanceKeyTable());
    NZoom::NProtobuf::TSignalNameDeserializer signalNameDeserializer(context.Request.GetSignalNameTable());

    TInstant timestamp;
    TVector<NZoom::NProtobuf::TProtobufWrappingValue> valueWrappers(Reserve(context.Request.GetSeriesCount()));
    TVector<TSingleValueRefWithKey> flatValues(Reserve(context.Request.GetSeriesCount())); // references items from valueWrappers
    THashMap<TString, TItypeStats> statsByItype;
    for (const auto& hostRecord : context.Request.GetHostRecords()) {
        timestamp = TInstant::Seconds(hostRecord.GetTimestamp());

        const bool isGroup = hostRecord.GetIsGroup();
        const auto hostName = hostNameDeserializer.Deserialize(isGroup ? hostRecord.GetGroup() : hostRecord.GetHost());
        if (!isGroup) {
            Fresh.EmplaceHostIndex(hostNameDeserializer.Deserialize(hostRecord.GetGroup()), hostName, timestamp);
        }

        for (const auto& tagRecord : hostRecord.GetTagRecords()) {
            const auto instanceKey(instanceKeyDeserializer.Deserialize(tagRecord.GetInstanceKey()));
            ui32 points = 0;
            ui32 commons = 0;
            NZoom::NProtobuf::DeserializeRecordsByWrappingValuesAndVisit(tagRecord.GetRecord(), signalNameDeserializer,
                [&valueWrappers, &flatValues, hostName, instanceKey, &points, &commons](auto signalName, const auto& value) {
                    valueWrappers.push_back(value);
                    flatValues.emplace_back(signalName, hostName, instanceKey, valueWrappers.back());
                    ++points;

                    const auto& name = signalName.GetName();
                    if (name.StartsWith(TStringBuf("cpu-"))
                        || name.StartsWith(TStringBuf("cgroup-"))
                        || name.StartsWith(TStringBuf("disk-"))
                        || name.StartsWith(TStringBuf("fdstat-"))
                        || name.StartsWith(TStringBuf("instances-"))
                        || name.StartsWith(TStringBuf("iostat-"))
                        || name.StartsWith(TStringBuf("loadavg-"))
                        || name.StartsWith(TStringBuf("mem-"))
                        || name.StartsWith(TStringBuf("netstat-"))
                        || name.StartsWith(TStringBuf("sockstat-"))
                        || name.StartsWith(TStringBuf("hbf4-"))
                        || name.StartsWith(TStringBuf("hbf6-")))
                    {
                        ++commons;
                    }
                }
            );

            auto& stats = statsByItype[instanceKey.GetItype()];
            if (instanceKey.GetAggregated().empty()) {
                stats.aggregates += points;
            } else {
                stats.points += points;
            }
            stats.commons += commons;
        }
    }

    size_t totalValues = flatValues.size();
    NYasmServer::AcceptanceStat acceptanceStat;
    size_t ignoredValues = 0;

    try {
        if (!flatValues.empty()) {
            acceptanceStat = Fresh.PushSignal(timestamp, flatValues);
            ignoredValues = totalValues - acceptanceStat.accepted;
        }
    } catch (...) {
        context.Response.SetStatus(NYasm::NInterfaces::NInternal::TSDB_FAILED);
        context.Response.SetMessage(CurrentExceptionMessage());
        context.ReplyingContext.ResponceCode = HttpCodes::HTTP_SERVICE_UNAVAILABLE;
        return;
    }

    context.Response.SetAcceptedValues(acceptanceStat.accepted);
    context.Response.SetIgnoredValues(ignoredValues);
    context.Response.SetStatus(NYasm::NInterfaces::NInternal::TSDB_OK);

    auto lag = Now() - timestamp;
    auto* registry = NMonitoring::TMetricRegistry::Instance();
    for (auto& [itype, stats] : statsByItype) {
        registry->Rate({{"sensor", "tsdb.accepted.points"}, {"itype", itype}})->Add(stats.points);
        registry->Rate({{"sensor", "tsdb.accepted.aggregates"}, {"itype", itype}})->Add(stats.aggregates);
        registry->Rate({{"sensor", "tsdb.accepted.commons"}, {"itype", itype}})->Add(stats.commons);
        registry->HistogramRate({{"sensor", "tsdb.accepted.lag_seconds"}, {"itype", itype}}, []() {
            return NMonitoring::ExponentialHistogram(13, 2, 1);
        })->Record(lag.SecondsFloat(), stats.points);
        registry->Rate({{"sensor", "tsdb.accepted.points"}, {"itype", "total"}})->Add(stats.points);
        registry->Rate({{"sensor", "tsdb.accepted.aggregates"}, {"itype", "total"}})->Add(stats.aggregates);
        registry->Rate({{"sensor", "tsdb.accepted.commons"}, {"itype", "total"}})->Add(stats.commons);
        registry->HistogramRate({{"sensor", "tsdb.accepted.lag_seconds"}, {"itype", "total"}}, []() {
            return NMonitoring::ExponentialHistogram(13, 2, 1);
        })->Record(lag.SecondsFloat(), stats.points);
    }

    TUnistat::Instance().PushSignalUnsafe(NMetrics::GROUP_SMALL_HGRAMS, acceptanceStat.acceptedGroupSmallHgram);
    TUnistat::Instance().PushSignalUnsafe(NMetrics::GROUP_HGRAMS, acceptanceStat.acceptedGroupSmallHgram + acceptanceStat.acceptedGroupNormalHgram);
    TUnistat::Instance().PushSignalUnsafe(NMetrics::HOST_SMALL_HGRAMS, acceptanceStat.acceptedHostSmallHgram);
    TUnistat::Instance().PushSignalUnsafe(NMetrics::HOST_HGRAMS, acceptanceStat.acceptedHostSmallHgram + acceptanceStat.acceptedHostNormalHgram);

    TUnistat::Instance().PushSignalUnsafe(NMetrics::PUSH_SIGNALS_ACCEPTED_VALUES_COUNT, acceptanceStat.accepted);
    TUnistat::Instance().PushSignalUnsafe(NMetrics::PUSH_SIGNALS_IGNORED_VALUES_COUNT, ignoredValues);
}

void TReadAggregatedProtobufHandler::Handle(TProtoContext& context) {
    context.ReplyingContext.RequestId = context.Request.GetRequestId();

    TInstant deadline(READ_TIMEOUT.ToDeadLine());
    if (context.Request.GetDeadlineTimestamp()) {
        deadline = Min(deadline, TInstant::Seconds(context.Request.GetDeadlineTimestamp()));
    }

    if (Fresh.IsStopped() || TInstant::Now() > deadline) {
        context.ReplyingContext.ResponceCode = HttpCodes::HTTP_SERVICE_UNAVAILABLE;
        return;
    }

    size_t totalSeriesCount = 0;
    TVector<THistoryRequest> requests(THistoryRequest::FromProto(context.Request));
    THistoryResponseWriter responseWriter(context.Response);
    responseWriter.Reserve(requests.size());
    for (const auto& request : requests) {
        if (request.SignalName.IsOld()) {
            THistoryResponse response(request, EStatusCode::THistoryAggregatedSeries_EStatusCode_NOT_FOUND);
            responseWriter.Add(response);
            continue;
        } else if (request.Start > request.End || request.Period.GetValue() % ITERATION_SIZE.GetValue() != 0) {
            THistoryResponse response(request, EStatusCode::THistoryAggregatedSeries_EStatusCode_INTERNAL_ERROR);
            responseWriter.Add(response);
            continue;
        } else if (TInstant::Now() > deadline) {
            THistoryResponse response(request, EStatusCode::THistoryAggregatedSeries_EStatusCode_TIME_EXCEEDED);
            responseWriter.Add(response);
            continue;
        } else if (totalSeriesCount > MAX_SERIES_COUNT) {
            THistoryResponse response(request, EStatusCode::THistoryAggregatedSeries_EStatusCode_LIMIT_EXCEEDED);
            responseWriter.Add(response);
            continue;
        }

        TMetagroupMerger merger(request);
        try {
            totalSeriesCount += merger.Aggregate(Fresh);
        } catch(...) {
            merger.StatusCode = EStatusCode::THistoryAggregatedSeries_EStatusCode_INTERNAL_ERROR;
            Logger << TLOG_ERR << "Can't aggregate " << request.RequestKey.GetRequestKey() << ":" << request.SignalName
                   << " because of " << CurrentExceptionMessage();
        }

        THistoryResponse response(merger.CreateResponse());
        responseWriter.Add(response);
    }

    TUnistat::Instance().PushSignalUnsafe(NMetrics::READ_AGGREGATED_SERIES_READ, totalSeriesCount);
}
