#include "proto_handlers.h"

#include <infra/monitoring/common/proto_replier_ut.h>
#include <infra/yasm/zoom/components/serialization/history/history.h>
#include <infra/yasm/zoom/components/record/record.h>
#include <infra/yasm/zoom/python/pipelines/tsdb.h>
#include <infra/yasm/common/const.h>
#include <infra/yasm/common/tests.h>

#include <library/cpp/logger/stream.h>
#include <library/cpp/testing/unittest/gtest.h>
#include <library/cpp/testing/unittest/registar.h>

using namespace NYasmServer;
using namespace NMonitoring;
using namespace NTags;
using namespace NZoom::NHost;
using namespace NZoom::NSignal;
using namespace NZoom::NRecord;
using namespace NZoom::NValue;
using namespace NZoom::NProtobuf;
using NYasm::NCommon::ITERATION_SIZE;
using NYasm::NInterfaces::NInternal::THistoryReadAggregatedRequest;
using NYasm::NInterfaces::NInternal::THistoryReadAggregatedResponse;
using NYasm::NInterfaces::NInternal::TTsdbPushSignalRequest;
using NYasm::NInterfaces::NInternal::TTsdbPushSignalResponse;
using EStatusCode = NYasm::NInterfaces::NInternal::THistoryAggregatedSeries::EStatusCode;

namespace {
    static const THostName GROUP_NAME(TStringBuf("SAS.000"));
    static const THostName HOST_NAME(TStringBuf("sas1-1234.search.yandex.net"));
    static const TSignalName SIGNAL_NAME(TStringBuf("signal_summ"));

    struct TRecordsBuilder {
        void Add(TInstanceKey instanceKey, TSignalName signal, TValue value) {
            Records[instanceKey].GetValues().emplace_back(signal, std::move(value));
        }

        THashMap<TInstanceKey, TRecord> Records;
    };

    class TFreshClient {
    public:
        TFreshClient()
            : Logger(NYasm::NCommon::NTest::CreateLog())
            , Fresh(Logger)
            , PushHandler(Fresh, Logger)
            , ReadHandler(Fresh, Logger)
        {
        }

        void RawPush(const TTsdbPushSignalRequest& request, TTsdbPushSignalResponse& response) {
            auto httpResponse = NMonitoring::THttpRequestBuilder()
                .FromProtoRequest(request)
                .Execute(PushHandler);
            UNIT_ASSERT_VALUES_EQUAL(httpResponse.Code, 200);
            httpResponse.ToProto(response);
        }

        void RawRead(const THistoryReadAggregatedRequest& request, THistoryReadAggregatedResponse& response) {
            auto httpResponse = NMonitoring::THttpRequestBuilder()
                .FromProtoRequest(request)
                .Execute(ReadHandler);
            UNIT_ASSERT_VALUES_EQUAL(httpResponse.Code, 200);
            httpResponse.ToProto(response);
        }

        void PushSignals(TInstant timestamp, THostName host, THostName group, const THashMap<TInstanceKey, TRecord>& records) {
            NZoom::NPython::TTsdbRequestState state;
            auto packer(state.CreatePacker(timestamp, host, group));

            packer.SetCount(records.size());
            for (const auto& [ key, record ] : records) {
                packer.AddRecord(key, record);
            }

            state.FillRequest();
            TTsdbPushSignalResponse response;
            RawPush(state.GetRequest(), response);
        }

        TVector<THistoryResponse> ReadSignals(const TVector<THistoryRequest>& historyRequests) {
            THistoryReadAggregatedRequest request;
            THistoryReadAggregatedResponse response;

            {
                THistoryRequestWriter requestWriter(request);
                requestWriter.Reserve(historyRequests.size());
                for (const auto& historyRequest : historyRequests) {
                    requestWriter.Add(historyRequest);
                }
            }

            RawRead(request, response);
            return THistoryResponse::FromProto(response);
        }

        THistoryResponse ReadOneSignal(const THistoryRequest& historyRequest) {
            auto responses(ReadSignals(TVector<THistoryRequest>{historyRequest}));
            UNIT_ASSERT_VALUES_EQUAL(responses.size(), 1);
            return std::move(responses.back());
        }

    private:
        TLog Logger;
        TFreshStorage Fresh;
        TPushSignalsProtobufHandler PushHandler;
        TReadAggregatedProtobufHandler ReadHandler;
    };
}

enum class EAggregationPolicy {
    Aggregate,
    DontAggregate,
};

Y_UNIT_TEST_SUITE(TestReadAggregatedHandler) {
    Y_UNIT_TEST(TestSimpleReading) {
        TFreshClient client;

        TInstant start = TInstant::Seconds(100000);
        TInstanceKey instanceKey = TInstanceKey::FromNamed("newsd|ctype=prod");
        TRecordsBuilder builder;
        builder.Add(instanceKey, SIGNAL_NAME, TValue(42));
        client.PushSignals(start, HOST_NAME, GROUP_NAME, builder.Records);

        THistoryRequest request{
            .Start = start,
            .End = start,
            .Period = ITERATION_SIZE,
            .HostName = HOST_NAME,
            .RequestKey = TStringBuf("itype=newsd"),
            .SignalName = SIGNAL_NAME
        };
        auto response(client.ReadOneSignal(request));
        UNIT_ASSERT_VALUES_EQUAL(response.Series.GetStartTimestamp(), start);
        UNIT_ASSERT_VALUES_EQUAL(response.StatusCode, EStatusCode::THistoryAggregatedSeries_EStatusCode_OK);

        TVector<TValue> expectedValues;
        expectedValues.emplace_back(TValue(42));
        UNIT_ASSERT_VALUES_EQUAL(response.Series.GetValues(), expectedValues);
    }

    Y_UNIT_TEST(TestTimeRangeWithAbsentPoints) {
        auto fetchMetrics = [&](EAggregationPolicy aggrPolicy) {
            TFreshClient client;

            TInstant start = TInstant::Seconds(100000);
            TInstanceKey instanceKey = TInstanceKey::FromNamed("newsd|ctype=prod");
            auto valuesCnt = 3u;
            for (size_t i = 0; i != valuesCnt; ++i) {
                TRecordsBuilder builder;
                builder.Add(instanceKey, SIGNAL_NAME, TValue(i));
                client.PushSignals(start + ITERATION_SIZE * i, HOST_NAME, GROUP_NAME, builder.Records);
            }

            THistoryRequest request{
                .Start = start,
                .End = start + ITERATION_SIZE * valuesCnt,
                .Period = ITERATION_SIZE,
                .HostName = HOST_NAME,
                .RequestKey = TStringBuf("itype=newsd;ctype=prod"),
                .SignalName = SIGNAL_NAME,
                .DontAggregate = aggrPolicy == EAggregationPolicy::DontAggregate,
            };

            return client.ReadOneSignal(request);
        };

        TVector<TValue> expectedValues;
        expectedValues.emplace_back(TValue(0));
        expectedValues.emplace_back(TValue(1));
        expectedValues.emplace_back(TValue(2));
        {
            auto response = fetchMetrics(EAggregationPolicy::DontAggregate);
            UNIT_ASSERT_VALUES_EQUAL(response.Series.GetValues(), expectedValues);
        }

        // when we don't use DontAggregate flag, TSDB will insert as many zeroes as needed to fill the interval, even if there are not as many actual points
        {
            auto response = fetchMetrics(EAggregationPolicy::Aggregate);

            expectedValues.emplace_back(TValue(0));
            UNIT_ASSERT_VALUES_EQUAL(response.Series.GetValues(), expectedValues);
        }
    }

    Y_UNIT_TEST(TestTimeRangeWithAbsentPointsInTheBeginning) {
        TInstant start = TInstant::Seconds(100000);
        TDuration shift = TDuration::Minutes(4);
        size_t valuesCnt = 3;

        auto fetchMetrics = [&](EAggregationPolicy aggrPolicy) {
            TFreshClient client;
            TInstanceKey instanceKey = TInstanceKey::FromNamed("newsd|ctype=prod");
            for (size_t i = 0; i != valuesCnt; ++i) {
                TRecordsBuilder builder;
                builder.Add(instanceKey, SIGNAL_NAME, TValue(i));
                auto ts = start + shift + ITERATION_SIZE * i;
                client.PushSignals(ts, HOST_NAME, GROUP_NAME, builder.Records);
            }

            THistoryRequest request{
                .Start = start,
                .End = start + shift + (valuesCnt - 1) * ITERATION_SIZE,
                .Period = ITERATION_SIZE,
                .HostName = HOST_NAME,
                .RequestKey = TStringBuf("itype=newsd;ctype=prod"),
                .SignalName = SIGNAL_NAME,
                .DontAggregate = aggrPolicy == EAggregationPolicy::DontAggregate,
            };

            return client.ReadOneSignal(request);
        };

        {
            TVector<TValue> expectedValues;
            for (size_t i = 0; i != shift / ITERATION_SIZE; ++i) {
                expectedValues.emplace_back(TValue());
            }
            for (size_t i = 0; i != valuesCnt; ++i) {
                expectedValues.emplace_back(TValue(i));
            }

            auto response = fetchMetrics(EAggregationPolicy::DontAggregate);
            UNIT_ASSERT_VALUES_EQUAL(response.Series.GetValues(), expectedValues);
        }

        // // when we don't use DontAggregate flag, TSDB will insert as many zeroes as needed to fill the interval, even if there are not as many actual points
        {
            TVector<TValue> expectedValues;
            for (size_t i = 0; i != shift / ITERATION_SIZE; ++i) {
                expectedValues.emplace_back(TValue(0));
            }
            for (size_t i = 0; i != valuesCnt; ++i) {
                expectedValues.emplace_back(TValue(i));
            }

            auto response = fetchMetrics(EAggregationPolicy::Aggregate);
            UNIT_ASSERT_VALUES_EQUAL(response.Series.GetValues(), expectedValues);
        }
    }

    Y_UNIT_TEST(TestNotFound) {
        TFreshClient client;
        TInstant start = TInstant::Seconds(100000);
        THistoryRequest request{
            .Start = start,
            .End = start,
            .Period = ITERATION_SIZE,
            .HostName = HOST_NAME,
            .RequestKey = TStringBuf("itype=newsd"),
            .SignalName = SIGNAL_NAME
        };
        auto response(client.ReadOneSignal(request));
        UNIT_ASSERT_VALUES_EQUAL(response.Series.GetStartTimestamp(), start);
        UNIT_ASSERT_VALUES_EQUAL(response.StatusCode, EStatusCode::THistoryAggregatedSeries_EStatusCode_NOT_FOUND);
        UNIT_ASSERT_VALUES_EQUAL(response.Series.GetValues(), TVector<TValue>{});
    }

    Y_UNIT_TEST(TestAggregation) {
        TFreshClient client;

        TInstant start = TInstant::Seconds(100000);
        size_t valuesCount = 3;
        TInstanceKey firstKey = TInstanceKey::FromNamed("newsd|ctype=prod;geo=sas");
        TInstanceKey secondKey = TInstanceKey::FromNamed("newsd|ctype=prod;geo=man");
        TInstanceKey thirdKey = TInstanceKey::FromNamed("newsd|ctype=prestable;geo=vla");
        TVector<TInstanceKey> instanceKeys{firstKey, secondKey, thirdKey};
        for (const auto idx : xrange(valuesCount)) {
            TRecordsBuilder builder;
            for (const auto& instanceKey : instanceKeys) {
                builder.Add(instanceKey, SIGNAL_NAME, TValue(1 + idx));
            }
            client.PushSignals(start + ITERATION_SIZE * idx, HOST_NAME, GROUP_NAME, builder.Records);
        }

        THistoryRequest request{
            .Start = start,
            .End = start + ITERATION_SIZE * valuesCount,
            .Period = ITERATION_SIZE,
            .HostName = HOST_NAME,
            .RequestKey = TStringBuf("itype=newsd;ctype=prod"),
            .SignalName = SIGNAL_NAME
        };
        auto response(client.ReadOneSignal(request));
        UNIT_ASSERT_VALUES_EQUAL(response.Series.GetStartTimestamp(), start);
        UNIT_ASSERT_VALUES_EQUAL(response.StatusCode, EStatusCode::THistoryAggregatedSeries_EStatusCode_OK);

        TVector<TValue> expectedValues;
        expectedValues.emplace_back(TValue(2));
        expectedValues.emplace_back(TValue(4));
        expectedValues.emplace_back(TValue(6));
        expectedValues.emplace_back(TValue(0));
        UNIT_ASSERT_VALUES_EQUAL(response.Series.GetValues(), expectedValues);
    }

    Y_UNIT_TEST(TestAggregationLimit) {
        TFreshClient client;

        const size_t valuesCount = 3;
        const size_t expectedSeriesLimit = 1000; // change when test starts failing because of the change in actual limit
        const size_t instanceKeysCount = expectedSeriesLimit + 10;

        TVector<TInstanceKey> instanceKeys;
        for (const auto instanceKeyIdx: xrange(instanceKeysCount)) {
            TStringStream namedInstanceKey;
            namedInstanceKey << "newsd|ctype=prod;geo=sas" << instanceKeyIdx;
            instanceKeys.push_back(TInstanceKey::FromNamed(namedInstanceKey.Str()));
        }

        TInstant start = TInstant::Seconds(100000);
        for (const auto idx : xrange(valuesCount)) {
            TRecordsBuilder builder;
            for (const auto& instanceKey : instanceKeys) {
                builder.Add(instanceKey, SIGNAL_NAME, TValue(1 + idx));
            }
            client.PushSignals(start + ITERATION_SIZE * idx, HOST_NAME, GROUP_NAME, builder.Records);
        }

        THistoryRequest request{
            .Start = start,
            .End = start + ITERATION_SIZE * valuesCount,
            .Period = ITERATION_SIZE,
            .HostName = HOST_NAME,
            .RequestKey = TStringBuf("itype=newsd;ctype=prod"),
            .SignalName = SIGNAL_NAME
        };
        auto response(client.ReadOneSignal(request));
        UNIT_ASSERT_VALUES_EQUAL(response.Series.GetStartTimestamp(), start);
        UNIT_ASSERT_VALUES_EQUAL(response.StatusCode, EStatusCode::THistoryAggregatedSeries_EStatusCode_LIMIT_EXCEEDED);

        TVector<TValue> expectedValues;
        expectedValues.emplace_back(TValue(1 * expectedSeriesLimit));
        expectedValues.emplace_back(TValue(2 * expectedSeriesLimit));
        expectedValues.emplace_back(TValue(3 * expectedSeriesLimit));
        expectedValues.emplace_back(TValue(0));
        UNIT_ASSERT_VALUES_EQUAL(response.Series.GetValues(), expectedValues);
    }

    Y_UNIT_TEST(TestRollup) {
        TFreshClient client;

        TInstant start = TInstant::Seconds(960);
        // 3 minutes
        size_t valuesCount = 36;
        TInstanceKey firstKey = TInstanceKey::FromNamed("newsd|ctype=prod;geo=sas");
        TInstanceKey secondKey = TInstanceKey::FromNamed("newsd|ctype=prod;geo=man");
        for (const auto idx : xrange(valuesCount)) {
            TRecordsBuilder builder;
            builder.Add(firstKey, SIGNAL_NAME, TValue(1));
            builder.Add(secondKey, SIGNAL_NAME, TValue(2));
            client.PushSignals(start + ITERATION_SIZE * (idx + 1), HOST_NAME, GROUP_NAME, builder.Records);
        }

        THistoryRequest request{
            .Start = start,
            .End = start + TDuration::Minutes(3),
            .Period = TDuration::Minutes(1),
            .HostName = HOST_NAME,
            .RequestKey = TStringBuf("itype=newsd;ctype=prod"),
            .SignalName = SIGNAL_NAME
        };
        auto response(client.ReadOneSignal(request));
        UNIT_ASSERT_VALUES_EQUAL(response.Series.GetStartTimestamp(), start);
        UNIT_ASSERT_VALUES_EQUAL(response.StatusCode, EStatusCode::THistoryAggregatedSeries_EStatusCode_OK);

        TVector<TValue> expectedValues;
        expectedValues.emplace_back(TValue(33));
        expectedValues.emplace_back(TValue(36));
        expectedValues.emplace_back(TValue(36));
        expectedValues.emplace_back(TValue(3));
        UNIT_ASSERT_VALUES_EQUAL(response.Series.GetValues(), expectedValues);
    }

    Y_UNIT_TEST(TestRollupWithoutAggregation) {
        TFreshClient client;
        auto period = TDuration::Seconds(30);

        TInstant start = TInstant::Seconds(30);
        size_t valuesCount = period / ITERATION_SIZE;
        TInstanceKey firstKey = TInstanceKey::FromNamed("newsd|ctype=prod;geo=sas");
        for (const auto idx : xrange(valuesCount)) {
            TRecordsBuilder builder;
            builder.Add(firstKey, SIGNAL_NAME, TValue(1));
            client.PushSignals(start + ITERATION_SIZE * idx, HOST_NAME, GROUP_NAME, builder.Records);
        }

        auto fetchMetrics = [&](EAggregationPolicy aggrPolicy) {
            THistoryRequest request{
                .Start = start,
                .End = start + period + TDuration::Seconds(5),
                .Period = period,
                .HostName = HOST_NAME,
                .RequestKey = TStringBuf("itype=newsd;ctype=prod;geo=sas"),
                .SignalName = SIGNAL_NAME,
                .DontAggregate = aggrPolicy == EAggregationPolicy::DontAggregate,
            };

            auto response = client.ReadOneSignal(request);
            UNIT_ASSERT_VALUES_EQUAL(response.Series.GetStartTimestamp(), start);
            UNIT_ASSERT_VALUES_EQUAL(response.StatusCode, EStatusCode::THistoryAggregatedSeries_EStatusCode_OK);
            return response;
        };

        TVector<TValue> expectedValues;
        expectedValues.emplace_back(TValue(6));

        {
            auto response = fetchMetrics(EAggregationPolicy::DontAggregate);
            UNIT_ASSERT_VALUES_EQUAL(response.Series.GetValues(), expectedValues);
        }

        {
            auto response = fetchMetrics(EAggregationPolicy::Aggregate);
            expectedValues.emplace_back(TValue(0));
            expectedValues.emplace_back(TValue(0));
            UNIT_ASSERT_VALUES_EQUAL(response.Series.GetValues(), expectedValues);
        }
    }

    Y_UNIT_TEST(TestRollupWithoutAggregationAndWithHoles) {
        TFreshClient client;
        auto period = TDuration::Seconds(30);

        TInstant start = TInstant::Seconds(30);
        size_t periodsCnt = 6;
        TInstanceKey firstKey = TInstanceKey::FromNamed("newsd|ctype=prod;geo=sas");
        for (const auto periodIdx : xrange(periodsCnt)) {
            if (periodIdx % 2 == 0) {
                continue;
            }

            for (size_t i = 0; i != 3; ++i) {
                TRecordsBuilder builder;
                builder.Add(firstKey, SIGNAL_NAME, TValue(1));
                auto ts = start + period * periodIdx + ITERATION_SIZE * i;
                client.PushSignals(ts, HOST_NAME, GROUP_NAME, builder.Records);
            }
        }

        auto fetchMetrics = [&](EAggregationPolicy aggrPolicy) {
            THistoryRequest request{
                .Start = start,
                .End = start + period * periodsCnt,
                .Period = period,
                .HostName = HOST_NAME,
                .RequestKey = TStringBuf("itype=newsd;ctype=prod;geo=sas"),
                .SignalName = SIGNAL_NAME,
                .DontAggregate = aggrPolicy == EAggregationPolicy::DontAggregate,
            };

            auto response = client.ReadOneSignal(request);
            UNIT_ASSERT_VALUES_EQUAL(response.Series.GetStartTimestamp(), start);
            UNIT_ASSERT_VALUES_EQUAL(response.StatusCode, EStatusCode::THistoryAggregatedSeries_EStatusCode_OK);
            return response;
        };

        {
            TVector<TValue> expectedValues;
            expectedValues.emplace_back(TValue());
            expectedValues.emplace_back(TValue(3));
            expectedValues.emplace_back(TValue());
            expectedValues.emplace_back(TValue(3));
            expectedValues.emplace_back(TValue());
            expectedValues.emplace_back(TValue(3));

            auto response = fetchMetrics(EAggregationPolicy::DontAggregate);
            UNIT_ASSERT_VALUES_EQUAL(response.Series.GetValues(), expectedValues);
        }

        {
            TVector<TValue> expectedValues;
            expectedValues.emplace_back(TValue(0));
            expectedValues.emplace_back(TValue(3));
            expectedValues.emplace_back(TValue(0));
            expectedValues.emplace_back(TValue(3));
            expectedValues.emplace_back(TValue(0));
            expectedValues.emplace_back(TValue(3));
            expectedValues.emplace_back(TValue(0));

            auto response = fetchMetrics(EAggregationPolicy::Aggregate);
            UNIT_ASSERT_VALUES_EQUAL(response.Series.GetValues(), expectedValues);
        }
    }

    Y_UNIT_TEST(TestInvalidPeriod) {
        TFreshClient client;
        THistoryRequest request{
            .Start = TInstant::Seconds(500),
            .End = TInstant::Seconds(1000),
            .Period = TDuration::Seconds(1),
            .HostName = HOST_NAME,
            .RequestKey = TStringBuf("itype=newsd"),
            .SignalName = SIGNAL_NAME
        };
        auto response(client.ReadOneSignal(request));
        UNIT_ASSERT_VALUES_EQUAL(response.StatusCode, EStatusCode::THistoryAggregatedSeries_EStatusCode_INTERNAL_ERROR);
    }

    Y_UNIT_TEST(TestInvalidSignal) {
        TFreshClient client;
        THistoryRequest request{
            .Start = TInstant::Seconds(500),
            .End = TInstant::Seconds(1000),
            .Period = TDuration::Seconds(5),
            .HostName = HOST_NAME,
            .RequestKey = TStringBuf("itype=newsd;ctype=prod"),
            .SignalName = TStringBuf("wrong")
        };
        auto response(client.ReadOneSignal(request));
        UNIT_ASSERT_VALUES_EQUAL(response.StatusCode, EStatusCode::THistoryAggregatedSeries_EStatusCode_NOT_FOUND);
    }
}
