#include "server_monitoring.h"

#include <infra/yasm/common/const.h>

#include <library/cpp/monlib/metrics/metric_registry.h>

#include <util/datetime/uptime.h>
#include <util/stream/file.h>
#include <util/string/cast.h>
#include <util/string/split.h>

namespace NYasmServer {

// This code is copied from library/cpp/actors/core/process_stats.h
struct TProcStat {
    ui64 Rss;
    ui64 VolCtxSwtch;
    ui64 NonvolCtxSwtch;

    int Pid;
    char State;
    int Ppid;
    int Pgrp;
    int Session;
    int TtyNr;
    int TPgid;
    unsigned Flags;
    unsigned long MinFlt;
    unsigned long CMinFlt;
    unsigned long MajFlt;
    unsigned long CMajFlt;
    unsigned long Utime;
    unsigned long Stime;
    long CUtime;
    long CStime;
    long Priority;
    long Nice;
    long NumThreads;
    long ItRealValue;
    // StartTime is measured from system boot
    unsigned long long StartTime;
    unsigned long Vsize;
    long RssPages;
    unsigned long RssLim;
    ui64 FileRss;
    ui64 AnonRss;
    ui64 CGroupMemLim = 0;

    TDuration Uptime;
    TDuration SystemUptime;
    // ...

    TProcStat() {
        Zero(*this);
        Y_UNUSED(PageSize);
    }

    bool Fill(pid_t pid);

private:
    long PageSize = 0;

    long ObtainPageSize();
};

#ifdef _linux_

namespace {
        template <typename TVal>
        static bool ExtractVal(const TString& str, const TString& name, TVal& res) {
            if (!str.StartsWith(name))
                return false;
            size_t pos = name.size();
            while (pos < str.size() && (str[pos] == ' ' || str[pos] == '\t')) {
                pos++;
            }
            res = atol(str.data() + pos);
            return true;
        }

        float TicksPerMillisec() {
#ifdef _SC_CLK_TCK
            return sysconf(_SC_CLK_TCK) / 1000.0;
#else
            return 1.f;
#endif
        }
    }

    bool TProcStat::Fill(pid_t pid) {
        try {
            TString strPid(ToString(pid));
            TFileInput proc("/proc/" + strPid + "/status");
            TString str;
            while (proc.ReadLine(str)) {
                if (ExtractVal(str, "VmRSS:", Rss))
                    continue;
                if (ExtractVal(str, "voluntary_ctxt_switches:", VolCtxSwtch))
                    continue;
                if (ExtractVal(str, "nonvoluntary_ctxt_switches:", NonvolCtxSwtch))
                    continue;
            }
            // Convert from kB to bytes
            Rss *= 1024;

            float tickPerMillisec = TicksPerMillisec();

            TFileInput procStat("/proc/" + strPid + "/stat");
            procStat.ReadLine(str);
            if (!str.empty()) {
                sscanf(str.data(),
                       "%d %*s %c %d %d %d %d %d %u %lu %lu "
                       "%lu %lu %lu %lu %ld %ld %ld %ld %ld "
                       "%ld %llu %lu %ld %lu",
                       &Pid, &State, &Ppid, &Pgrp, &Session, &TtyNr, &TPgid, &Flags, &MinFlt, &CMinFlt,
                       &MajFlt, &CMajFlt, &Utime, &Stime, &CUtime, &CStime, &Priority, &Nice, &NumThreads,
                       &ItRealValue, &StartTime, &Vsize, &RssPages, &RssLim);
                Utime /= tickPerMillisec;
                Stime /= tickPerMillisec;
                CUtime /= tickPerMillisec;
                CStime /= tickPerMillisec;
                SystemUptime = ::Uptime();
                Uptime = SystemUptime - TDuration::MilliSeconds(StartTime / TicksPerMillisec());
            }

            TFileInput statm("/proc/" + strPid + "/statm");
            statm.ReadLine(str);
            TVector<TString> fields;
            StringSplitter(str).Split(' ').SkipEmpty().Collect(&fields);
            if (fields.size() >= 7) {
                ui64 resident = FromString<ui64>(fields[1]);
                ui64 shared = FromString<ui64>(fields[2]);
                if (PageSize == 0) {
                    PageSize = ObtainPageSize();
                }
                FileRss = shared * PageSize;
                AnonRss = (resident - shared) * PageSize;
            }

            TFileInput cgroup("/proc/" + strPid + "/cgroup");
            TString line;
            TString memoryCGroup;
            while (cgroup.ReadLine(line) > 0) {
                StringSplitter(line).Split(':').Collect(&fields);
                if (fields.size() > 2 && fields[1] == "memory") {
                    memoryCGroup = fields[2];
                    break;
                }
            }
            if (!memoryCGroup.empty()) {
                TFileInput limit("/sys/fs/cgroup/memory" + memoryCGroup + "/memory.limit_in_bytes");
                if (limit.ReadLine(line) > 0) {
                    CGroupMemLim = FromString<ui64>(line);
                    if (CGroupMemLim > (1ULL << 40)) {
                        CGroupMemLim = 0;
                    }
                }
            }

        } catch (...) {
            return false;
        }
        return true;
    }

    long TProcStat::ObtainPageSize() {
        long sz = sysconf(_SC_PAGESIZE);
        return sz;
    }

#else

bool TProcStat::Fill(pid_t pid) {
    Y_UNUSED(pid);
    return false;
}

long TProcStat::ObtainPageSize() {
    return 0;
}

#endif

// ---------------------------------------------------------------------------------------------------------------------

TServerMonitoring::TServerMonitoring(TLog& logger)
    : Logger_(logger)
{
}

void TServerMonitoring::Start() {
    Logger_ << "Start server monitoring";
    MonitoringJob_ = NMonitoring::StartPeriodicJob(
            [this] {
                ReportProcessMetrics();
            },
            NMonitoring::TPeriodCalculator(NYasm::NCommon::ITERATION_SIZE));
}

void TServerMonitoring::ReportProcessMetrics() {
    TProcStat procStat;
    procStat.Fill(getpid());

    auto* registry = NMonitoring::TMetricRegistry::Instance();
    if (registry) {
        registry->IntGauge({{"sensor", "process.VmSize"}})->Set(procStat.Vsize);
        registry->IntGauge({{"sensor", "process.AnonRssSize"}})->Set(procStat.AnonRss);
        registry->IntGauge({{"sensor", "process.FileRssSize"}})->Set(procStat.FileRss);
        registry->IntGauge({{"sensor", "process.CGroupMemLimit"}})->Set(procStat.CGroupMemLim);
        registry->IntGauge({{"sensor", "process.UptimeSeconds"}})->Set(procStat.Uptime.Seconds());
        registry->IntGauge({{"sensor", "process.NumThreads"}})->Set(procStat.NumThreads);
        registry->IntGauge({{"sensor", "system.UptimeSeconds"}})->Set(procStat.SystemUptime.Seconds());

        NMonitoring::TRate* userTime = registry->Rate({{"sensor", "process.UserTime"}});
        NMonitoring::TRate* sysTime = registry->Rate({{"sensor", "process.SystemTime"}});
        NMonitoring::TRate* minorPageFaults = registry->Rate({{"sensor", "process.MinorPageFaults"}});
        NMonitoring::TRate* majorPageFaults = registry->Rate({{"sensor", "process.MajorPageFaults"}});

        userTime->Reset();
        userTime->Add(procStat.Utime);

        sysTime->Reset();
        sysTime->Add(procStat.Stime);

        minorPageFaults->Reset();
        minorPageFaults->Add(procStat.MinFlt);

        majorPageFaults->Reset();
        majorPageFaults->Add(procStat.MajFlt);
    }
}

void TServerMonitoring::ShutDown() {
    Logger_ << "Shut down server monitoring";
    MonitoringJob_.Reset();
}

} // namespace NYasmServer
