#include "counted_sum_list.h"

#include <library/cpp/testing/unittest/gtest.h>
#include <library/cpp/testing/unittest/registar.h>

namespace NYasmServer {
    using namespace NYasm::NCommon;

    Y_UNIT_TEST_SUITE(TestCountedSumList) {
        Y_UNIT_TEST(TestConstantlyIncrementingValues) {
            TCountedSumList storage;

            auto start = TInstant::Seconds(100000);
            for (size_t i = 0; i < 1000; i++) {
                UNIT_ASSERT(storage.PushValue(start + ITERATION_SIZE * i, TCountedSum(i, (double)i*2)));
            }

            auto value = storage.GetValueAt(start); // too old
            UNIT_ASSERT(!value.Defined());

            for (size_t i = 700; i < 1000; i++) {
                auto value = storage.GetValueAt(start + ITERATION_SIZE * i);
                UNIT_ASSERT(value.Defined());
                UNIT_ASSERT_EQUAL(value->GetCount(), i);
                UNIT_ASSERT_EQUAL(value->GetSum(), i * 2);
            }
        }

        Y_UNIT_TEST(TestSameValues) {
            TCountedSumList storage;

            auto start = TInstant::Seconds(100000);
            for (size_t i = 0; i < 300; i++) {
                UNIT_ASSERT(storage.PushValue(start + ITERATION_SIZE * i, TCountedSum(15, 123.5)));
            }

            for (size_t i = 0; i < 300; i++) {
                auto value = storage.GetValueAt(start + ITERATION_SIZE * i);
                UNIT_ASSERT(value.Defined());
                UNIT_ASSERT_EQUAL_C(value->GetCount(), 15, "Difference at position " << i);
                UNIT_ASSERT_DOUBLES_EQUAL_C(value->GetSum(), 123.5, 0.001, "Difference at position " << i);
            }
        }

        Y_UNIT_TEST(TestNullEverySecondValue) {
            TCountedSumList storage;

            auto start = TInstant::Seconds(100000);
            for (size_t i = 0; i < 300; i += 2) {
                UNIT_ASSERT(storage.PushValue(start + ITERATION_SIZE * i, TCountedSum(15, 123.5)));
            }

            for (size_t i = 0; i < 300; i += 2) {
                auto value = storage.GetValueAt(start + ITERATION_SIZE * i);
                if (i % 2 == 0) {
                    UNIT_ASSERT(value.Defined());
                    UNIT_ASSERT_EQUAL(value->GetCount(), 15);
                    UNIT_ASSERT_DOUBLES_EQUAL(value->GetSum(), 123.5, 0.001);
                } else {
                    UNIT_ASSERT(!value.Defined());
                }
            }
        }

        Y_UNIT_TEST(TestALotOfNulls) {
            TCountedSumList storage;

            auto start = TInstant::Seconds(100000);
            UNIT_ASSERT(storage.PushValue(start + ITERATION_SIZE * 0, TCountedSum(15, 123.5)));
            UNIT_ASSERT(storage.PushValue(start + ITERATION_SIZE * 299, TCountedSum(15, 123.5)));

            auto value = storage.GetValueAt(start + ITERATION_SIZE * 0);
            UNIT_ASSERT(value.Defined());
            UNIT_ASSERT_DOUBLES_EQUAL(value->GetSum(), 123.5, 0.001);

            for (size_t i = 1; i < 299; i += 2) {
                value = storage.GetValueAt(start + ITERATION_SIZE * i);
                UNIT_ASSERT(!value.Defined());
            }

            value = storage.GetValueAt(start + ITERATION_SIZE * 299);
            UNIT_ASSERT(value.Defined());
            UNIT_ASSERT_DOUBLES_EQUAL(value->GetSum(), 123.5, 0.001);
        }

        Y_UNIT_TEST(TestRandomStreamWithLargeVariations) {
            std::array<double, 50> sums = {{-3071.05322, -2017.56442, 4714.55233, 1979.39916, 2899.65185, -1810.0502, -3639.74775, 3116.27517, -629.97372, 3364.05679,
                                            4539.19949, -4466.16969, 262.24611, -4450.96286, -1799.42882, 3856.91784, -3631.3559, -824.73669, -3252.01158, 4062.58133,
                                            613.2597, 4017.07825, 592.67932, 2663.23199, 2346.06195, -4769.22133, -2670.88256, 3881.45846, -2791.53765, -1797.30928,
                                            -2652.76118, -1210.51468, -3350.76019, -4424.36761, 2026.83798, 3652.22186, 2841.87602, -934.01873, -3519.35956, -2884.97552,
                                            3545.77857, -3716.00701, -2983.12844, -505.70156, -1232.85379, -4285.72937, 3224.68962, 4128.58792, 3342.07756, 2274.16516}};

            std::array<size_t, 50> counts = {{43366046333, 47036037694, 106006838143, 73389989192, 1597292386, 92085800808, 47903458341, 22608176530, 117761663523, 116613211805,
                                              44210846034, 23995472148, 82143544596, 65313818653, 6760136998, 23881603365, 94504258730, 67631603782, 86852291704, 83962421677,
                                              8096276771, 37532652106, 77598687449, 39973573899, 35159313706, 17773254368, 19846169472, 113225025736, 74818170017, 80415007507,
                                              120740593485, 74190576469, 27063745823, 92853495737, 118321661340, 111961676402, 40242142301, 36846150643, 28665544572, 75804859206,
                                              30588706201, 42239812060, 3193218631, 94637635246, 77412319792, 10813213384, 110514231737, 38661428552, 35304303490, 64493908683}};

            TCountedSumList storage;

            auto start = TInstant::Seconds(100000);
            for (size_t i = 0; i < sums.size(); i++) {
                UNIT_ASSERT(storage.PushValue(start + ITERATION_SIZE * i, TCountedSum(counts[i], sums[i])));
            }

            for (size_t i = 0; i < sums.size(); i++) {
                auto value = storage.GetValueAt(start + ITERATION_SIZE * i);
                UNIT_ASSERT(value.Defined());
                UNIT_ASSERT_EQUAL(value->GetCount(), counts[i]);
                UNIT_ASSERT_DOUBLES_EQUAL(value->GetSum(), sums[i], 0.001);
            }
        }
    }
} // namespace NYasmServer
