#include "float_list.h"

#include <library/cpp/testing/unittest/gtest.h>
#include <library/cpp/testing/unittest/registar.h>

#include <util/generic/xrange.h>

namespace NYasmServer {

    Y_UNIT_TEST_SUITE(TestFloatList) {
        inline static void AssertDefinedDouble(TMaybe<double> value, double expected) {
            UNIT_ASSERT(value.Defined());
            UNIT_ASSERT_DOUBLES_EQUAL(value.GetRef(), expected, 0.001);
        }

        inline static void AssertUndefinedDouble(TMaybe<double> value) {
            UNIT_ASSERT_C(!value.Defined(), value.GetRef());
        }

        Y_UNIT_TEST(TestSingleValueVeryCloseToZeroTimestamp) {
            // writing nulls might not work correctly for smaller timstamps
            TFloatList storage;
            UNIT_ASSERT(storage.PushValue(TInstant::Seconds(50), 15));

            AssertDefinedDouble(storage.GetValueAt(TInstant::Seconds(50)), 15);
        }

        Y_UNIT_TEST(TestIteratingOverMultipleChunks) {
            TFloatList storage;
            auto start = TInstant::Seconds(30000);
            const auto interval = TDuration::Seconds(35);
            for (auto idx : xrange(40)) {
                UNIT_ASSERT(storage.PushValue(start + interval * idx, idx));
            }

            struct Visitor : public ISeriesVisitor {
                void OnHeader(TInstant timestamp, size_t count) override {
                    UNIT_ASSERT_EQUAL(timestamp, Timestamp);
                    // one value every 35 seconds 39 times, plus the first value
                    UNIT_ASSERT_EQUAL(count, 35 / 5 * 39 + 1);
                }
                void OnValue(NZoom::NValue::TValueRef value) override {
                    const auto valuesInterval = 7;
                    if (Count % valuesInterval == 0) {
                        UNIT_ASSERT_EQUAL_C(value, NZoom::NValue::TValue(Count / valuesInterval).GetValue(), Count);
                    } else {
                        UNIT_ASSERT_EQUAL_C(value, NZoom::NValue::TValue().GetValue(), Count);
                    }
                    Count++;
                }

                TInstant Timestamp;
                size_t Count = 0;
            } visitor;
            visitor.Timestamp = start;

            storage.IterValues(TInstant::Zero(), start + interval * 40, visitor);
        }

        Y_UNIT_TEST(TestConstantlyIncrementingValues) {
            TFloatList storage;

            auto start = TInstant::Seconds(100000);
            for (size_t i = 0; i < 1000; i++) {
                UNIT_ASSERT(storage.PushValue(start + ITERATION_SIZE * i, (double)i));
            }

            auto value = storage.GetValueAt(start); // too old
            UNIT_ASSERT(!value.Defined());

            for (size_t i = 700; i < 1000; i++) {
                auto value = storage.GetValueAt(start + ITERATION_SIZE * i);
                UNIT_ASSERT(value.Defined());
                UNIT_ASSERT_EQUAL(value.GetRef(), i);
            }
        }

        Y_UNIT_TEST(TestSameValues) {
            TFloatList storage;

            auto start = TInstant::Seconds(100000);
            for (size_t i = 0; i < 300; i++) {
                UNIT_ASSERT(storage.PushValue(start + ITERATION_SIZE * i, 123.5));
            }

            for (size_t i = 0; i < 300; i++) {
                auto value = storage.GetValueAt(start + ITERATION_SIZE * i);
                UNIT_ASSERT(value.Defined());
                UNIT_ASSERT_DOUBLES_EQUAL_C(value.GetRef(), 123.5, 0.001, "Difference at position " << i);
            }
        }

        Y_UNIT_TEST(TestNullEverySecondValue) {
            TFloatList storage;

            auto start = TInstant::Seconds(100000);
            for (size_t i = 0; i < 300; i += 2) {
                UNIT_ASSERT(storage.PushValue(start + ITERATION_SIZE * i, 123.5));
            }

            for (size_t i = 0; i < 300; i += 2) {
                auto value = storage.GetValueAt(start + ITERATION_SIZE * i);
                if (i % 2 == 0) {
                    UNIT_ASSERT(value.Defined());
                    UNIT_ASSERT_DOUBLES_EQUAL(value.GetRef(), 123.5, 0.001);
                } else {
                    UNIT_ASSERT(!value.Defined());
                }
            }
        }

        Y_UNIT_TEST(TestALotOfNulls) {
            TFloatList storage;

            auto start = TInstant::Seconds(100000);
            UNIT_ASSERT(storage.PushValue(start + ITERATION_SIZE * 0, 123.5));
            UNIT_ASSERT(storage.PushValue(start + ITERATION_SIZE * 299, 123.5));

            auto value = storage.GetValueAt(start + ITERATION_SIZE * 0);
            UNIT_ASSERT(value.Defined());
            UNIT_ASSERT_DOUBLES_EQUAL(value.GetRef(), 123.5, 0.001);

            for (size_t i = 1; i < 299; i += 2) {
                value = storage.GetValueAt(start + ITERATION_SIZE * i);
                UNIT_ASSERT(!value.Defined());
            }

            value = storage.GetValueAt(start + ITERATION_SIZE * 299);
            UNIT_ASSERT(value.Defined());
            UNIT_ASSERT_DOUBLES_EQUAL(value.GetRef(), 123.5, 0.001);
        }

        Y_UNIT_TEST(TestRandomStreamWithLargeVariations) {
            std::array<double, 50> values = {{-3071.05322, -2017.56442, 4714.55233, 1979.39916, 2899.65185, -1810.0502, -3639.74775, 3116.27517, -629.97372, 3364.05679,
                                              4539.19949, -4466.16969, 262.24611, -4450.96286, -1799.42882, 3856.91784, -3631.3559, -824.73669, -3252.01158, 4062.58133,
                                              613.2597, 4017.07825, 592.67932, 2663.23199, 2346.06195, -4769.22133, -2670.88256, 3881.45846, -2791.53765, -1797.30928,
                                              -2652.76118, -1210.51468, -3350.76019, -4424.36761, 2026.83798, 3652.22186, 2841.87602, -934.01873, -3519.35956, -2884.97552,
                                              3545.77857, -3716.00701, -2983.12844, -505.70156, -1232.85379, -4285.72937, 3224.68962, 4128.58792, 3342.07756, 2274.16516}};

            TFloatList storage;

            auto start = TInstant::Seconds(100000);
            for (size_t i = 0; i < values.size(); i++) {
                UNIT_ASSERT(storage.PushValue(start + ITERATION_SIZE * i, values[i]));
            }

            for (size_t i = 0; i < values.size(); i++) {
                auto value = storage.GetValueAt(start + ITERATION_SIZE * i);
                UNIT_ASSERT(value.Defined());
                UNIT_ASSERT_DOUBLES_EQUAL(value.GetRef(), values[i], 0.001);
            }
        }

        Y_UNIT_TEST(TestSimplePushAndGet) {
            auto now = TInstant::Seconds(100000);
            TFloatList series;

            AssertUndefinedDouble(series.GetValueAt(now));

            series.PushValue(now, 12345);
            AssertDefinedDouble(series.GetValueAt(now), 12345);
        }

        Y_UNIT_TEST(TestMissingSample) {
            auto now = TInstant::Seconds(100000);
            TFloatList series;

            series.PushValue(now, 1);
            // no value in the next 10 seconds
            series.PushValue(now + TDuration::Seconds(15), 4);

            AssertDefinedDouble(series.GetValueAt(now), 1);
            AssertUndefinedDouble(series.GetValueAt(now + TDuration::Seconds(5)));
            AssertUndefinedDouble(series.GetValueAt(now + TDuration::Seconds(10)));
            AssertDefinedDouble(series.GetValueAt(now + TDuration::Seconds(15)), 4);
            AssertUndefinedDouble(series.GetValueAt(now + TDuration::Seconds(20)));
        }

        Y_UNIT_TEST(TestExtremeGetRequests) {
            auto now = TInstant::Seconds(100000);
            TFloatList series;

            series.PushValue(now, 1);

            AssertUndefinedDouble(series.GetValueAt(now - TDuration::Hours(5)));
            AssertUndefinedDouble(series.GetValueAt(now + TDuration::Hours(5)));
        }

        Y_UNIT_TEST(TestWrapping) {
            auto now = TInstant::Seconds(100000 - 5);
            TFloatList series;
            for (size_t idx = 0; idx < 1000; idx++) {
                now += ITERATION_SIZE;
                series.PushValue(now, idx);
            }

            AssertDefinedDouble(series.GetValueAt(now - ITERATION_SIZE * 5), 999 - 5);
            AssertDefinedDouble(series.GetValueAt(now - ITERATION_SIZE * 200), 999 - 200);
            // whoops, overwritten
            AssertUndefinedDouble(series.GetValueAt(now - ITERATION_SIZE * 500));
        }

        Y_UNIT_TEST(TestValuesAfterDifferentIntervals) {
            auto start = TInstant::Seconds(100000);
            TVector<TInstant> timestamps;
            for (size_t interval : xrange(1, 600)) {
                timestamps.clear();

                TFloatList series;
                auto now = start;
                for (size_t i = 0; i < 20; i++) {
                    now += ITERATION_SIZE * interval;
                    timestamps.push_back(now);
                    series.PushValue(now, 2.0);
                }

                for (auto ts : timestamps) {
                    if (ts >= series.GetStartTime() && ts <= series.GetEndTime()) {
                        AssertDefinedDouble(series.GetValueAt(ts), 2.0);
                    } else {
                        AssertUndefinedDouble(series.GetValueAt(ts));
                    }
                }
            }
        }
    }
} // namespace NYasmServer
