#include "fresh.h"

#include "counted_sum_list.h"
#include "float_list.h"
#include "histogram_list.h"
#include "counters.h"

#include <infra/yasm/common/interval.h>
#include <infra/yasm/server/lib/metrics.h>
#include <infra/yasm/server/lib/zoom_writers.h>
#include <infra/yasm/server/persistence/reader.h>

#include <infra/monitoring/common/collections.h>
#include <infra/monitoring/common/perf.h>

#include <util/random/random.h>

namespace {
    // defined interval in seconds, when we must start clean up
    // GOLOVAN-6367
    const TDuration MIN_START_OFFSET = TDuration::Seconds(20);
    const TDuration MAX_START_OFFSET = NYasmServer::CHUNK_SIZE + MIN_START_OFFSET;
}

namespace NYasmServer {
    using namespace NYasm::NCommon::NInterval;

    using NPersistence::TDataChunk;
    using NTags::TInstanceKey;
    using NZoom::NSignal::TSignalName;
    using NZoom::NHost::THostName;

    template <bool CanCreate>
    class TFreshStorage::TSeriesWriter {
    public:
        TSeriesWriter(TSignalMap& signals,
                      TVector<TSingleValueRefWithKey>& toCreate)
            : Signals(signals)
            , ToCreate(toCreate)
        {
        }

        void Add(const TSingleValueRefWithKey& value, TInstant timestamp, TLog& logger, AcceptanceStat& acceptanceStat) {
            IRecordList* series = FindSeries(value);
            if (series) {
                TZoomValueWriter writer(series, timestamp);
                try {
                    value.GetValue().UpdateFlat(writer);
                } catch (...) {
                    logger << "Failed write value on tag '" << value.GetInstanceKey()
                           << "', signal '" << value.GetSignalName()
                           << "', host '" << value.GetHostName()
                           << "': " << CurrentExceptionMessage();
                }
                if (writer.WasAccepted()) {
                    TZoomValueWriter::AcceptanceType acceptanceType = writer.GetAcceptanceType();
                    if (acceptanceType == TZoomValueWriter::AcceptanceType::ACCEPTED_SMALL) {
                        if (CurrentHost->first.IsGroup()) {
                            ++acceptanceStat.acceptedGroupSmallHgram;
                        } else {
                            ++acceptanceStat.acceptedHostSmallHgram;
                        }
                    }
                    if (acceptanceType == TZoomValueWriter::AcceptanceType::ACCEPTED_NORMAL) {
                        if (CurrentHost->first.IsGroup()) {
                            ++acceptanceStat.acceptedGroupNormalHgram;
                        } else {
                            ++acceptanceStat.acceptedHostNormalHgram;
                        }
                    }
                    ++acceptanceStat.accepted;
                }
            } else if (!CanCreate) {
                ToCreate.emplace_back(value);
            }
        }

    private:
        THostMap* FindHostIndex(const TSingleValueRefWithKey& value) {
            if (CurrentSignal.Defined() && CurrentSignal->first == value.GetSignalName()) {
                return CurrentSignal->second;
            } else if (CurrentSignal.Defined()) {
                CurrentSignal.Clear();
                CurrentHost.Clear();
            }

            auto it = Signals.find(value.GetSignalName());
            if (it == Signals.end()) {
                if (CanCreate) {
                    it = Signals.emplace_hint(it, value.GetSignalName(), MakeHolder<THostMap>());
                } else {
                    return nullptr;
                }
            }

            Y_ASSERT(CurrentSignal.Empty());
            CurrentSignal.ConstructInPlace(it->first, it->second.Get());
            return it->second.Get();

        }

        TTagMap* FindTagIndex(const TSingleValueRefWithKey& value) {
            THostMap* hostMap = FindHostIndex(value);
            if (!hostMap) {
                return nullptr;
            }

            if (CurrentHost.Defined() && CurrentHost->first == value.GetHostName()) {
                return CurrentHost->second;
            } else if (CurrentHost.Defined()) {
                CurrentHost.Clear();
            }

            auto it = hostMap->find(value.GetHostName());
            if (it == hostMap->end()) {
                if (CanCreate) {
                    it = hostMap->emplace_hint(it, value.GetHostName(), MakeHolder<TTagMap>());
                } else {
                    return nullptr;
                }
            }

            Y_ASSERT(CurrentHost.Empty());
            CurrentHost.ConstructInPlace(it->first, it->second.Get());
            return it->second.Get();
        }

        TSeriesPtr CreateSeries(const TSingleValueRefWithKey& value) {
            using NZoom::NValue::EValueType;
            switch (value.GetValue().GetType()) {
                case EValueType::NONE: {
                    return nullptr;
                }
                case EValueType::FLOAT: {
                    return MakeAtomicShared<TFloatList>();
                }
                case EValueType::COUNTED_SUM: {
                    return MakeAtomicShared<TCountedSumList>();
                }
                case EValueType::VEC:
                case EValueType::SMALL_HGRAM:
                case EValueType::NORMAL_HGRAM:
                case EValueType::USER_HGRAM: {
                    return MakeAtomicShared<THistogramList>();
                }
                case EValueType::HYPER_LOGLOG: {
                    Y_FAIL("not implemented");
                }
            }
        }

        IRecordList* FindSeries(const TSingleValueRefWithKey& value) {
            TTagMap* tagIndex = FindTagIndex(value);
            if (!tagIndex) {
                return nullptr;
            }

            auto it = tagIndex->find(value.GetInstanceKey());
            if (it == tagIndex->end()) {
                if (CanCreate) {
                    auto series = CreateSeries(value);
                    if (series) {
                        it = tagIndex->emplace_hint(it, value.GetInstanceKey(), std::move(series));
                    } else {
                        return nullptr;
                    }
                } else {
                    return nullptr;
                }
            }

            return it->second.Get();
        }

        TSignalMap& Signals;
        TVector<TSingleValueRefWithKey>& ToCreate;

        TMaybe<std::pair<TSignalName, THostMap*>> CurrentSignal;
        TMaybe<std::pair<THostName, TTagMap*>> CurrentHost;
    };

    TFreshStorage::TFreshStorage(TLog& logger, NPersistence::ISnapshotManager& snapshotter)
        : DataBuckets()
        , Logger(logger)
        , Snapshotter(snapshotter)
        , HostIndex(logger)
    {
    }

    bool TFreshStorage::PushSignal(TInstanceKey key, TSignalName signal, THostName host,
                                   TInstant timestamp, double value) {
        CheckStopped();
        return CastSeries<TFloatList>(EmplaceSeries(key, signal, host, ESeriesKind::Double))->PushValue(timestamp, value);
    }

    bool TFreshStorage::PushSignal(TInstanceKey key, TSignalName signal, THostName host,
                                   TInstant timestamp, TCountedSum value) {
        CheckStopped();
        return CastSeries<TCountedSumList>(EmplaceSeries(key, signal, host, ESeriesKind::CountedSum))->PushValue(timestamp, value);
    }

    bool TFreshStorage::PushSignal(TInstanceKey key, TSignalName signal, THostName host,
                                   TInstant timestamp, THistogram value) {
        CheckStopped();
        return CastSeries<THistogramList>(EmplaceSeries(key, signal, host, ESeriesKind::Histogram))->PushValue(timestamp, std::move(value));
    }

    size_t TFreshStorage::PushSignal(TInstanceKey key, TSignalName signal, THostName host, ESeriesKind kind, TInstant start,
                                     const TVector<NZoom::NValue::TValue>& values) {
        CheckStopped();
        auto series = EmplaceSeries(key, signal, host, kind);
        size_t acceptedValues = 0;
        auto timestamp = start;
        for (const auto& value : values) {
            TZoomValueWriter writer(series.Get(), timestamp);
            try {
                value.Update(writer);
            } catch (...) {
                Logger << "Failed write value on tag '" << key
                       << "', signal '" << signal
                       << "', host '" << host
                       << "': " << CurrentExceptionMessage();
                return acceptedValues; // ignore the rest of the chunk
            }

            if (writer.WasAccepted()) {
                acceptedValues++;
            }
            timestamp += ITERATION_SIZE;
        }
        return acceptedValues;
    }

    bool TFreshStorage::PushSignal(TInstanceKey key, TSignalName signal, THostName host, const TInstant timestamp,
                                   const NZoom::NValue::TValue& value) {
        CheckStopped();
        auto series = FindSeries(key, signal, host);
        if (series == nullptr) {
            TValueTyper typer;
            value.Update(typer);

            if (!typer.GetType().Defined()) {
                return false;
            }

            auto kind = typer.GetType().GetRef();
            series = EmplaceSeries(key, signal, host, kind);
        }

        TZoomValueWriter writer(series.Get(), timestamp);
        try {
            value.Update(writer);
        } catch (...) {
            Logger << "Failed write value on tag '" << key
                   << "', signal '" << signal
                   << "', host '" << host
                   << "': " << CurrentExceptionMessage();
            return false; // ignore the rest of the chunk
        }

        return writer.WasAccepted();
    }

    AcceptanceStat TFreshStorage::PushSignal(TInstant timestamp, TVector<TSingleValueRefWithKey>& values) {
        CheckStopped();

        // TSeriesFinder expect sorted vector to reduce hash map lookups
        Sort(values, [this](const TSingleValueRefWithKey& left, const TSingleValueRefWithKey& right) {
            const size_t leftChunk = GetBucketIdForSignal(left.GetSignalName());
            const size_t rightChunk = GetBucketIdForSignal(right.GetSignalName());
            return std::tie(leftChunk, left.GetSignalName(), left.GetHostName()) <
                std::tie(rightChunk, right.GetSignalName(), right.GetHostName());
        });
        TVector<TSingleValueRefWithKey> toCreate(Reserve(values.size()));
        AcceptanceStat acceptanceStat;
        size_t valuePos = 0;
        while (valuePos < values.size()) {
            toCreate.clear();
            const size_t curBucketId = GetBucketIdForSignal(values[valuePos].GetSignalName());
            auto& data = DataBuckets[curBucketId];
            {
                TLightReadGuard guard(data.DataMutex);
                TSeriesWriter<false> writer(data.Data, toCreate);
                while (valuePos < values.size() && curBucketId == GetBucketIdForSignal(values[valuePos].GetSignalName())) {
                    writer.Add(values[valuePos], timestamp, Logger, acceptanceStat);
                    ++valuePos;
                }
            }
            if (!toCreate.empty()) {
                TLightWriteGuard guard(data.DataMutex);
                TSeriesWriter<true> writer(data.Data, toCreate);
                for (auto& value : toCreate) {
                    writer.Add(value, timestamp, Logger, acceptanceStat);
                }
            }
        }

        return acceptanceStat;
    }

    TMaybe<double> TFreshStorage::GetFloatValue(TInstanceKey key, TSignalName signal, THostName host,
                                                TInstant timestamp) {
        auto series = FindSeries(key, signal, host);
        return series == nullptr ? Nothing() : CastSeries<TFloatList>(series)->GetValueAt(timestamp);
    }

    TMaybe<TCountedSum> TFreshStorage::GetCountedSumValue(TInstanceKey key, TSignalName signal, THostName host,
                                                          TInstant timestamp) {
        auto series = FindSeries(key, signal, host);
        return series == nullptr ? Nothing() : CastSeries<TCountedSumList>(series)->GetValueAt(timestamp);
    }

    TMaybe<THistogram> TFreshStorage::GetHistogramValue(TInstanceKey key, TSignalName signal, THostName host,
                                                        TInstant timestamp) {
        auto series = FindSeries(key, signal, host);
        return series == nullptr ? Nothing() : CastSeries<THistogramList>(series)->GetValueAt(timestamp);
    }

    void TFreshStorage::Start() {
        auto offset = MIN_START_OFFSET + TDuration::Seconds(RandomNumber(
            (MAX_START_OFFSET - MIN_START_OFFSET).Seconds()
        ));
        CleanerJob = NMonitoring::StartPeriodicJob(
            [this] {
                const auto now = TInstant::Now();
                Cleanup(now - FRESH_DURATION, now - CHUNK_SIZE);
            },
            // start cleanup in [MIN_START_OFFSET, MAX_START_OFFSET] seconds interval after each chunk should be completed
            // MIN_START_OFFSET seconds to account for possible batch delay in writers
            NMonitoring::TPeriodCalculator(
                NormalizeToIntervalDown(TInstant::Now(), CHUNK_SIZE) + offset,
                TDuration::Minutes(5)));
    }

    void TFreshStorage::Clear() {
        HostIndex.Clear();
        for (auto& bucket: DataBuckets) {
            TLightWriteGuard guard(bucket.DataMutex);
            bucket.Data.clear();
        }
    }

    void TFreshStorage::LoadSnapshots(const TFsPath& directory, TInstant now) {
        NMonitoring::TMeasuredMethod perf(Logger, "snapshots loading", NMetrics::FRESH_LOADING_TIME);
        Logger << "Loading snapshots";
        TGuard<TAdaptiveLock> guard(PersistenceMutex);
        NPersistence::TSnapshotReader reader(Logger);
        auto timestampFinder(TLastDumpTimeFinder::CreateForFresh(now));
        auto startCutoff(NormalizeToIntervalDown(now - FRESH_DURATION, CHUNK_SIZE));
        reader.ReadDirectory(
            directory,
            startCutoff,
            [this, &timestampFinder](TDataChunk&& chunk) {
                auto series = EmplaceSeries(chunk.Key, chunk.Signal, chunk.Host, chunk.Kind);
                try {
                    series->AppendChunk(chunk.Start, chunk.Data, chunk.ValuesCount);
                } catch (...) {
                    Logger << "Error while restoring chunk from host "
                           << "'" << chunk.Host << "', signal "
                           << "'" << chunk.Signal.GetName() << "', tag "
                           << "'" << chunk.Key.ToNamed() << "' with " << chunk.ValuesCount
                           << " values, bytes: " << BitStreamToString(chunk.Data, 128);
                }
                if (chunk.ValuesCount == ITERATIONS_IN_CHUNK) {
                    timestampFinder.AddFinished(chunk.Start);
                } else {
                    timestampFinder.AddIncomplete(chunk.Start);
                }
            },
            /* threads */ 9, NPersistence::EErrorMode::Ignore);
        const auto lastFilledTimestamp(timestampFinder.GetTimestamp());
        if (lastFilledTimestamp.Defined()) {
            LastFilledDumpStart = lastFilledTimestamp.GetRef();
        }
    }

    inline void TFreshStorage::CheckStopped() const {
        if (Stopped) {
            ythrow TWriteForbiddenError() << "Fresh storage has been stopped, writing is forbidden";
        }
    }

    void TFreshStorage::IterKnownTags(const TItypeName& itype, TSignalName signal, THostName host,
                                      const std::function<void(TInstanceKey key)>& callback) const {
        auto& bucket = DataBuckets[GetBucketIdForSignal(signal)];
        TLightReadGuard guard(bucket.DataMutex);

        const auto signalData = bucket.Data.find(signal);
        if (signalData == bucket.Data.end()) {
            return;
        }

        const auto hostData = signalData->second->find(host);
        if (hostData == signalData->second->end()) {
            return;
        }

        for (const auto& dataForHost : *hostData->second) {
            if (dataForHost.first.GetItype() == itype) {
                callback(dataForHost.first);
            }
        }
    }

    void TFreshStorage::IterKnownSignals(const TItypeName& itype, const std::function<void(TSignalName signal)>& callback) const {
        for (const auto& bucket: DataBuckets) {
            TLightReadGuard guard(bucket.DataMutex);
            for (const auto& signalData : bucket.Data) {
                bool haveSignal = false;
                for (const auto& hostData : *signalData.second) {
                    for (const auto& tagData : *hostData.second) {
                        if (tagData.first.GetItype() == itype) {
                            callback(signalData.first);
                            haveSignal = true;
                            break;
                        }
                    }
                    if (haveSignal) {
                        break;
                    }
                }
            }
        }
    }

    void TFreshStorage::IterAllSignals(TInstant start, TInstant end,
                                       const std::function<void(THostName host, TInstanceKey key, TSignalName signal)>& callback) const {
        TVector<TSignalName> toIterate;
        for (const auto& bucket: DataBuckets) {
            toIterate.clear();
            {
                TLightReadGuard guard(bucket.DataMutex);
                toIterate.reserve(bucket.Data.size());
                for (const auto& signalData : bucket.Data) {
                    toIterate.push_back(signalData.first);
                }
            }
            for (const auto& signal: toIterate) {
                TLightReadGuard guard(bucket.DataMutex);
                auto signalIt = bucket.Data.find(signal);
                if (signalIt != bucket.Data.end()) {
                    for (const auto& hostData : *signalIt->second) {
                        for (const auto& tagData : *hostData.second) {
                            if (tagData.second->GetStartTime() <= end && tagData.second->GetEndTime() >= start) {
                                callback(hostData.first, tagData.first, signalIt->first);
                            }
                        }
                    }
                }
            }
        }
    }

    TSeriesPtr TFreshStorage::EmplaceSeries(TInstanceKey key, TSignalName signal, THostName host, ESeriesKind kind) {
        auto series = FindSeries(key, signal, host);
        if (series != nullptr) {
            return series;
        }
        switch (kind) {
            case ESeriesKind::Double:
                return CreateSeries<TFloatList>(key, signal, host);
            case ESeriesKind::CountedSum:
                return CreateSeries<TCountedSumList>(key, signal, host);
            case ESeriesKind::Histogram:
                return CreateSeries<THistogramList>(key, signal, host);
        }
    }

    TSeriesPtr TFreshStorage::FindSeries(TInstanceKey key, TSignalName signal, THostName host) const {
        const auto& bucket = DataBuckets[GetBucketIdForSignal(signal)];
        TLightReadGuard guard(bucket.DataMutex);
        auto signalMapIt = bucket.Data.find(signal);
        if (signalMapIt == bucket.Data.end()) {
            return nullptr;
        }
        auto hostMapIt = signalMapIt->second->find(host);
        if (hostMapIt == signalMapIt->second->end()) {
            return nullptr;
        }
        auto seriesIt = hostMapIt->second->find(key);
        return seriesIt == hostMapIt->second->end() ? nullptr : seriesIt->second;
    }

    template <class T>
    TSeriesPtr TFreshStorage::CreateSeries(TInstanceKey key, TSignalName signal, THostName host) {
        auto& bucket = DataBuckets[GetBucketIdForSignal(signal)];
        TLightWriteGuard guard(bucket.DataMutex);
        auto& signalDataPtr = bucket.Data[signal];
        if (!signalDataPtr) {
            signalDataPtr.Reset(new THostMap());
        }
        auto& hostDataPtr = (*signalDataPtr)[host];
        if (!hostDataPtr) {
            hostDataPtr.Reset(new TTagMap());
        }
        auto it = hostDataPtr->find(key);
        TSeriesPtr series;
        if (it == hostDataPtr->end()) {
            series = (*hostDataPtr)[key] = MakeAtomicShared<T>();
            // update additional indexes
        } else {
            // account for possible race conditions
            series = it->second;
        }
        return series;
    }

    void TFreshStorage::IterHostsInItype(const TItypeName& itype, const std::function<void(THostName host)>& callback) const {
        for (const auto& bucket: DataBuckets) {
            TLightReadGuard guard(bucket.DataMutex);
            // this is not particularly efficient, maybe add another index?
            for (const auto& signalData : bucket.Data) {
                for (const auto& hostData : *signalData.second) {
                    for (const auto& tagData : *hostData.second) {
                        if (tagData.first.GetItype() == itype) {
                            callback(hostData.first);
                            break;
                        }
                    }
                }
            }
        }
    }

    bool TFreshStorage::IsRecordListExpired(const IRecordList& recordList, TInstant deadline) {
        // zero end time means there's no values yet
        // some other thread is likely writing into it right now
        const auto endTime = recordList.GetEndTime();
        return endTime < deadline && endTime != TInstant::Zero();
    }

    std::pair<size_t, bool> TFreshStorage::RemoveSeriesForSignal(TSignalName signal,
        const TVector<std::pair<THostName, TVector<TInstanceKey>>>& keys) {

        std::pair<size_t, bool> result(0, false);

        auto& bucket = DataBuckets[GetBucketIdForSignal(signal)];
        TLightWriteGuard guard(bucket.DataMutex);
        auto signalIt = bucket.Data.find(signal);
        if (signalIt == bucket.Data.end()) {
            return result;
        }

        bool rehashSignalData = false;
        auto& signalData = *signalIt->second;
        for (const auto& [host, hostKeys]: keys) {
            auto hostIt = signalData.find(host);
            if (hostIt == signalData.end()) {
                continue;
            }
            bool rehashHostData = false;
            auto& hostData = *hostIt->second;
            for (const auto& key: hostKeys) {
                if (hostData.erase(key)) {
                    ++result.first;
                    rehashHostData = true;
                }
            }
            if (hostData.empty()) {
                signalData.erase(hostIt);
                rehashSignalData = true;
            } else if (rehashHostData) {
                hostData.rehash(0);
            }
        }
        if (signalData.empty()) {
            bucket.Data.erase(signalIt);
            result.second = true;
        } else if (rehashSignalData) {
            signalData.rehash(0);
        }
        return result;
    }

    void TFreshStorage::RemoveOutdatedSeries(TInstant deadline) {
        Logger << "Starting fresh cleanup";
        size_t removedSeries = 0, totalSeries = 0;

        THashSet<TInstanceKey> allTags;
        TVector<TSignalName> toIterate;
        TVector<std::pair<THostName, TVector<TInstanceKey>>> emptyKeys;

        // reverse iteration to have no more than one intersection with read/write calls
        for (auto bucketIt = DataBuckets.rbegin(); bucketIt != DataBuckets.rend(); ++bucketIt) {
            auto& bucket = *bucketIt;
            toIterate.clear();
            {
                TLightReadGuard guard(bucket.DataMutex);
                toIterate.reserve(bucket.Data.size());
                for (const auto& signalData : bucket.Data) {
                    toIterate.push_back(signalData.first);
                }
            }
            bool rehashSignals = false;
            for (const auto& signal: toIterate) {
                emptyKeys.clear();
                {
                    TLightReadGuard guard(bucket.DataMutex);
                    auto signalIt = bucket.Data.find(signal);
                    if (signalIt != bucket.Data.end()) {
                        for (const auto& [host, hostData]: *signalIt->second) {
                            for (const auto& [key, keyData]: *hostData) {
                                totalSeries++;
                                allTags.insert(key);
                                if (IsRecordListExpired(*keyData, deadline)) {
                                    if (emptyKeys.empty() || emptyKeys.back().first != host) {
                                        emptyKeys.emplace_back(host, TVector<TInstanceKey>());
                                    }
                                    emptyKeys.back().second.push_back(key);
                                }
                            }
                        }
                    }
                }
                if (emptyKeys.empty()) {
                    continue;
                }

                bool removedSignal = false;
                size_t removedSeriesForSignal = 0;

                std::tie(removedSeriesForSignal, removedSignal) = RemoveSeriesForSignal(signal, emptyKeys);

                rehashSignals |= removedSignal;
                removedSeries += removedSeriesForSignal;
            }
            if (rehashSignals) {
                TLightWriteGuard guard(bucket.DataMutex);
                bucket.Data.rehash(0);
            }
        }

        Logger << "Removed " << removedSeries << " outdated series ["
               << totalSeries << " series total in " << allTags.size() << " tags]";

        TUnistat::Instance().PushSignalUnsafe(NMetrics::FRESH_SERIES_COUNT, totalSeries);
        TUnistat::Instance().PushSignalUnsafe(NMetrics::FRESH_TAGS_COUNT, allTags.size());
    }

    inline TMaybe<TInstant> TFreshStorage::GetMinStartTime() const {
        auto ts = TInstant::Max();
        for (const auto& bucket: DataBuckets) {
            TLightReadGuard guard(bucket.DataMutex);
            for (const auto& signalToHost : bucket.Data) {
                for (const auto& hostToTag : *signalToHost.second) {
                    for (const auto& tagToSeries : *hostToTag.second) {
                        const auto startTime(tagToSeries.second->GetStartTime());
                        if (startTime != TInstant::Zero()) {
                            ts = Min(ts, startTime);
                        }
                    }
                }
            }
        }
        if (ts == TInstant::Max()) {
            return Nothing();
        } else {
            return ts;
        }
    }

    void TFreshStorage::DumpChunksStartingAt(TInstant timestamp, ESnapshotMode mode) const {
        Logger << "Dumping chunks starting at " << timestamp.ToStringLocalUpToSeconds();
        TVector<TSignalName> toIterate;

        TVector<TDataChunk> chunks;
        for (const auto& bucket: DataBuckets) {
            toIterate.clear();
            {
                TLightReadGuard guard(bucket.DataMutex);
                toIterate.reserve(bucket.Data.size());
                for (const auto& signalData : bucket.Data) {
                    toIterate.push_back(signalData.first);
                }
            }
            for (const auto& signal: toIterate) {
                TLightReadGuard guard(bucket.DataMutex);
                auto signalIt = bucket.Data.find(signal);
                if (signalIt == bucket.Data.end()) {
                    continue;
                }
                for (const auto& hostToTag : *signalIt->second) {
                    for (const auto& tagToSeries : *hostToTag.second) {
                        const auto& series = tagToSeries.second;
                        auto chunk = series->GetChunkStartingAt(timestamp, mode);
                        if (chunk.Defined()) {
                            chunks.push_back(TDataChunk{
                                .Key = tagToSeries.first,
                                .Signal = signalIt->first,
                                .Host = hostToTag.first.GetName(),
                                .Start = chunk->StartTime,
                                .Kind = series->GetKind(),
                                .Data = std::move(chunk->Data),
                                .ValuesCount = chunk->ValuesCount,
                            });
                        }
                    }
                }
            }
        }
        if (chunks) {
            Snapshotter.WriteChunks(chunks, ToString(timestamp.Seconds()));
        } else {
            Logger << "No chunks to write";
        }
    }

    void TFreshStorage::DumpChunksStartingUntil(TInstant timestamp) {
        TGuard<TAdaptiveLock> guard(PersistenceMutex);
        auto dumpUntil = NormalizeToIntervalDown(timestamp, CHUNK_SIZE);

        {
            bool dataIsEmpty = true;
            for (const auto& bucket: DataBuckets) {
                TLightReadGuard bucketGuard(bucket.DataMutex);
                if (!bucket.Data.empty()) {
                    dataIsEmpty = false;
                    break;
                }
            }
            if (dataIsEmpty) {
                LastFilledDumpStart = dumpUntil;
                return;
            }
            if (LastFilledDumpStart == TInstant::Zero()) {
                const auto startTime(GetMinStartTime());
                if (startTime.Defined()) {
                    LastFilledDumpStart = startTime.GetRef() - CHUNK_SIZE;
                } else {
                    LastFilledDumpStart = dumpUntil;
                }
            }
        }

        // we might have missed some iterations for whatever reason, so dump them in a loop
        while (LastFilledDumpStart < dumpUntil) {
            DumpChunksStartingAt(LastFilledDumpStart + CHUNK_SIZE, ESnapshotMode::PadNulls);
            LastFilledDumpStart += CHUNK_SIZE;
        }
    }

    void TFreshStorage::Cleanup(TInstant deadline, TInstant startTimeToDump) {
        if (Stopped) {
            // we're shutting down, no point running cleanup anymore and it might just introduce some race conditions
            return;
        }
        NMonitoring::TMeasuredMethod perf(Logger, "fresh cleanup and persistence", NMetrics::FRESH_CLEANUP_TIME);
        RemoveOutdatedSeries(deadline);
        DumpChunksStartingUntil(startTimeToDump);
        Snapshotter.Cleanup(LastFilledDumpStart - SNAPSHOT_EXPIRATION);
    }

    void TFreshStorage::Shutdown() {
        Logger << "Shutdown requested, forbidding all writes and dumping the database";
        Stopped = true;
        CleanerJob.Reset();

        auto maxEndTime = TInstant::Zero();

        {
            for (const auto& bucket: DataBuckets) {
                TLightReadGuard guard(bucket.DataMutex);
                for (const auto& signalData : bucket.Data) {
                    for (const auto& hostData : *signalData.second) {
                        for (const auto& tagData : *hostData.second) {
                            maxEndTime = Max(tagData.second->GetEndTime(), maxEndTime);
                        }
                    }
                }
            }
        }
        auto normalized = NormalizeToIntervalDown(maxEndTime, CHUNK_SIZE);
        // in case we haven't dumped some full chunks, do it now
        DumpChunksStartingUntil(normalized - CHUNK_SIZE);
        {
            TGuard<TAdaptiveLock> guard(PersistenceMutex);
            // dump last chunks as is, without padding them with nulls
            DumpChunksStartingAt(normalized, ESnapshotMode::AsIs);
        }
    }
} // namespace NYasmServer
