#pragma once

#include "host_index.h"

#include <infra/yasm/zoom/components/compression/zoom_types.h>
#include <infra/yasm/zoom/components/compression/series.h>
#include <infra/yasm/server/persistence/snapshot_manager.h>

#include <infra/yasm/common/labels/tags/instance_key.h>
#include <infra/yasm/common/labels/signal/signal_name.h>
#include <infra/yasm/common/labels/host/host.h>
#include <infra/yasm/common/points/value/types.h>

#include <infra/monitoring/common/periodic_task.h>

#include <library/cpp/logger/global/global.h>
#include <library/cpp/threading/light_rw_lock/lightrwlock.h>
#include <library/cpp/containers/flat_hash/flat_hash.h>

#include <util/generic/hash.h>

namespace NYasmServer {
    using TItypeName = TString;

    class TWriteForbiddenError: public yexception {
    };

    class TSingleValueRefWithKey {
    public:
        TSingleValueRefWithKey(NZoom::NSignal::TSignalName signalName, NZoom::NHost::THostName hostName,
            NTags::TInstanceKey instanceKey, const NZoom::NValue::IFlatValue& value)
            : SignalName(signalName)
            , HostName(hostName)
            , InstanceKey(instanceKey)
            , Value(&value) {
        }
        TSingleValueRefWithKey(const TSingleValueRefWithKey&) = default;

        const NZoom::NSignal::TSignalName& GetSignalName() const {
            return SignalName;
        };
        const NZoom::NHost::THostName& GetHostName() const {
            return HostName;
        };
        const NTags::TInstanceKey& GetInstanceKey() const {
            return InstanceKey;
        }
        const NZoom::NValue::IFlatValue& GetValue() const {
            return *Value;
        }
        bool operator < (const TSingleValueRefWithKey& other) const noexcept {
            return std::tie(SignalName, HostName) < std::tie(other.SignalName, other.HostName);
        }

    private:
        NZoom::NSignal::TSignalName SignalName;
        NZoom::NHost::THostName HostName;
        NTags::TInstanceKey InstanceKey;
        const NZoom::NValue::IFlatValue* Value;
    };

    struct AcceptanceStat {
        size_t accepted = 0;

        size_t acceptedGroupSmallHgram = 0;
        size_t acceptedGroupNormalHgram = 0;
        size_t acceptedHostSmallHgram = 0;
        size_t acceptedHostNormalHgram = 0;
    };

    class TFreshStorage {
    public:
        TFreshStorage(TLog& logger = TLoggerOperator<TNullLog>::Log(),
                      NPersistence::ISnapshotManager& snapshotter = NPersistence::TDevNullManager::GetSingleton());

        bool PushSignal(NTags::TInstanceKey key, NZoom::NSignal::TSignalName signal, NZoom::NHost::THostName host,
                        TInstant timestamp, double value);
        bool PushSignal(NTags::TInstanceKey key, NZoom::NSignal::TSignalName signal, NZoom::NHost::THostName host,
                        TInstant timestamp, TCountedSum value);
        bool PushSignal(NTags::TInstanceKey key, NZoom::NSignal::TSignalName signal, NZoom::NHost::THostName host,
                        TInstant timestamp, THistogram value);

        // returns number of accepted values
        size_t PushSignal(NTags::TInstanceKey key, NZoom::NSignal::TSignalName signal, NZoom::NHost::THostName host, ESeriesKind kind, TInstant start,
                          const TVector<NZoom::NValue::TValue>& values);

        bool PushSignal(NTags::TInstanceKey key, NZoom::NSignal::TSignalName signal, NZoom::NHost::THostName host, TInstant start,
                        const NZoom::NValue::TValue& value);

        AcceptanceStat PushSignal(TInstant timestamp, TVector<TSingleValueRefWithKey>& values);

        TMaybe<double> GetFloatValue(NTags::TInstanceKey key, NZoom::NSignal::TSignalName signal, NZoom::NHost::THostName host,
                                     TInstant timestamp);
        TMaybe<TCountedSum> GetCountedSumValue(NTags::TInstanceKey key, NZoom::NSignal::TSignalName signal, NZoom::NHost::THostName host,
                                               TInstant timestamp);
        TMaybe<THistogram> GetHistogramValue(NTags::TInstanceKey key, NZoom::NSignal::TSignalName signal, NZoom::NHost::THostName host,
                                             TInstant timestamp);

        void IterKnownTags(const TItypeName& itype, NZoom::NSignal::TSignalName signal, NZoom::NHost::THostName host,
                           const std::function<void(NTags::TInstanceKey key)>& callback) const;

        void IterKnownSignals(const TItypeName& itype, const std::function<void(NZoom::NSignal::TSignalName signal)>& callback) const;

        void IterAllSignals(TInstant start, TInstant end,
                            const std::function<void(NZoom::NHost::THostName host, NTags::TInstanceKey key, NZoom::NSignal::TSignalName signal)>& callback) const;

        TSeriesPtr EmplaceSeries(NTags::TInstanceKey key, NZoom::NSignal::TSignalName signal, NZoom::NHost::THostName host, ESeriesKind kind);

        TSeriesPtr FindSeries(NTags::TInstanceKey key, NZoom::NSignal::TSignalName signal, NZoom::NHost::THostName host) const;

        void EmplaceHostIndex(NZoom::NHost::THostName group, const THashMap<NZoom::NHost::THostName, THostIndex::TTimestampPair>& hosts) {
            HostIndex.Emplace(group, hosts);
        }

        void EmplaceHostIndex(NZoom::NHost::THostName group, NZoom::NHost::THostName host, TInstant time) {
            HostIndex.Emplace(group, host, time);
        }

        TVector<NZoom::NHost::THostName> GetHostsInGroup(NZoom::NHost::THostName group) const {
            return HostIndex.GetHostsInGroup(group);
        }

        void IterHostsInGroup(NZoom::NHost::THostName group, TInstant from, TInstant to, const std::function<void(NZoom::NHost::THostName host)>& callback) const {
            HostIndex.IterHostsInGroup(group, from, to, callback);
        }

        void IterHostsInItype(const TItypeName& itype, const std::function<void(NZoom::NHost::THostName host)>& callback) const;

        void Clear();

        void Cleanup(TInstant deadline, TInstant startTimeToDump);

        void Shutdown();

        void Start();

        void LoadSnapshots(const TFsPath& directory, TInstant now);

        bool IsStopped() const {
            return Stopped;
        }

        TInstant GetLastFilledDumpStart() const {
            return LastFilledDumpStart;
        }

    private:
        template <bool CanCreate>
        class TSeriesWriter;

        using TTagMap = NFH::TFlatHashMap<NTags::TInstanceKey, TSeriesPtr>;
        using THostMap = NFH::TFlatHashMap<NZoom::NHost::THostName, THolder<TTagMap>>;
        using TSignalMap = NFH::TFlatHashMap<NZoom::NSignal::TSignalName, THolder<THostMap>>;
        struct TDataBucketWithLock {
            TSignalMap Data;
            TLightRWLock DataMutex;
        };

        static bool IsRecordListExpired(const IRecordList& recordList, TInstant deadline);

        template <class T>
        TSeriesPtr CreateSeries(NTags::TInstanceKey key, NZoom::NSignal::TSignalName signal, NZoom::NHost::THostName host);

        void CheckStopped() const;
        void RemoveOutdatedSeries(TInstant deadline);
        /* returns number of series removed and a flag showing that signal was cleaned up by the call */
        std::pair<size_t, bool> RemoveSeriesForSignal(NZoom::NSignal::TSignalName signal,
            const TVector<std::pair<NZoom::NHost::THostName, TVector<NTags::TInstanceKey>>>& keysToRemove);
        void DumpChunksStartingUntil(TInstant timestamp);
        void DumpChunksStartingAt(TInstant timestamp, ESnapshotMode mode) const;
        TMaybe<TInstant> GetMinStartTime() const;
        size_t GetBucketIdForSignal(NZoom::NSignal::TSignalName signal) const noexcept {
            return signal.Hash() % DataBuckets.size();
        }

        std::array<TDataBucketWithLock, 128> DataBuckets;
        TAdaptiveLock PersistenceMutex;

        TLog& Logger;
        NPersistence::ISnapshotManager& Snapshotter;
        THostIndex HostIndex;
        NMonitoring::TPeriodicTaskPtr CleanerJob;
        TInstant LastFilledDumpStart;
        bool Stopped = false;
    };
} // namespace NYasmServer
