#include "fresh.h"

#include <library/cpp/testing/unittest/gtest.h>
#include <library/cpp/testing/unittest/registar.h>

namespace NYasmServer {
    using NTags::TInstanceKey;
    using NZoom::NHost::THostName;
    using NZoom::NSignal::TSignalName;

    const auto SIGNAL = TSignalName{TStringBuf("signal")};
    const auto HOST = THostName{TStringBuf("host")};

    Y_UNIT_TEST_SUITE(TestFresh) {
        Y_UNIT_TEST(TestWorkingWithFloatSignal) {
            TFreshStorage storage;
            auto now = TInstant::Seconds(100000);
            auto instanceKey = TInstanceKey::FromNamed("tag");
            storage.PushSignal(instanceKey, SIGNAL, HOST, now, 12345);
            UNIT_ASSERT_DOUBLES_EQUAL(storage.GetFloatValue(instanceKey, SIGNAL, HOST, now).GetRef(), 12345, 0.001);

            storage.PushSignal(instanceKey, SIGNAL, HOST, now, 43534);
            // nope, we don't allow changing values
            UNIT_ASSERT_DOUBLES_EQUAL(storage.GetFloatValue(instanceKey, SIGNAL, HOST, now).GetRef(), 12345, 0.001);
        }

        Y_UNIT_TEST(TestRequestingMissingSignal) {
            TFreshStorage storage;
            auto now = TInstant::Seconds(100000);
            UNIT_ASSERT(!storage.GetFloatValue(TInstanceKey::FromNamed("tag"), SIGNAL, HOST, now).Defined());
        }

        Y_UNIT_TEST(TestWorkingWithCountedSumSignals) {
            TFreshStorage storage;
            auto now = TInstant::Seconds(100000);

            storage.PushSignal(TInstanceKey::FromNamed("tag"), SIGNAL, HOST, now, TCountedSum(12345, 10.0));
            auto value = storage.GetCountedSumValue(TInstanceKey::FromNamed("tag"), SIGNAL, HOST, now);
            UNIT_ASSERT(value.Defined());
            UNIT_ASSERT_EQUAL_C(value.GetRef().GetCount(), 12345, value.GetRef().GetCount());
            UNIT_ASSERT_DOUBLES_EQUAL(value.GetRef().GetSum(), 10.0, 0.001);
        }

        Y_UNIT_TEST(TestWorkingWithUgram) {
            TFreshStorage storage;
            auto now = TInstant::Seconds(100000);

            TUserHistogram hist;
            hist.MutableBuckets().push_back(TUserHistogram::TBucket{.LowerBound = 0, .Weight = 10});
            hist.MutableBuckets().push_back(TUserHistogram::TBucket{.LowerBound = 1, .Weight = 5});
            hist.MutableBuckets().push_back(TUserHistogram::TBucket{.LowerBound = 2, .Weight = 0});

            storage.PushSignal(TInstanceKey::FromNamed("tag"), SIGNAL, HOST, now, std::move(hist));
            auto value = storage.GetHistogramValue(TInstanceKey::FromNamed("tag"), SIGNAL, HOST, now);
            UNIT_ASSERT(value.Defined());
            UNIT_ASSERT(value.GetRef().GetKind() == EHistogramKind::User);
            const auto& buckets = value.GetRef().AsUserHistogram().GetBuckets();

            UNIT_ASSERT_EQUAL(buckets[0].Weight, 10);
            UNIT_ASSERT_EQUAL(buckets[0].LowerBound, 0);
            UNIT_ASSERT_EQUAL(buckets[1].Weight, 5);
            UNIT_ASSERT_EQUAL(buckets[1].LowerBound, 1);
        }

        Y_UNIT_TEST(TestWorkingWithLogHistogram) {
            TFreshStorage storage;
            auto now = TInstant::Seconds(100000);

            TLogHistogram hist;
            hist.SetZeroCount(10);
            hist.SetStartPower(-5);
            hist.MutableWeights().push_back(5);

            UNIT_ASSERT(storage.PushSignal(TInstanceKey::FromNamed("tag"), SIGNAL, HOST, now, std::move(hist)));
            auto value = storage.GetHistogramValue(TInstanceKey::FromNamed("tag"), SIGNAL, HOST, now);
            UNIT_ASSERT(value.Defined());
            UNIT_ASSERT(value.GetRef().GetKind() == EHistogramKind::Log);
            const auto& result = value.GetRef().AsLogHistogram();

            UNIT_ASSERT_EQUAL(result.GetZeroCount(), 10);
            UNIT_ASSERT_EQUAL(result.GetStartPower(), -5);
            UNIT_ASSERT_EQUAL(result.GetWeights()[0], 5);
        }

        Y_UNIT_TEST(TestCleanup) {
            TFreshStorage storage;
            auto past = TInstant::Seconds(100000) - TDuration::Minutes(30);
            auto now = TInstant::Seconds(100000);

            UNIT_ASSERT(storage.PushSignal(TInstanceKey::FromNamed("tag1"), TStringBuf("signal1"), TStringBuf("host1"), past, 15));
            UNIT_ASSERT(storage.PushSignal(TInstanceKey::FromNamed("tag1"), TStringBuf("signal1"), TStringBuf("host2"), now, 15));
            UNIT_ASSERT(storage.PushSignal(TInstanceKey::FromNamed("tag1"), TStringBuf("signal2"), TStringBuf("host1"), past, 15));
            UNIT_ASSERT(storage.PushSignal(TInstanceKey::FromNamed("tag2"), TStringBuf("signal3"), TStringBuf("host1"), past, 15));

            // remove all signals from the 'past'
            storage.Cleanup(now - TDuration::Minutes(5), now);

            UNIT_ASSERT_EQUAL(storage.FindSeries(TInstanceKey::FromNamed("tag1"), TStringBuf("signal1"), TStringBuf("host1")), nullptr);
            UNIT_ASSERT_UNEQUAL(storage.FindSeries(TInstanceKey::FromNamed("tag1"), TStringBuf("signal1"), TStringBuf("host2")), nullptr);
            UNIT_ASSERT_EQUAL(storage.FindSeries(TInstanceKey::FromNamed("tag1"), TStringBuf("signal2"), TStringBuf("host1")), nullptr);
            UNIT_ASSERT_EQUAL(storage.FindSeries(TInstanceKey::FromNamed("tag2"), TStringBuf("signal3"), TStringBuf("host1")), nullptr);

            TVector<TSignalName> knownSignals;
            storage.IterKnownSignals("tag1", [&knownSignals](TSignalName signal) { knownSignals.push_back(signal); });
            UNIT_ASSERT_VALUES_EQUAL(knownSignals, (TVector<TSignalName>{TStringBuf("signal1")})); // signals 2 and 3 removed
        }

        Y_UNIT_TEST(TestIterAllSignals) {
            auto now = TInstant::Seconds(100000);

            TFreshStorage storage;
            const TInstanceKey instanceKey(TInstanceKey::FromNamed("tag1"));
            const THostName host(TStringBuf("host1"));
            const TSignalName signal(TStringBuf("signal1"));
            UNIT_ASSERT(storage.PushSignal(instanceKey, signal, host, now, 42));

            bool hasSignals = false;
            storage.IterAllSignals(now, now, [&](const THostName& givenHost, TInstanceKey givenTag, TSignalName givenSignal) {
                UNIT_ASSERT_VALUES_EQUAL(host, givenHost);
                UNIT_ASSERT_VALUES_EQUAL(instanceKey, givenTag);
                UNIT_ASSERT_VALUES_EQUAL(signal, givenSignal);
                hasSignals = true;
            });
            UNIT_ASSERT(hasSignals);
        }
    }
} // namespace NYasmServer
