#include "histogram_list.h"

#include <library/cpp/testing/unittest/gtest.h>
#include <library/cpp/testing/unittest/registar.h>

namespace NYasmServer {

    Y_UNIT_TEST_SUITE(TestHistogramList) {
        Y_UNIT_TEST(TestReadingAndWritingManyHistograms) {
            THistogramList storage;

            auto start = TInstant::Seconds(100000);
            for (size_t i = 0; i < 1000; i++) {
                TSimpleHistogram hist;
                hist.SetZeroCount(i);
                UNIT_ASSERT(storage.PushValue(start + ITERATION_SIZE * i, std::move(hist)));
            }

            auto value = storage.GetValueAt(start); // too old
            UNIT_ASSERT(!value.Defined());

            for (size_t i = 700; i < 1000; i++) {
                value = storage.GetValueAt(start + ITERATION_SIZE * i);
                UNIT_ASSERT(value.Defined());
                UNIT_ASSERT_EQUAL(value->GetKind(), EHistogramKind::Simple);
                UNIT_ASSERT_EQUAL_C(value->AsSimpleHistogram().GetZeroCount(), i, "Got " << value->AsSimpleHistogram().GetZeroCount() << " instead of " << i);
            }
        }

        Y_UNIT_TEST(TestReadingAndWritingManyChangingHistograms) {
            THistogramList storage;

            auto start = TInstant::Seconds(100000);
            for (size_t i = 0; i < 100; i++) {
                TSimpleHistogram hist;
                hist.SetZeroCount(i);
                hist.MutableValues() = {{(double)i, (double)i + 0.015}};
                UNIT_ASSERT(storage.PushValue(start + ITERATION_SIZE * i, std::move(hist)));

                auto value = storage.GetValueAt(start + ITERATION_SIZE * i);
                UNIT_ASSERT(value.Defined());

                UNIT_ASSERT_EQUAL(value->AsSimpleHistogram().GetZeroCount(), i);
                UNIT_ASSERT_DOUBLES_EQUAL(value->AsSimpleHistogram().GetValues()[0], (double)i, 0.001);
                UNIT_ASSERT_DOUBLES_EQUAL(value->AsSimpleHistogram().GetValues()[1], (double)i + 0.015, 0.001);
            }
        }

        Y_UNIT_TEST(TestReadingAndWritingDifferentKinds) {
            THistogramList storage;

            auto start = TInstant::Seconds(100000);

            TSimpleHistogram simpleEmpty;
            UNIT_ASSERT(storage.PushValue(start + ITERATION_SIZE * 0, std::move(simpleEmpty)));

            TSimpleHistogram zeroCount;
            zeroCount.SetZeroCount(15);
            UNIT_ASSERT(storage.PushValue(start + ITERATION_SIZE * 1, std::move(zeroCount)));

            TSimpleHistogram singleValue;
            singleValue.MutableValues().push_back(28);
            UNIT_ASSERT(storage.PushValue(start + ITERATION_SIZE * 2, std::move(singleValue)));

            TSimpleHistogram simple;
            simple.SetZeroCount(89);
            simple.MutableValues().push_back(1);
            simple.MutableValues().push_back(2);
            simple.MutableValues().push_back(3.5);
            UNIT_ASSERT(storage.PushValue(start + ITERATION_SIZE * 3, std::move(simple)));

            // whoos, ITERATION_SIZE * 4 is a null value!

            TUserHistogram user;
            user.MutableBuckets().push_back(TUserHistogram::TBucket{.LowerBound = 0, .Weight = 1});
            user.MutableBuckets().push_back(TUserHistogram::TBucket{.LowerBound = 2, .Weight = 3});
            user.MutableBuckets().push_back(TUserHistogram::TBucket{.LowerBound = 4, .Weight = 5});
            UNIT_ASSERT(storage.PushValue(start + ITERATION_SIZE * 5, std::move(user)));

            TLogHistogram log;
            log.SetZeroCount(15);
            log.SetStartPower(-3);
            log.MutableWeights().push_back(1);
            log.MutableWeights().push_back(2);
            UNIT_ASSERT(storage.PushValue(start + ITERATION_SIZE * 6, std::move(log)));

            TMaybe<THistogram> value;

            // totally empty
            value = storage.GetValueAt(start + ITERATION_SIZE * 0);
            UNIT_ASSERT(value.Defined());
            UNIT_ASSERT_EQUAL(value->GetKind(), EHistogramKind::Simple);
            UNIT_ASSERT_EQUAL(value->AsSimpleHistogram().GetZeroCount(), 0);
            UNIT_ASSERT(value->AsSimpleHistogram().GetValues().empty());

            // zero count
            value = storage.GetValueAt(start + ITERATION_SIZE * 1);
            UNIT_ASSERT(value.Defined());
            UNIT_ASSERT_EQUAL(value->GetKind(), EHistogramKind::Simple);
            UNIT_ASSERT_EQUAL(value->AsSimpleHistogram().GetZeroCount(), 15);
            UNIT_ASSERT(value->AsSimpleHistogram().GetValues().empty());

            // single value
            value = storage.GetValueAt(start + ITERATION_SIZE * 2);
            UNIT_ASSERT(value.Defined());
            UNIT_ASSERT_EQUAL(value->GetKind(), EHistogramKind::Simple);
            UNIT_ASSERT_EQUAL(value->AsSimpleHistogram().GetZeroCount(), 0);
            UNIT_ASSERT_EQUAL(value->AsSimpleHistogram().GetValues().size(), 1);
            UNIT_ASSERT_EQUAL(value->AsSimpleHistogram().GetValues()[0], 28);

            // regular simple histogram
            value = storage.GetValueAt(start + ITERATION_SIZE * 3);
            UNIT_ASSERT(value.Defined());
            UNIT_ASSERT_EQUAL(value->GetKind(), EHistogramKind::Simple);
            UNIT_ASSERT_EQUAL(value->AsSimpleHistogram().GetZeroCount(), 89);
            UNIT_ASSERT_EQUAL(value->AsSimpleHistogram().GetValues().size(), 3);
            UNIT_ASSERT_EQUAL(value->AsSimpleHistogram().GetValues(), (TVector<double>{1, 2, 3.5}));

            // null value
            value = storage.GetValueAt(start + ITERATION_SIZE * 4);
            UNIT_ASSERT(!value.Defined());

            // ugram
            value = storage.GetValueAt(start + ITERATION_SIZE * 5);
            UNIT_ASSERT(value.Defined());
            UNIT_ASSERT_EQUAL(value->GetKind(), EHistogramKind::User);
            UNIT_ASSERT_EQUAL(value->AsUserHistogram().GetBuckets().size(), 3);
            UNIT_ASSERT_EQUAL(value->AsUserHistogram().GetBuckets()[0], (TUserHistogram::TBucket{.LowerBound = 0, .Weight = 1}));
            UNIT_ASSERT_EQUAL(value->AsUserHistogram().GetBuckets()[1], (TUserHistogram::TBucket{.LowerBound = 2, .Weight = 3}));
            UNIT_ASSERT_EQUAL(value->AsUserHistogram().GetBuckets()[2], (TUserHistogram::TBucket{.LowerBound = 4, .Weight = 5}));

            // logarithmic
            value = storage.GetValueAt(start + ITERATION_SIZE * 6);
            UNIT_ASSERT(value.Defined());
            UNIT_ASSERT_EQUAL(value->GetKind(), EHistogramKind::Log);
            UNIT_ASSERT_EQUAL(value->AsLogHistogram().GetZeroCount(), 15);
            UNIT_ASSERT_EQUAL(value->AsLogHistogram().GetStartPower(), -3);
            UNIT_ASSERT_EQUAL(value->AsLogHistogram().GetWeights(), (TVector<ui64>{1, 2}));
        }
    }
} // namespace NYasmServer
