#include "host_index.h"

#include <infra/yasm/server/lib/metrics.h>

#include <infra/yasm/server/common/const.h>

#include <infra/monitoring/common/collections.h>
#include <infra/monitoring/common/perf.h>

#include <library/cpp/unistat/unistat.h>

namespace NYasmServer {
    using NZoom::NHost::THostName;

    THostIndex::THostIndex(TLog& logger)
        : Logger(logger)
    {
        CleanerJob = NMonitoring::StartPeriodicJob(
            [this] {
                Cleanup(TInstant::Now() - FRESH_DURATION);
            },
            TDuration::Minutes(5));
    }

    inline void THostIndex::UpdateGroup(THostsSet& storage, THostName host, TTimestampPair seen) {
        auto it = storage.find(host);
        if (it.IsEnd()) {
            storage.insert(TGroupHost{.HostName = host, .FirstSeen = seen.FirstSeen, .LastSeen = seen.LastSeen});
        } else {
            // this is fine as long as LastSeen is not used in hash
            auto& mut = const_cast<TGroupHost&>(*it);
            mut.LastSeen = Max(it->LastSeen, seen.LastSeen);
            mut.FirstSeen = Min(it->FirstSeen, seen.FirstSeen);
        }
    }

    inline void THostIndex::UpdateGroup(THostsSet& storage, THostName host, TInstant time) {
        auto it = storage.find(host);
        if (it.IsEnd()) {
            storage.insert(TGroupHost{.HostName = host, .FirstSeen = time, .LastSeen = time});
        } else {
            // this is fine as long as LastSeen is not used in hash
            auto& mut = const_cast<TGroupHost&>(*it);
            mut.LastSeen = Max(it->LastSeen, time);
        }
    }

    void THostIndex::Clear() {
        TGuard<TAdaptiveLock> guard{Lock};
        HostsInGroups.clear();
    }

    void THostIndex::Cleanup(TInstant minTime) {
        NMonitoring::TMeasuredMethod perf(Logger, "groups index cleanup", NMetrics::GROUPS_INDEX_CLEANUP_TIME);
        Logger << "Starting groups index cleanup";
        TVector<THostName> knownGroups;
        {
            // copy the keys first to avoid blocking for too long
            TGuard<TAdaptiveLock> guard{Lock};
            knownGroups = NMonitoring::MapKeys(HostsInGroups);
        }

        TVector<THostName> hostsToDelete;
        for (const auto& groupName : knownGroups) {
            hostsToDelete.clear();
            TGuard<TAdaptiveLock> guard{Lock};
            auto& groupData = HostsInGroups.at(groupName);
            for (const auto& host : groupData) {
                if (host.LastSeen < minTime) {
                    hostsToDelete.push_back(host.HostName);
                }
            }
            for (const auto& host : hostsToDelete) {
                groupData.erase(groupData.find(host));
            }
            if (groupData.empty()) {
                HostsInGroups.erase(groupName);
            }
        }
    }

    void THostIndex::Emplace(THostName group, const THashMap<THostName, TTimestampPair>& hosts) {
        TGuard<TAdaptiveLock> guard{Lock};
        auto& hostSet = HostsInGroups[group];
        for (const auto& pair : hosts) {
            UpdateGroup(hostSet, pair.first, pair.second);
        }
        TUnistat::Instance().PushSignalUnsafe(NMetrics::GROUPS_INDEX_SIZE, HostsInGroups.size());
    }

    void THostIndex::Emplace(THostName group, THostName host, TInstant time) {
        TGuard<TAdaptiveLock> guard{Lock};
        auto& hostSet = HostsInGroups[group];
        UpdateGroup(hostSet, host, time);
        TUnistat::Instance().PushSignalUnsafe(NMetrics::GROUPS_INDEX_SIZE, HostsInGroups.size());
    }

    TVector<THostName> THostIndex::GetHostsInGroup(THostName group) const {
        // returns hosts in group by its name, empty vector if not found
        TVector<THostName> output;
        TGuard<TAdaptiveLock> guard{Lock};
        auto it = HostsInGroups.find(group);
        if (it.IsEnd()) {
            return output;
        }
        output.reserve(it->second.size());
        for (const auto& host : it->second) {
            output.emplace_back(host.HostName);
        }
        return output;
    }

    void THostIndex::IterHostsInGroup(THostName group, TInstant from, TInstant to, const std::function<void(THostName host)>& callback) const {
        TGuard<TAdaptiveLock> guard{Lock};
        auto it = HostsInGroups.find(group);
        if (it.IsEnd()) {
            return;
        }
        for (const auto& host : it->second) {
            // this is not strictly correct compared to python version as there might actually not be any values between to and from
            // for example this host is seen once every 25 minutes and we request some 5 minutes between its appearances
            if (host.FirstSeen <= to && host.LastSeen >= from) {
                callback(host.HostName);
            }
        }
    }
} // namespace NYasmServer
