#pragma once

#include <infra/yasm/common/labels/host/host.h>
#include <infra/monitoring/common/periodic_task.h>

#include <library/cpp/logger/global/global.h>

#include <util/datetime/base.h>
#include <util/generic/hash.h>
#include <util/generic/hash_set.h>
#include <util/generic/vector.h>
#include <util/system/spinlock.h>

namespace NYasmServer {
    class THostIndex {
    public:
        struct TTimestampPair {
            TInstant FirstSeen = TInstant::Max();
            TInstant LastSeen = TInstant::Zero();

            TTimestampPair() {
            }

            TTimestampPair(TInstant timestamp)
                : FirstSeen(timestamp)
                , LastSeen(timestamp)
            {
            }

            void Merge(TInstant firstSeen, TInstant lastSeen) {
                FirstSeen = Min(FirstSeen, firstSeen);
                LastSeen = Max(LastSeen, lastSeen);
            }
        };

    public:
        THostIndex(TLog& logger);

        THostIndex()
            : THostIndex(TLoggerOperator<TNullLog>::Log())
        {
        }

        void Emplace(NZoom::NHost::THostName group, const THashMap<NZoom::NHost::THostName, TTimestampPair>& hosts);
        void Emplace(NZoom::NHost::THostName group, NZoom::NHost::THostName host, TInstant time);

        TVector<NZoom::NHost::THostName> GetHostsInGroup(NZoom::NHost::THostName group) const;
        void IterHostsInGroup(NZoom::NHost::THostName group, TInstant from, TInstant to, const std::function<void(NZoom::NHost::THostName host)>& callback) const;

        void Cleanup(TInstant minTime);
        void Clear();

    private:
        struct TGroupHost {
            NZoom::NHost::THostName HostName;
            TInstant FirstSeen;
            TInstant LastSeen;
        };

        struct TGroupHostHasher {
            inline size_t operator()(const TGroupHost& host) const noexcept {
                return host.HostName.Hash();
            }

            inline size_t operator()(const NZoom::NHost::THostName& host) const noexcept {
                return host.Hash();
            }
        };

        struct TGroupHostEquals {
            inline bool operator()(const TGroupHost& left, const TGroupHost& right) const noexcept {
                return left.HostName == right.HostName;
            }

            inline bool operator()(const TGroupHost& left, const NZoom::NHost::THostName& right) const noexcept {
                return left.HostName == right;
            }
        };

        using THostsSet = THashSet<TGroupHost, TGroupHostHasher, TGroupHostEquals>;

    private:
        static void UpdateGroup(THostsSet& storage, NZoom::NHost::THostName host, TTimestampPair seen);
        static void UpdateGroup(THostsSet& storage, NZoom::NHost::THostName host, TInstant time);

    private:
        TAdaptiveLock Lock;
        THashMap<NZoom::NHost::THostName, THostsSet> HostsInGroups;
        NMonitoring::TPeriodicTaskPtr CleanerJob;
        TLog& Logger;
    };
} // namespace NYasmServer
