#include "fresh.h"
#include "record_list.h"

#include <infra/yasm/common/interval.h>
#include <infra/yasm/server/persistence/reader.h>

#include <infra/monitoring/common/test_helpers.h>

#include <library/cpp/testing/unittest/gtest.h>
#include <library/cpp/testing/unittest/registar.h>

#include <util/folder/path.h>
#include <util/folder/tempdir.h>

#include <util/generic/xrange.h>

namespace NYasmServer {

    using namespace NYasm::NCommon::NInterval;
    using NPersistence::TSnapshotManager;
    using NTags::TInstanceKey;
    using NZoom::NSignal::TSignalName;
    using NZoom::NHost::THostName;

    const auto HOST = THostName{TStringBuf("host")};
    const auto SIGNAL = TSignalName{TStringBuf("signal")};
    const auto INSTANCE_KEY = TInstanceKey::FromNamed("tag");

    Y_UNIT_TEST_SUITE(TestFreshPersistence) {
        Y_UNIT_TEST(TestTailWriteAndRead) {
            TTempDir dir;
            auto now = TInstant::Seconds(100000);
            {
                TSnapshotManager snapshotter(dir.Path(), GetLogger());
                TFreshStorage storage(GetLogger(), snapshotter);
                storage.PushSignal(INSTANCE_KEY, SIGNAL, HOST, now, 12345);
                storage.Shutdown();
            }
            TFreshStorage newStorage(GetLogger());
            newStorage.LoadSnapshots(dir.Path(), now + FRESH_DURATION);
            UNIT_ASSERT_DOUBLES_EQUAL(newStorage.GetFloatValue(INSTANCE_KEY, SIGNAL, HOST, now).GetRef(), 12345, 0.001);
            UNIT_ASSERT_VALUES_EQUAL(newStorage.GetLastFilledDumpStart(), TInstant::Zero());

            // now try writing to ensure we loaded encoder state properly
            newStorage.PushSignal(INSTANCE_KEY, SIGNAL, HOST, now + ITERATION_SIZE, 12346);

            UNIT_ASSERT_DOUBLES_EQUAL(newStorage.GetFloatValue(INSTANCE_KEY, SIGNAL, HOST, now + ITERATION_SIZE).GetRef(), 12346, 0.001);
        }

        Y_UNIT_TEST(TestRandomlyWrittenSignal) {
            TTempDir dir;
            auto now = TInstant::Seconds(100000);
            TVector<size_t> offsets{5, 27, 68, 122, 240};
            {
                TSnapshotManager snapshotter(dir.Path(), GetLogger());
                TFreshStorage storage(GetLogger(), snapshotter);
                for (auto i : offsets) {
                    storage.PushSignal(INSTANCE_KEY, SIGNAL, HOST, now + ITERATION_SIZE * i, i);
                }
                storage.Shutdown();
            }
            TFreshStorage newStorage(GetLogger());
            newStorage.LoadSnapshots(dir.Path(), now + FRESH_DURATION);
            UNIT_ASSERT_VALUES_EQUAL(newStorage.GetLastFilledDumpStart(),
                                     NormalizeToIntervalUp(now, CHUNK_SIZE) + CHUNK_SIZE * 2);
            for (auto i : offsets) {
                UNIT_ASSERT_EQUAL(newStorage.GetFloatValue(INSTANCE_KEY, SIGNAL, HOST, now + ITERATION_SIZE * i), i);
            }
        }

        Y_UNIT_TEST(TestWriteOnlyNew) {
            TTempDir dir;
            auto now = TInstant::Seconds(100000);

            TVector<TInstant> times;

            for (size_t offset : xrange(0, 10)) {
                times.push_back(now + CHUNK_SIZE * offset);
            }

            {
                TSnapshotManager snapshotter(dir.Path(), GetLogger());
                TFreshStorage storage(GetLogger(), snapshotter);
                for (auto curTime : times) {
                    storage.PushSignal(INSTANCE_KEY, SIGNAL, HOST, curTime, 1);
                    storage.Cleanup(curTime - FRESH_DURATION, curTime - CHUNK_SIZE);
                }
                storage.Shutdown();
            }
            {
                TVector<TString> chunks;
                dir.Path().ListNames(chunks);
                UNIT_ASSERT_VALUES_EQUAL(chunks.size(), times.size());
            }

            NPersistence::TSnapshotReader reader(GetLogger());
            size_t readChunks = 0;
            reader.ReadDirectory(
                    dir.Path(),
                    NormalizeToIntervalDown(times.back() - FRESH_DURATION, CHUNK_SIZE),
                    [&](NPersistence::TDataChunk&&) {
                        ++readChunks;
                    });
            UNIT_ASSERT_VALUES_EQUAL(readChunks, FRESH_DURATION.Seconds() / CHUNK_SIZE.Seconds() + 1);
        }

        Y_UNIT_TEST(TestNormalPersistenceOfConstantlyWrittenSignals) {
            TTempDir dir;
            auto now = TInstant::Seconds(100000);
            {
                TSnapshotManager snapshotter(dir.Path(), GetLogger());
                TFreshStorage storage(GetLogger(), snapshotter);
                for (size_t i = 0; i < 50; i++) {
                    storage.PushSignal(INSTANCE_KEY, SIGNAL, TStringBuf("host1"), now + ITERATION_SIZE * i, i);
                }
                for (size_t i = 100; i < 250; i++) {
                    storage.PushSignal(INSTANCE_KEY, SIGNAL, TStringBuf("host2"), now + ITERATION_SIZE * i, i);
                }
                storage.Shutdown();
            }
            TFreshStorage newStorage(GetLogger());
            newStorage.LoadSnapshots(dir.Path(), now + FRESH_DURATION);
            UNIT_ASSERT_VALUES_EQUAL(newStorage.GetLastFilledDumpStart(),
                                     NormalizeToIntervalUp(now, CHUNK_SIZE) + CHUNK_SIZE * 2);
            for (size_t i = 0; i < 50; i++) {
                UNIT_ASSERT_EQUAL(newStorage.GetFloatValue(INSTANCE_KEY, SIGNAL, TStringBuf("host1"), now + ITERATION_SIZE * i), i);
            }
            for (size_t i = 100; i < 250; i++) {
                UNIT_ASSERT_EQUAL(newStorage.GetFloatValue(INSTANCE_KEY, SIGNAL, TStringBuf("host2"), now + ITERATION_SIZE * i), i);
            }
        }

        Y_UNIT_TEST(TestTailOverwriting) {
            TTempDir dir;
            auto now = TInstant::Seconds(100000);
            {
                TSnapshotManager snapshotter(dir.Path(), GetLogger());
                TFreshStorage storage(GetLogger(), snapshotter);
                for (size_t i = 0; i < 90; i++) {
                    storage.PushSignal(INSTANCE_KEY, SIGNAL, HOST, now + ITERATION_SIZE * i, i);
                }
                // two chunks, 60+30 values
                storage.Shutdown();
            }
            {
                TSnapshotManager snapshotter(dir.Path(), GetLogger());
                TFreshStorage storage(GetLogger(), snapshotter);
                storage.LoadSnapshots(dir.Path(), now + FRESH_DURATION);
                UNIT_ASSERT_VALUES_EQUAL(storage.GetLastFilledDumpStart(),
                                         NormalizeToIntervalDown(now, CHUNK_SIZE));
                for (size_t i = 90; i < 180; i++) {
                    storage.PushSignal(INSTANCE_KEY, SIGNAL, HOST, now + ITERATION_SIZE * i, i);
                }
                // this should overwrite the last chunk and we'll have 60+60+60
                storage.Shutdown();
            }

            TFreshStorage newStorage(GetLogger());
            newStorage.LoadSnapshots(dir.Path(), now + FRESH_DURATION);
            UNIT_ASSERT_VALUES_EQUAL(newStorage.GetLastFilledDumpStart(),
                                     NormalizeToIntervalUp(now, CHUNK_SIZE) + CHUNK_SIZE);
            for (size_t i = 0; i < 180; i++) {
                UNIT_ASSERT_EQUAL(newStorage.GetFloatValue(INSTANCE_KEY, SIGNAL, HOST, now + ITERATION_SIZE * i), i);
            }
        }

        Y_UNIT_TEST(TestOutdatedSeries) {
            TTempDir dir;
            auto now = TInstant::Seconds(100000);
            {
                TSnapshotManager snapshotter(dir.Path(), GetLogger());
                TFreshStorage storage(GetLogger(), snapshotter);
                for (size_t i = 0; i < 90; i++) {
                    storage.PushSignal(INSTANCE_KEY, SIGNAL, HOST, now + ITERATION_SIZE * i, i);
                }
                // two chunks, 60+30 values
                storage.Shutdown();
            }

            now += TDuration::Days(1);
            {
                TSnapshotManager snapshotter(dir.Path(), GetLogger());
                TFreshStorage storage(GetLogger(), snapshotter);
                storage.LoadSnapshots(dir.Path(), now + FRESH_DURATION);
                UNIT_ASSERT_VALUES_EQUAL(storage.GetLastFilledDumpStart(), TInstant::Zero());
                // race between series creation and writing to it
                storage.EmplaceSeries(TInstanceKey::FromNamed("other"), SIGNAL, HOST, ESeriesKind::Double);
                storage.Cleanup(now - FRESH_DURATION, now - CHUNK_SIZE);
                UNIT_ASSERT_VALUES_EQUAL(storage.GetLastFilledDumpStart(),
                                         NormalizeToIntervalDown(now - CHUNK_SIZE, CHUNK_SIZE));
            }
        }
    }
} // namespace NYasmServer
