#pragma once

#include <infra/yasm/common/interval.h>
#include <infra/yasm/server/common/const.h>
#include <infra/yasm/server/common/helpers.h>
#include <infra/yasm/zoom/components/compression/zoom_converters.h>
#include <infra/yasm/zoom/components/compression/bit_stream.h>
#include <infra/yasm/zoom/components/compression/series.h>

#include <infra/yasm/common/points/value/impl.h>
#include <infra/yasm/common/points/value/types.h>

#include <util/datetime/base.h>
#include <util/generic/array_ref.h>
#include <util/generic/bitmap.h>
#include <util/generic/maybe.h>
#include <util/generic/vector.h>
#include <util/system/spinlock.h>

#include <array>

namespace NYasmServer {
    namespace NImpl {
        template <class TDecoder>
        class TChunkedListIterator {
        public:
            TChunkedListIterator(TConstArrayRef<TString> chunks, size_t iterationsInChunk)
                : Chunks(chunks)
                , IterationsInChunk(iterationsInChunk) {
                ShiftToChunk(0);
            }

            void MoveToValue(size_t idx) {
                ShiftToChunk(idx / IterationsInChunk);
                int valuesToSkip = idx % IterationsInChunk;
                for (int i = 0; i < valuesToSkip; i++) {
                    SkipOne();
                }
            }

            TMaybe<typename TDecoder::TValueType> ReadOne() {
                Y_VERIFY(CurrentChunk < Chunks.size());
                bool notNull = ReadFromBitStream(Chunks[CurrentChunk], BitPosition, 1);
                if (!notNull) {
                    OnValueRead();
                    // '0' header, a null value
                    return Nothing();
                }
                auto value = Decoder.Read(Chunks[CurrentChunk], BitPosition);
                OnValueRead();
                return value;
            }

        private:
            void OnValueRead() {
                ReadInCurrentChunk++;
                if (ReadInCurrentChunk == IterationsInChunk) {
                    ShiftToChunk(CurrentChunk + 1);
                }
            }

            void SkipOne() {
                Y_VERIFY(CurrentChunk < Chunks.size());
                bool notNull = ReadFromBitStream(Chunks[CurrentChunk], BitPosition, 1);
                if (notNull) {
                    Decoder.Read(Chunks[CurrentChunk], BitPosition);
                }
                OnValueRead();
            }

            void ShiftToChunk(size_t position) {
                CurrentChunk = position;
                BitPosition = 0;
                ReadInCurrentChunk = 0;
                Decoder.Clear();
            }

        private:
            const TConstArrayRef<TString> Chunks;
            size_t IterationsInChunk;
            TDecoder Decoder;
            size_t BitPosition;
            size_t CurrentChunk;
            size_t ReadInCurrentChunk;
        };
    } // namespace NImpl

    using namespace NYasm::NCommon;

    template <class T>
    inline TAtomicSharedPtr<T> CastSeries(const TSeriesPtr& ptr) {
        return DynamicPointerCast<T, IRecordList>(ptr);
    }

    constexpr TDuration CHUNK_SIZE = TDuration::Minutes(5);
    constexpr size_t ITERATIONS_IN_CHUNK = CHUNK_SIZE / ITERATION_SIZE;

    template <class TEncoder, class TDecoder>
    class TChunkedRecordList : public IRecordList {
    private:
        static const size_t CHUNKS_COUNT = FRESH_DURATION / CHUNK_SIZE + 1;
        static const size_t MAX_VALUES_IN_STREAM = CHUNKS_COUNT * ITERATIONS_IN_CHUNK;

    public:
        using TValueType = typename TDecoder::TValueType;

        TChunkedRecordList() {
            ClearTemporaries();
        }

        TInstant GetStartTime() const override {
            return StartTime;
        }

        TInstant GetEndTime() const override {
            TGuard<TAdaptiveLock> guard{DataLock};
            return GetEndTimeUnsafe();
        }

        void AppendChunk(TInstant start, TString data, size_t valuesCount) override {
            TGuard<TAdaptiveLock> guard{DataLock};
            Y_VERIFY(start.GetValue() % CHUNK_SIZE.GetValue() == 0, "start time must be aligned to 5 minutes");

            if (StartTime == TInstant::Zero()) {
                StartTime = start;
            } else {
                Y_VERIFY(GetEndTimeUnsafe() < start, "chunks must be written sequentally");
                // ensure that we write data sequentially
                WriteNullsUntil(start);
            }

            ClearTemporaries();
            if (valuesCount == ITERATIONS_IN_CHUNK) {
                // full chunk, encoder state doesn't have to be recreated
                Chunks[GetCurrentBufferIdx()] = std::move(data);
                ValuesWritten += ITERATIONS_IN_CHUNK;
                if (ValuesWritten == MAX_VALUES_IN_STREAM) {
                    RotateBuffers();
                }
            } else {
                // rewrite values one-by-one to recreate encoder state
                NImpl::TChunkedListIterator<TDecoder> reader{TConstArrayRef<TString>(&data, 1), ITERATIONS_IN_CHUNK};
                auto ts = start;
                for (size_t i = 0; i < valuesCount; i++) {
                    auto value = reader.ReadOne();
                    if (value.Defined()) {
                        PushValueUnsafe(ts, std::move(*value));
                    }
                    ts += ITERATION_SIZE;
                }
            }
        }

        bool PushValue(TInstant timestamp, TValueType value) {
            Y_VERIFY(timestamp.GetValue() % ITERATION_SIZE.GetValue() == 0);
            TGuard<TAdaptiveLock> guard{DataLock};
            if (timestamp <= GetEndTimeUnsafe()) {
                return false; // don't rewrite past values
            }
            PushValueUnsafe(timestamp, std::move(value));
            return true;
        }

        void IterValues(TInstant from, TInstant to, ISeriesVisitor& visitor) const override {
            Y_VERIFY(from <= to);

            TGuard<TAdaptiveLock> guard{DataLock};
            if (from > GetEndTimeUnsafe() || to < StartTime) {
                visitor.OnHeader(TInstant::Zero(), 0);
                return;
            }

            auto start = Max(from, StartTime);
            auto end = Min(to, GetEndTimeUnsafe());

            size_t offsetValues = (start - StartTime).Seconds() / ITERATION_SIZE.Seconds();
            NImpl::TChunkedListIterator<TDecoder> iter(Chunks, ITERATIONS_IN_CHUNK);
            iter.MoveToValue(offsetValues);

            visitor.OnHeader(start, (end - start) / ITERATION_SIZE + 1);

            for (auto timestamp = start; timestamp <= end; timestamp += ITERATION_SIZE) {
                auto read = iter.ReadOne();
                if (read.Defined()) {
                    // todo: this calls new/delete on every value including floats, which is not very efficient
                    auto val = ToZoom(read.GetRef());
                    visitor.OnValue(val.GetValue());
                } else {
                    visitor.OnValue(NZoom::NValue::TNoneValue::GetSingleton());
                }
            }
        }

        TMaybe<TValueType> GetValueAt(TInstant timestamp) const {
            TGuard<TAdaptiveLock> guard{DataLock};
            if (timestamp < StartTime || timestamp > GetEndTimeUnsafe()) {
                return Nothing();
            }
            size_t offsetValues = (timestamp - StartTime).Seconds() / ITERATION_SIZE.Seconds();
            NImpl::TChunkedListIterator<TDecoder> iter(Chunks, ITERATIONS_IN_CHUNK);
            iter.MoveToValue(offsetValues);
            return iter.ReadOne();
        }

        TMaybe<TSeriesChunk> GetChunkStartingAt(TInstant timestamp, ESnapshotMode mode) const override {
            Y_VERIFY(timestamp.GetValue() % CHUNK_SIZE.GetValue() == 0, "chunks always start at 5 minutes boundary");

            TGuard<TAdaptiveLock> guard{DataLock};
            if (timestamp > GetEndTimeUnsafe() || timestamp < StartTime) {
                return Nothing();
            }
            size_t offset = (timestamp - StartTime) / CHUNK_SIZE;
            TString data = Chunks.at(offset);
            ui32 valuesCount = ITERATIONS_IN_CHUNK;
            if (offset == GetCurrentBufferIdx()) {
                // we're dumping the last chunk
                valuesCount = ValuesWritten % ITERATIONS_IN_CHUNK;
                if (mode == ESnapshotMode::PadNulls) {
                    ui32 bits = BitsWritten;
                    while (valuesCount < ITERATIONS_IN_CHUNK) {
                        AddToBitStream(0, 1, data, bits);
                        valuesCount++;
                    }
                }
            }
            return TSeriesChunk{.StartTime = timestamp, .Data = std::move(data), .ValuesCount = valuesCount};
        }

    private:
        void PushValueUnsafe(TInstant timestamp, TValueType&& value) {
            WriteNullsUntil(timestamp);
            auto& buffer = GetCurrentBuffer();
            // '1' - not a null value
            AddToBitStream(1, 1, buffer, BitsWritten);
            Encoder.Write(std::move(value), buffer, BitsWritten);

            OnValueWritten();
        }

        void AppendNull() {
            // append a single 0 bit
            AddToBitStream(0, 1, GetCurrentBuffer(), BitsWritten);
            OnValueWritten();
        }

        void ClearTemporaries() {
            BitsWritten = 0;
            Encoder.Clear();
        }

        TInstant GetEndTimeUnsafe() const {
            return ValuesWritten == 0 ? StartTime : StartTime + ITERATION_SIZE * (ValuesWritten - 1);
        }

        size_t GetCurrentBufferIdx() const {
            Y_VERIFY(ValuesWritten < MAX_VALUES_IN_STREAM);
            return ValuesWritten / ITERATIONS_IN_CHUNK;
        }

        TString& GetCurrentBuffer() {
            return Chunks[GetCurrentBufferIdx()];
        }

        void WriteNullsUntil(TInstant timestamp) {
            size_t valuesToWrite = (timestamp - GetEndTimeUnsafe() - ITERATION_SIZE) / ITERATION_SIZE;

            if (valuesToWrite >= MAX_VALUES_IN_STREAM || ValuesWritten == 0) {
                // everything is invalid, start from scratch, and align start time to 5 minutes boundary
                StartTime = NInterval::NormalizeToIntervalDown(timestamp, CHUNK_SIZE);
                ValuesWritten = 0;
                for (auto& buffer : Chunks) {
                    buffer.clear();
                }
                ClearTemporaries();
                valuesToWrite = (timestamp - StartTime) / ITERATION_SIZE;
            }

            // push nulls for every missing value
            // todo: this can be optimized nicely for cases when we have a lot of nulls
            // we can write a lot of values (whole chunks) with one memset
            for (size_t i = 0; i < valuesToWrite; i++) {
                AppendNull();
            }
        }

        void OnValueWritten() {
            ValuesWritten++;
            if (ValuesWritten % ITERATIONS_IN_CHUNK == 0) {
                ClearTemporaries();

                if (ValuesWritten == MAX_VALUES_IN_STREAM) {
                    RotateBuffers();
                }
            }
        }

        void RotateBuffers() {
            std::rotate(Chunks.begin(), Chunks.begin() + 1, Chunks.end());
            Chunks.back().clear();
            StartTime += CHUNK_SIZE;
            ValuesWritten -= ITERATIONS_IN_CHUNK;
        }

    private:
        std::array<TString, CHUNKS_COUNT> Chunks;
        // first timestamp we have written
        TInstant StartTime = TInstant::Zero();
        // number of values we have written into the entire stream
        ui32 ValuesWritten = 0;
        // number of bits we have written into the current buffer
        ui32 BitsWritten;
        // value encoder
        TEncoder Encoder;
        // series-level lock
        TAdaptiveLock DataLock;
    }; // namespace NYasmServer

} // namespace NYasmServer
