#include "zoom_writers.h"

#include "storage/counted_sum_list.h"
#include "storage/float_list.h"
#include "storage/histogram_list.h"

#include <util/generic/ymath.h>

namespace NYasmServer {
    namespace {
        template <class T>
        T* TypedSeries(IRecordList* series) {
            return VerifyDynamicCast<T*, IRecordList*>(series);
        }
    }

    inline static void EnsureSeriesKind(const IRecordList* series, ESeriesKind expected) {
        if (series->GetKind() != expected) {
            ythrow yexception() << "Wrong series type: expected " << expected << ", got " << series->GetKind();
        }
    }

    void TZoomValueWriter::MulFloat(const double value) {
        EnsureSeriesKind(Series, ESeriesKind::Double);
        if (TypedSeries<TFloatList>(Series)->PushValue(Timestamp, value)) {
            Accepted = AcceptanceType::ACCEPTED_NON_HGRAM;
        } else {
            Accepted = AcceptanceType::DENIED;
        }
    }

    void TZoomValueWriter::MulVec(const TVector<double>& values) {
        EnsureSeriesKind(Series, ESeriesKind::Histogram);
        TSimpleHistogram hist;
        hist.MutableValues() = values;
        if (TypedSeries<THistogramList>(Series)->PushValue(Timestamp, std::move(hist))) {
            Accepted = AcceptanceType::ACCEPTED_SMALL;
        } else {
            Accepted = AcceptanceType::DENIED;
        }
    }

    void TZoomValueWriter::MulCountedSum(const double sum, const ui64 count) {
        EnsureSeriesKind(Series, ESeriesKind::CountedSum);
        if (TypedSeries<TCountedSumList>(Series)->PushValue(Timestamp, TCountedSum(count, sum))) {
            Accepted = AcceptanceType::ACCEPTED_NON_HGRAM;
        } else {
            Accepted = AcceptanceType::DENIED;
        }
    }

    void TZoomValueWriter::OnStoreSmall(const TArrayRef<const double>& values, size_t zeroes) {
        EnsureSeriesKind(Series, ESeriesKind::Histogram);
        TSimpleHistogram hist;
        hist.SetZeroCount(zeroes);
        auto& targetValues = hist.MutableValues();
        targetValues.reserve(values.size());
        std::copy(values.begin(), values.end(), std::back_inserter(targetValues));
        if (TypedSeries<THistogramList>(Series)->PushValue(Timestamp, std::move(hist))) {
            Accepted = AcceptanceType::ACCEPTED_SMALL;
        } else {
            Accepted = AcceptanceType::DENIED;
        }
    }

    void TZoomValueWriter::OnStoreSmall(const TVector<double>& values, const size_t zeroes) {
        OnStoreSmall(TArrayRef<const double>(values.data(), values.size()), zeroes);
    }

    void TZoomValueWriter::OnStoreNormal(const TArrayRef<const double>& values, const size_t zeros, const i16 startPower) {
        EnsureSeriesKind(Series, ESeriesKind::Histogram);
        TLogHistogram hist;
        hist.SetZeroCount(zeros);
        hist.SetStartPower(startPower);
        hist.MutableWeights().resize(values.size());
        for (size_t i = 0; i < values.size(); i++) {
            hist.MutableWeights()[i] = (ui64)round(values[i]);
        }
        if (TypedSeries<THistogramList>(Series)->PushValue(Timestamp, std::move(hist))) {
            Accepted = AcceptanceType::ACCEPTED_NORMAL;
        } else {
            Accepted = AcceptanceType::DENIED;
        }
    }

    void TZoomValueWriter::OnStoreNormal(const TVector<double>& values, const size_t zeroes, const i16 startPower) {
        OnStoreNormal(TArrayRef<const double>(values.data(), values.size()), zeroes, startPower);
    }

    void TZoomValueWriter::OnStoreUgram(const NZoom::NHgram::TUgramBuckets& buckets) {
        EnsureSeriesKind(Series, ESeriesKind::Histogram);
        TUserHistogram hist;
        if (!buckets.empty()) {
            hist.MutableBuckets().reserve(buckets.size());

            for (size_t i = 0; i < buckets.size() - 1; i++) {
                auto& current = buckets[i];
                hist.AppendBucket(current.LowerBound, (ui64)round(current.Weight));

                if (current.UpperBound != buckets[i + 1].LowerBound) {
                    hist.AppendBucket(current.UpperBound, 0);
                }
            }
            // last bucket
            auto& last = buckets.back();
            hist.AppendBucket(last.LowerBound, (ui64)round(last.Weight));
            if (last.LowerBound != last.UpperBound) {
                hist.AppendBucket(last.UpperBound, 0);
            }
        }

        if (TypedSeries<THistogramList>(Series)->PushValue(Timestamp, std::move(hist))) {
            Accepted = AcceptanceType::ACCEPTED_UGRAM;
        } else {
            Accepted = AcceptanceType::DENIED;
        }
    }

} // namespace NYasmServer
