#include "zoom_writers.h"
#include "storage/counted_sum_list.h"
#include "storage/float_list.h"
#include "storage/histogram_list.h"

#include <infra/yasm/common/points/value/types.h>

#include <library/cpp/testing/unittest/gtest.h>
#include <library/cpp/testing/unittest/registar.h>

namespace NYasmServer {
    using NZoom::NHgram::THgram;
    using NZoom::NValue::TValue;

    Y_UNIT_TEST_SUITE(TestZoomValueWriting) {
        constexpr auto NOW = TInstant::Seconds(10000);

        Y_UNIT_TEST(TestNull) {
            TFloatList series;
            TZoomValueWriter writer(&series, NOW);
            TValue().Update(writer);

            auto read = series.GetValueAt(NOW);
            UNIT_ASSERT(!read.Defined());
        }

        Y_UNIT_TEST(TestDouble) {
            TFloatList series;
            TZoomValueWriter writer(&series, NOW);
            TValue(15.5).Update(writer);

            auto read = series.GetValueAt(NOW);
            UNIT_ASSERT_EQUAL(read, 15.5);
        }

        Y_UNIT_TEST(TestCountedSum) {
            TCountedSumList series;
            TZoomValueWriter writer(&series, NOW);
            TValue(15.5, 25).Update(writer);

            auto read = series.GetValueAt(NOW);
            UNIT_ASSERT_EQUAL(read, TCountedSum(25, 15.5));
        }

        Y_UNIT_TEST(TestSimpleVector) {
            THistogramList series;
            TZoomValueWriter writer(&series, NOW);
            TValue({15.5, 3.5, 21.6}).Update(writer);

            TSimpleHistogram expected;
            expected.MutableValues() = {15.5, 3.5, 21.6};

            auto read = series.GetValueAt(NOW);
            UNIT_ASSERT_EQUAL(read, expected);
        }

        Y_UNIT_TEST(TestLogHistogram) {
            THistogramList series;
            TZoomValueWriter writer(&series, NOW);
            TValue(THgram::Normal({15.2, 3, 21.6}, 15, -5)).Update(writer);

            TLogHistogram expected;
            expected.MutableWeights() = {15, 3, 22}; // we're rounding these
            expected.SetZeroCount(15);
            expected.SetStartPower(-5);

            auto read = series.GetValueAt(NOW);
            UNIT_ASSERT_EQUAL(read, expected);
        }

        Y_UNIT_TEST(TestSimpleHistogram) {
            THistogramList series;
            TZoomValueWriter writer(&series, NOW);
            TValue(THgram::Small({15.2, 3, 21.6}, 15)).Update(writer);

            TSimpleHistogram expected;
            expected.MutableValues() = {15.2, 3, 21.6};
            expected.SetZeroCount(15);

            auto read = series.GetValueAt(NOW);
            UNIT_ASSERT_EQUAL(read, expected);
        }

        Y_UNIT_TEST(TestUserHistogram) {
            THistogramList series;
            TZoomValueWriter writer(&series, NOW);
            TValue(THgram::Ugram({{1, 2, 10},
                                  {4, 5, 20},
                                  {5, 6, 30}}))
                .Update(writer);

            TUserHistogram expected;
            expected.AppendBucket(1, 10);
            expected.AppendBucket(2, 0);
            expected.AppendBucket(4, 20);
            expected.AppendBucket(5, 30);
            expected.AppendBucket(6, 0);

            auto read = series.GetValueAt(NOW);
            UNIT_ASSERT_EQUAL(read, expected);
        }

        Y_UNIT_TEST(TestEmptyUserHistogram) {
            THistogramList series;
            TZoomValueWriter writer(&series, NOW);
            TValue(THgram::EmptyUgram()).Update(writer);

            TUserHistogram expected;

            auto read = series.GetValueAt(NOW);
            UNIT_ASSERT_EQUAL(read, expected);
        }
    }

} // namespace NYasmServer
