#include "lib/storage/fresh.h"
#include "lib/server_monitoring.h"
#include "lib/tsdb_settings.h"

#include <infra/yasm/server/common/helpers.h>
#include <infra/yasm/server/lib/handlers.h>
#include <infra/yasm/server/lib/metrics.h>

#include <infra/monitoring/common/application.h>

using namespace NYasmServer;
using namespace NMonitoring;

class TYasmServerStatsInitializer: public TStatsInitializer {
public:
    void Init(TUnistat& creator) const override {
        TStatsInitializer::Init(creator);

        DrillAmmxHole(creator, NMetrics::GROUPS_INDEX_SIZE);
        DrillAmmxHole(creator, NMetrics::FRESH_SERIES_COUNT);
        DrillAmmxHole(creator, NMetrics::FRESH_TAGS_COUNT);

        DrillSummHole(creator, NMetrics::GROUP_SMALL_HGRAMS);
        DrillSummHole(creator, NMetrics::GROUP_HGRAMS);
        DrillSummHole(creator, NMetrics::HOST_SMALL_HGRAMS);
        DrillSummHole(creator, NMetrics::HOST_HGRAMS);

        DrillSummHole(creator, NMetrics::PUSH_SIGNALS_EMPTY_SIGNALS_COUNT);
        DrillSummHole(creator, NMetrics::PUSH_SIGNALS_EMPTY_TAGS_COUNT);
        DrillSummHole(creator, NMetrics::PUSH_SIGNALS_USELESS_WRITES_COUNT);
        DrillSummHole(creator, NMetrics::PUSH_SIGNALS_USEFUL_WRITES_COUNT);
        DrillSummHole(creator, NMetrics::PUSH_SIGNALS_ACCEPTED_VALUES_COUNT);
        DrillSummHole(creator, NMetrics::PUSH_SIGNALS_IGNORED_VALUES_COUNT);
        DrillSummHole(creator, NMetrics::SNAPSHOTTER_REMOVED_FILES_COUNT);
        DrillSummHole(creator, NMetrics::SNAPSHOTTER_SUBMITTED_CHUNKS_COUNT);

        DrillSummHole(creator, NMetrics::FETCH_DATA_SERIES_READ);
        DrillSummHole(creator, NMetrics::READ_AGGREGATED_SERIES_READ);

        DrillHistogramHole(creator, NMetrics::GROUPS_INDEX_CLEANUP_TIME);
        DrillHistogramHole(creator, NMetrics::FRESH_CLEANUP_TIME);
        DrillHistogramHole(creator, NMetrics::FRESH_LOADING_TIME);
        DrillHistogramHole(creator, NMetrics::FETCH_HOSTS_RESPONSE_TIME);
        DrillHistogramHole(creator, NMetrics::FETCH_DATA_RESPONSE_TIME);
        DrillHistogramHole(creator, NMetrics::READ_AGGREGATED_RESPONSE_TIME);
        DrillHistogramHole(creator, NMetrics::PUSH_SIGNALS_RESPONSE_TIME);
        DrillHistogramHole(creator, NMetrics::PUSH_SIGNALS_REQUEST_SIZE);
        DrillHistogramHole(creator, NMetrics::LIST_ALL_SIGNALS_RESPONSE_TIME);
        DrillHistogramHole(creator, NMetrics::SNAPSHOTTER_ITERATION_TIME);
        DrillHistogramHole(creator, NMetrics::SNAPSHOTTER_SNAPSHOT_SIZE_BYTES);
        DrillHistogramHole(creator, NMetrics::SNAPSHOTTER_SNAPSHOT_SIZE_CHUNKS);
    }
};

class TYasmServer: public TBaseApplication<TTsdbSettings> {
public:
    TYasmServer(TLog& logger, const TTsdbSettings& settings)
        : TBaseApplication(logger, settings)
    {
        auto& snapshotsDir = settings.GetSnapshotDirectory();
        if (snapshotsDir) {
            Logger << "Persistence enabled, snapshots will be written to " << snapshotsDir;
            Snapshotter = MakeHolder<NPersistence::TSnapshotManager>(snapshotsDir, Logger);
        } else {
            Logger << TLOG_WARNING << "Persistence disabled!";
            Snapshotter = MakeHolder<NPersistence::TDevNullManager>();
        }
        ServerMonitoring = MakeHolder<TServerMonitoring>(Logger);
        Fresh = MakeHolder<TFreshStorage>(Logger, *Snapshotter);
        if (snapshotsDir) {
            Fresh->LoadSnapshots(snapshotsDir, TInstant::Now());
        }
        HttpHandlers = MakeHolder<TFreshHandlersCollection>(*Fresh, Logger);
    }

    void Run() override {
        Logger << "Run yasm server";
        ServerMonitoring->Start();
        Fresh->Start();
        HttpHandlers->Register(HttpServer);
        HttpServer.StartServing();
        Event.Wait();
    }

    void Stop(int signal) override {
        TBaseApplication::Stop(signal);
        if (Fresh) {
            Fresh->Shutdown();
        }
        if (ServerMonitoring) {
            ServerMonitoring->ShutDown();
        }
    }

    static void FillParser(NLastGetopt::TOpts& options) {
        options.AddLongOption("snapshot-dir", "directory with snapshot files").Optional().RequiredArgument().DefaultValue("");
    }

    static void ProcessParsedOptions(const NLastGetopt::TOptsParseResult& parsed, TTsdbSettings& settings) {
        if (parsed.Has("snapshot-dir")) {
            settings.SetSnapshotDirectory(parsed.Get("snapshot-dir"));
        }

        settings.SetEnableCompression(true);
        settings.SetClientTimeout(TDuration::Seconds(10));
    }

    THolder<TStatsInitializer> GetStatsInitializer() override {
        return MakeHolder<TYasmServerStatsInitializer>();
    }

private:
    THolder<NPersistence::ISnapshotManager> Snapshotter;
    THolder<TServerMonitoring> ServerMonitoring;
    THolder<TFreshStorage> Fresh;
    THolder<TFreshHandlersCollection> HttpHandlers;
};

int main(int argc, const char** argv) {
    return RunFromMain<TYasmServer, TTsdbSettings, TThreadedLoggerFactory>(argc, argv);
}
