#include "reader.h"
#include "streams.h"
#include "impl.h"

#include <infra/yasm/common/labels/tags/instance_key.h>
#include <infra/yasm/common/labels/signal/signal_name.h>

#include <library/cpp/blockcodecs/codecs.h>
#include <library/cpp/blockcodecs/stream.h>
#include <library/cpp/threading/local_executor/local_executor.h>

#include <util/ysaveload.h>
#include <util/system/file.h>
#include <util/stream/file.h>
#include <util/generic/hash.h>

namespace NYasmServer {
    namespace NPersistence {
        using NTags::TInstanceKey;
        using NZoom::NSignal::TSignalName;

        void ValidateCrc(TFile& file) {
            TFileInput input(file);
            ui32 crc = 0;
            try {
                input.LoadOrFail(&crc, sizeof(crc));
            } catch (yexception &) {
                ythrow TCheckSumAbsent() << "Can't read checksum from file " << file.GetName();
            }
            TBufferedCrc32Output crcStream;
            input.ReadAll(crcStream);
            crcStream.Finish();
            if (crcStream.GetCrc() != crc) {
                ythrow TCheckSumMismatchError() << "Crc of file " << file.GetName() << " didn't match: "
                                                << crcStream.GetCrc() << " vs " << crc << " expected";
            }
            file.Seek(sizeof(ui32), SeekDir::sSet);
        }

        template <class T>
        TVector<T> ReadIndex(IInputStream* stream, const std::function<T(const TString&)>& parser) {
            TVector<T> out;
            out.reserve(LoadSize(stream));
            for (size_t idx = 0; idx < out.capacity(); idx++) {
                TString loaded;
                ::Load(stream, loaded);
                out.emplace_back(parser(loaded));
            }
            return out;
        }

        TSnapshotReader::TSnapshotReader(TLog& logger)
            : Logger(logger) {
        }

        bool TParsedSnapshotName::operator <(const TParsedSnapshotName& other) const noexcept {
            return Timestamp < other.Timestamp || (Timestamp == other.Timestamp && Part < other.Part);
        }

        bool TParsedSnapshotName::operator <=(const TParsedSnapshotName& other) const noexcept {
            return Timestamp < other.Timestamp || (Timestamp == other.Timestamp && Part <= other.Part);
        }

        void TSnapshotReader::ReadFileCore(const TFsPath& path, const TSnapshotCallback& callback) {
            TFile file(path, OpenExisting | RdOnly | Seq);
            ValidateCrc(file);

            TFileInput input(file);
            NBlockCodecs::TDecodedInput stream(&input);

            ui16 fileVersion;
            ::Load(&stream, fileVersion);

            if (fileVersion != FILE_VERSION) {
                ythrow TFileVersionMismatchError() << "Version of file " << path << " didn't match: "
                                                   << fileVersion << " vs " << FILE_VERSION << " expected";
            }

            auto tagIndex = ReadIndex<TInstanceKey>(&stream, [](const TString& key) { return TInstanceKey::FromNamed(key); });
            auto signalIndex = ReadIndex<TSignalName>(&stream, [](const TString& signal) { return TSignalName(signal); });
            auto hostIndex = ReadIndex<TString>(&stream, [](const TString& host) { return host; });

            size_t count = ::LoadSize(&stream);
            for (size_t i = 0; i < count; i++) {
                NImpl::TChunk chunk;
                ::Load(&stream, chunk);

                auto bytes = chunk.Data.size();
                callback(TDataChunk{
                    .Key = tagIndex.at(chunk.Header.Key),
                    .Signal = signalIndex.at(chunk.Header.Signal),
                    .Host = hostIndex.at(chunk.Header.Host),
                    .Start = TInstant::Seconds(chunk.Header.Start),
                    .Kind = (ESeriesKind)chunk.Header.Kind,
                    .Data = std::move(chunk.Data),
                    .ValuesCount = chunk.Header.ValuesCount,
                });
                Metrics.ReadMetrics->Inc();
                Metrics.ReadPoints->Add(chunk.Header.ValuesCount);
                Metrics.ReadBytes->Add(bytes);
                TDuration metricAge = TInstant::Now() - TInstant::Seconds(chunk.Header.Start);
                Metrics.MetricAgeMs->Set(metricAge.MilliSeconds());
                Metrics.MetricAgeHistMin->Record(metricAge.Minutes());
            }
        }

        void TSnapshotReader::ReadFile(const TFsPath& path, const TSnapshotCallback& callback,
                                       EErrorMode errorMode) {
            Logger << "Loading file " << path;
            try {
                ReadFileCore(path, callback);
                Metrics.ReadFiles->Inc();
            } catch (...) {
                if (errorMode == EErrorMode::Throw) {
                    throw;
                } else {
                    Logger << TLOG_ERR << "Error while loading file " << path << ": " << CurrentExceptionMessage();
                }
            }
        }

        void TSnapshotReader::NothingToRead() const {
            Metrics.MetricAgeMs->Set(0);
        }

        TVector<TParsedSnapshotName> TSnapshotReader::ListDirectory(const TFsPath& path) {
            TVector<TString> fileNames;
            path.ListNames(fileNames);
            // ensure oldest chunks are read first
            TVector<TParsedSnapshotName> parsed;

            for (const auto& name : fileNames) {
                TStringBuf baseName, extension;
                TStringBuf(name).RSplit('.', baseName, extension);
                if (extension != TStringBuf(FILE_EXTENSION).SubStr(1)) {
                    continue;
                }
                TStringBuf timestampString, partString;
                baseName.Split('-', timestampString, partString);
                ui64 ts, part;
                // read all chunks that have proper filenames
                if (TryFromString(timestampString, ts) && TryFromString(partString, part)) {
                    parsed.push_back(TParsedSnapshotName{.Timestamp = TInstant::Seconds(ts), .Part = part, .Path = path / name});
                }
            }
            Sort(parsed);

            return parsed;
        }

        void TSnapshotReader::ReadDirectory(const TFsPath& path, TInstant minStart, const TSnapshotCallback& callback, size_t threads,
                                            EErrorMode errorMode) {
            TInstant lastTimestamp = TInstant::Max();
            TVector<TVector<TParsedSnapshotName>> groups;

            for (auto p : ListDirectory(path)) {
                if (p.Timestamp < minStart) {
                    continue;
                }
                if (p.Timestamp != lastTimestamp) {
                    groups.emplace_back();
                    lastTimestamp = p.Timestamp;
                }
                groups.back().emplace_back(std::move(p));
            }

            NPar::TLocalExecutor executor;
            executor.RunAdditionalThreads(threads - 1);

            for (const auto& group : groups) {
                auto flags = NPar::TLocalExecutor::WAIT_COMPLETE | NPar::TLocalExecutor::HIGH_PRIORITY;
                executor.ExecRange([&](int position) {
                    ReadFile(group.at(position).Path, callback, errorMode);
                },
                                   NPar::TLocalExecutor::TExecRangeParams(0, group.size()), flags);
            }
        }
    } // namespace NPersistence
} // namespace NYasmServer
