#pragma once
#include "common.h"

#include <library/cpp/logger/global/global.h>
#include <library/cpp/monlib/metrics/metric_registry.h>
#include <library/cpp/monlib/metrics/metric.h>
#include <util/folder/path.h>

namespace NYasmServer {
    namespace NPersistence {
        class TSnapshotError : public yexception {
        };

        class TExpectedSnapshotError : public TSnapshotError {
        };

        class TCheckSumAbsent : public TExpectedSnapshotError {
        };

        class TCheckSumMismatchError : public TExpectedSnapshotError {
        };

        class TFileVersionMismatchError : public TSnapshotError {
        };

        enum class EErrorMode {
            Ignore,
            Throw
        };

        using TSnapshotCallback = std::function<void(TDataChunk&&)>;

        struct TParsedSnapshotName {
            TInstant Timestamp;
            size_t Part;
            TFsPath Path;

            bool operator <(const TParsedSnapshotName& other) const noexcept;
            bool operator <=(const TParsedSnapshotName& other) const noexcept;
        };

        struct TReaderMetrics {
        public:
            TReaderMetrics() {
                auto* registry = NMonitoring::TMetricRegistry::Instance();
                ReadFiles = registry->Rate({{"sensor", "reader.read.files"}});
                ReadMetrics = registry->Rate({{"sensor", "reader.read.metrics"}});
                ReadPoints = registry->Rate({{"sensor", "reader.read.points"}});
                ReadBytes = registry->Rate({{"sensor", "reader.read.bytes"}});
                MetricAgeMs = registry->IntGauge({{"sensor", "reader.metric.ageMs"}});
                MetricAgeHistMin = registry->HistogramRate(
                    {{"sensor", "reader.metric.hist.ageMin"}},
                    NMonitoring::ExponentialHistogram(30, 2, 1));
            }

        public:
            NMonitoring::IRate* ReadFiles{};
            NMonitoring::IRate* ReadMetrics{};
            NMonitoring::IRate* ReadPoints{};
            NMonitoring::IRate* ReadBytes{};
            NMonitoring::IIntGauge* MetricAgeMs{};
            NMonitoring::THistogram* MetricAgeHistMin{};
        };

        class TSnapshotReader {
        public:
            TSnapshotReader(TLog& logger = TLoggerOperator<TNullLog>::Log());

            void ReadFile(const TFsPath& path, const TSnapshotCallback& callback,
                          EErrorMode errorMode = EErrorMode::Ignore);

            void NothingToRead() const;

            void ReadDirectory(const TFsPath& path, TInstant minStart, const TSnapshotCallback& callback, size_t threads = 1,
                               EErrorMode errorMode = EErrorMode::Ignore);

            TVector<TParsedSnapshotName> ListDirectory(const TFsPath& path);

        private:
            void ReadFileCore(const TFsPath& path, const TSnapshotCallback& callback);

        private:
            TLog& Logger;
            TReaderMetrics Metrics;
        };
    } // namespace NPersistence
} // namespace NYasmServer
