#include "snapshot_manager.h"

#include "common.h"
#include "streams.h"

#include <infra/yasm/common/interval.h>

#include <infra/yasm/server/lib/metrics.h>

#include <infra/monitoring/common/perf.h>

#include <library/cpp/blockcodecs/codecs.h>
#include <library/cpp/blockcodecs/stream.h>

#include <util/ysaveload.h>
#include <util/generic/algorithm.h>
#include <util/generic/hash.h>
#include <util/generic/vector.h>
#include <util/stream/file.h>
#include <util/stream/fwd.h>
#include <util/stream/tee.h>
#include <util/string/cast.h>
#include <util/system/file.h>
#include <util/system/fs.h>

namespace NYasmServer {
    namespace NPersistence {
        using NTags::TInstanceKey;
        using NZoom::NSignal::TSignalName;

        namespace {
            struct TSnapshotBuffer {
                void Append(const TDataChunk& chunk) {
                    Chunks.push_back(NImpl::TChunk{
                        .Header = NImpl::TDataChunkHeader{
                            .Key = Emplace(InstanceKeys, chunk.Key),
                            .Signal = Emplace(Signals, chunk.Signal),
                            .Host = Emplace(Hosts, chunk.Host),
                            .Start = (ui32)chunk.Start.Seconds(),
                            .ValuesCount = chunk.ValuesCount,
                            .Kind = (ui8)chunk.Kind,
                        },
                        .Data = chunk.Data,
                    });
                }

                template <class T>
                ui32 Emplace(THashMap<T, ui32>& index, T value) {
                    typename THashMap<T, ui32>::insert_ctx ctx;
                    auto it = index.find(value, ctx);
                    if (it.IsEnd()) {
                        it = index.emplace_direct(ctx, value, index.size());
                    }
                    return it->second;
                }

                void Clear() {
                    InstanceKeys.clear();
                    Signals.clear();
                    Hosts.clear();
                    Chunks.clear();
                }

                THashMap<NTags::TInstanceKey, ui32> InstanceKeys;
                THashMap<NZoom::NSignal::TSignalName, ui32> Signals;
                THashMap<TString, ui32> Hosts;
                TVector<NImpl::TChunk> Chunks;
            };

            template <class T>
            inline void SaveIndex(IOutputStream* stream,
                                  const THashMap<T, ui32>& index,
                                  const std::function<TString(T)>& stringFunction) {
                TVector<TString> data;
                data.resize(index.size());
                for (const auto& pair : index) {
                    data[pair.second] = stringFunction(pair.first);
                }
                ::Save(stream, data);
            }

            inline size_t GetFileSize(const TFsPath& path) {
                TFileStat stat;
                path.Stat(stat);
                return stat.Size;
            }

            void WriteChunks(const TSnapshotBuffer& buffer, const TFsPath& filePath) {
                TBufferedCrc32Output crcOutput;
                TFile file(filePath, CreateNew | WrOnly | Seq);
                {
                    TFileOutput fileOutput(file);
                    // leave some space for crc
                    ::Save<ui32>(&fileOutput, 0);

                    TTeeOutput teeOutput{&crcOutput, &fileOutput};
                    // both crc stream and file get already compressed output so we don't need to decompress it to verify integrity
                    NBlockCodecs::TCodedOutput out(&teeOutput, NBlockCodecs::Codec("zstd06_1"), 1024 * 1024);

                    ::Save<ui16>(&out, FILE_VERSION);

                    // indexes
                    SaveIndex<TInstanceKey>(&out, buffer.InstanceKeys, [](TInstanceKey key) { return key.ToNamed(); });
                    SaveIndex<TSignalName>(&out, buffer.Signals, [](TSignalName signal) { return signal.GetName(); });
                    SaveIndex<TString>(&out, buffer.Hosts, [](TString host) { return host; });

                    // actual data
                    ::SaveSize(&out, buffer.Chunks.size());
                    for (const auto& chunk : buffer.Chunks) {
                        ::Save(&out, chunk);
                    }
                    out.Finish();
                    teeOutput.Finish();
                }
                // write crc to the beginning of the file
                ui32 crc = crcOutput.GetCrc();
                file.Seek(0, SeekDir::sSet);
                file.Write(&crc, sizeof(crc));
            }

            void WriteChunksThroughTemp(const TSnapshotBuffer& buffer, const TFsPath& filePath, const TFsPath& tempFilePath) {
                tempFilePath.DeleteIfExists();
                WriteChunks(buffer, tempFilePath);
                tempFilePath.RenameTo(filePath);
            }

        } // namespace

        TDevNullManager& TDevNullManager::GetSingleton() {
            static TDevNullManager manager;
            return manager;
        }

        TSnapshotManager::TSnapshotManager(const TFsPath& directory,
                                           TLog& logger,
                                           size_t maxBufferSize,
                                           size_t maxBufferCount)
            : Directory(directory)
            , Logger(logger)
            , MaxBufferSize(maxBufferSize)
            , MaxBufferCount(maxBufferCount) {
        }

        void TSnapshotManager::Cleanup(TInstant deadline) {
            NMonitoring::TMeasuredMethod perf(Logger, "snapshot cleanup");
            Logger << "Starting snapshots cleanup";
            TVector<TString> fileNames;
            Directory.ListNames(fileNames);

            // remove outdated files
            for (const auto& name : fileNames) {
                TStringBuf baseName, extension;
                TStringBuf(name).RSplit('.', baseName, extension);

                if (extension != TStringBuf(FILE_EXTENSION).SubStr(1)) {
                    Logger << TLOG_DEBUG << "Snapshotter ignoring file " << name << " because of wrong extension";
                    continue;
                }

                auto timestampPart = baseName.Before('-');
                ui64 seconds;
                if (!TryFromString(timestampPart, seconds)) {
                    Logger << TLOG_WARNING << "Snapshotter failed to parse filename " << timestampPart;
                    continue;
                }
                auto filePath = Directory / name;
                if (TInstant::Seconds(seconds) <= deadline) {
                    NFs::Remove(filePath);
                    TUnistat::Instance().PushSignalUnsafe(NMetrics::SNAPSHOTTER_REMOVED_FILES_COUNT, 1);
                }
            }
        }

        void TSnapshotManager::WriteChunks(const TVector<TDataChunk>& chunks, const TString& baseFileName) {
            TString name = baseFileName;
            if (name.empty()) {
                name = ToString(NYasm::NCommon::NInterval::NormalizeToIntervalDown(TInstant::Now(), TDuration::Seconds(1)));
            }
            DumpChunks(chunks, name, "chunks.temp");
        }

        void TSnapshotManager::DumpChunks(const TVector<TDataChunk>& chunks, const TString& baseFileName, const TString& tempFileName) {
            NMonitoring::TMeasuredMethod perf(Logger, "writing snapshots", NMetrics::SNAPSHOTTER_ITERATION_TIME);

            Logger << "Snapshotter writing " << chunks.size() << " chunks";
            auto maxCount = MaxBufferSize * MaxBufferCount;
            if (chunks.size() > maxCount) {
                ythrow yexception() << "Too many (" << chunks.size() << ") chunks submitted to persistence, allowed" << maxCount;
            }
            TUnistat::Instance().PushSignalUnsafe(NMetrics::SNAPSHOTTER_SUBMITTED_CHUNKS_COUNT, chunks.size());

            // group chunks by tag, we don't care about the actual order
            THashMap<TInstanceKey, TVector<const TDataChunk*>> groupedChunks;
            for (const auto& chunk : chunks) {
                groupedChunks[chunk.Key].push_back(&chunk);
            }

            TSnapshotBuffer buffer;
            size_t bufferId = 0;
            size_t seenChunks = 0;
            for (const auto& pair : groupedChunks) {
                for (auto chunk : pair.second) {
                    seenChunks++;
                    buffer.Append(*chunk);
                    if (buffer.Chunks.size() == MaxBufferSize || seenChunks == chunks.size()) {
                        // chunk done, dump it to disk
                        auto filePath = Directory / (baseFileName + "-" + ToString(bufferId) + FILE_EXTENSION);
                        auto tempFilePath = Directory / tempFileName;
                        WriteChunksThroughTemp(buffer, filePath, tempFilePath);

                        TUnistat::Instance().PushSignalUnsafe(NMetrics::SNAPSHOTTER_SNAPSHOT_SIZE_BYTES, GetFileSize(filePath));
                        TUnistat::Instance().PushSignalUnsafe(NMetrics::SNAPSHOTTER_SNAPSHOT_SIZE_CHUNKS, buffer.Chunks.size());

                        // prepare for next chunk
                        bufferId++;
                        buffer.Clear();
                    }
                }
            }
        }
    } // namespace NPersistence
} // namespace NYasmServer
