#include "snapshot_manager.h"
#include "reader.h"

#include <infra/monitoring/common/collections.h>
#include <infra/monitoring/common/test_helpers.h>

#include <library/cpp/testing/unittest/gtest.h>
#include <library/cpp/testing/unittest/registar.h>

#include <util/folder/path.h>
#include <util/folder/tempdir.h>
#include <util/system/file.h>

namespace NYasmServer {
    namespace NPersistence {
        using NTags::TInstanceKey;
        using NZoom::NSignal::TSignalName;

        Y_UNIT_TEST_SUITE(TestSnapshotManager) {
            Y_UNIT_TEST(TestMaximumQueueSize) {
                TTempDir dir;
                TSnapshotManager manager(dir.Path(), GetLogger(), 2, 2);
                TSignalName signal(TStringBuf("signal"));
                auto instanceKey = TInstanceKey::FromNamed("base");
                TVector<TDataChunk> chunks;

                // 4 elements max, we're trying to write 5
                for (size_t i = 0; i < 5; i++) {
                    chunks.push_back(TDataChunk{.Key = instanceKey, .Signal = signal, .Host = ToString(i)});
                }
                UNIT_ASSERT_EXCEPTION(manager.WriteChunks(chunks), yexception);
            }

            inline static size_t VerifyNotEmpty(const TFsPath& path) {
                UNIT_ASSERT(path.Exists());
                TFileStat stat;
                path.Stat(stat);
                UNIT_ASSERT(stat.Size > 0);
                return stat.Size;
            }

            Y_UNIT_TEST(TestWritingOfMultipleBuffers) {
                TTempDir dir;
                TSnapshotManager manager(dir.Path(), GetLogger(), 2, 2);
                TVector<TDataChunk> chunks;
                for (size_t i = 0; i < 3; i++) {
                    chunks.push_back(TDataChunk{
                        .Key = TInstanceKey::FromNamed("base"),
                        .Signal = TSignalName(TStringBuf("signal")),
                        .Host = ToString(i),
                        .Data = "some random stuff"});
                }
                manager.WriteChunks(chunks, "file");

                auto size1 = VerifyNotEmpty(dir.Path() / "file-0.chunks");
                auto size2 = VerifyNotEmpty(dir.Path() / "file-1.chunks");
                UNIT_ASSERT_C(size1 > size2, "first file should have two chunks, second - one");
            }

            Y_UNIT_TEST(TestCleanup) {
                TTempDir dir;
                TSnapshotManager manager(dir.Path(), GetLogger(), 2, 2);
                TVector<TDataChunk> chunks;
                for (size_t i = 0; i < 3; i++) {
                    chunks.push_back(TDataChunk{
                        .Key = TInstanceKey::FromNamed("base"),
                        .Signal = TSignalName(TStringBuf("signal")),
                        .Host = ToString(i),
                        .Data = "some random stuff"});
                }

                auto name1 = ToString((TInstant::Now() - TDuration::Hours(24)).Seconds());
                manager.WriteChunks(chunks, name1);

                auto name2 = ToString(TInstant::Now().Seconds());
                manager.WriteChunks(chunks, name2);

                // everything is here
                UNIT_ASSERT((dir.Path() / (name1 + "-0.chunks")).Exists());
                UNIT_ASSERT((dir.Path() / (name1 + "-1.chunks")).Exists());
                UNIT_ASSERT((dir.Path() / (name2 + "-0.chunks")).Exists());
                UNIT_ASSERT((dir.Path() / (name2 + "-1.chunks")).Exists());

                manager.Cleanup(TInstant::Now() - TDuration::Hours(1));
                // now both chunks of the "older" file is missing
                UNIT_ASSERT(!(dir.Path() / (name1 + "-0.chunks")).Exists());
                UNIT_ASSERT(!(dir.Path() / (name1 + "-1.chunks")).Exists());
                UNIT_ASSERT((dir.Path() / (name2 + "-0.chunks")).Exists());
                UNIT_ASSERT((dir.Path() / (name2 + "-1.chunks")).Exists());
            }

            Y_UNIT_TEST(TestBasicWritingAndReading) {
                TTempDir dir;
                TSnapshotManager manager(dir.Path(), GetLogger(), 2, 2);
                TDataChunk chunk1{
                    .Key = TInstanceKey::FromNamed("base"),
                    .Signal = TSignalName(TStringBuf("signal1")),
                    .Host = "host1",
                    .Start = TInstant::Seconds(12345),
                    .Kind = ESeriesKind::CountedSum,
                    .Data = "some random stuff",
                };
                TDataChunk chunk2{
                    .Key = TInstanceKey::FromNamed("news"),
                    .Signal = TSignalName(TStringBuf("signal2")),
                    .Host = "host2",
                    .Start = TInstant::Seconds(54321),
                    .Kind = ESeriesKind::Histogram,
                    .Data = "other stuff",
                };

                THashMap<TInstanceKey, TDataChunk> chunks{{chunk1.Key, chunk1}, {chunk2.Key, chunk2}};
                manager.WriteChunks(NMonitoring::MapValues(chunks), "file");

                TSnapshotReader reader;
                reader.ReadFile(dir.Path() / "file-0.chunks", [&](TDataChunk&& chunk) {
                    auto& expected = chunks.at(chunk.Key);

                    UNIT_ASSERT_EQUAL(expected.Signal, chunk.Signal);
                    UNIT_ASSERT_EQUAL(expected.Host, chunk.Host);
                    UNIT_ASSERT_EQUAL(expected.Start, chunk.Start);
                    UNIT_ASSERT_EQUAL(expected.Kind, chunk.Kind);
                    UNIT_ASSERT_EQUAL(expected.Data, chunk.Data);
                });
            }

            Y_UNIT_TEST(TestCrcValidationOnLargeFiles) {
                TTempDir dir;
                const size_t count = 1000000;
                TSnapshotManager manager(dir.Path(), GetLogger(), count);
                TVector<TDataChunk> chunks;
                chunks.reserve(count);
                for (size_t i = 0; i < count; i++) {
                    chunks.push_back(TDataChunk{
                        .Key = TInstanceKey::FromNamed("base"),
                        .Signal = TSignalName(TStringBuf("signal")),
                        .Host = ToString(i),
                    });
                }
                manager.WriteChunks(chunks, "file");

                TFileStat stat;
                (dir.Path() / "file-0.chunks").Stat(stat);
                // ensure that file is sufficiently large
                UNIT_ASSERT((stat.Size - sizeof(ui32)) > CRC_BUFFER_LENGTH * 5);
                // and we don't fit into the buffer properly
                UNIT_ASSERT((stat.Size - sizeof(ui32)) % CRC_BUFFER_LENGTH != 0);

                TSnapshotReader reader;
                UNIT_ASSERT_NO_EXCEPTION(reader.ReadFile(dir.Path() / "file-0.chunks", [](TDataChunk&&) {}));
            }

            Y_UNIT_TEST(TestReadingInvalidCrc) {
                TTempDir dir;
                TSnapshotManager manager(dir.Path(), GetLogger(), 2, 2);
                auto chunk = TDataChunk{
                    .Key = TInstanceKey::FromNamed("base"),
                    .Signal = TSignalName(TStringBuf("signal1")),
                    .Host = "host1",
                    .Data = "some random stuff"};

                manager.WriteChunks({chunk}, "file");
                {
                    TFile file(dir.Path() / "file-0.chunks", OpenExisting | WrOnly);
                    ui32 crc = 0;
                    file.Write(&crc, sizeof(crc));
                }
                {
                    TSnapshotReader reader{GetLogger()};
                    // errors enabled
                    UNIT_ASSERT_EXCEPTION(reader.ReadFile(dir.Path() / "file-0.chunks", [](TDataChunk&&) {}, EErrorMode::Throw), TCheckSumMismatchError);
                }
                {
                    TSnapshotReader reader{GetLogger()};
                    // errors disabled
                    UNIT_ASSERT_NO_EXCEPTION(reader.ReadFile(dir.Path() / "file-0.chunks", [](TDataChunk&&) {}, EErrorMode::Ignore));
                }
            }
        }
    } // namespace NPersistence
} // namespace NYasmServer
