#include "cluster.h"

#include <solomon/libs/cpp/grpc/interceptor/call_interceptor.h>

#include <library/cpp/json/json_value.h>
#include <library/cpp/json/json_reader.h>
#include <library/cpp/monlib/metrics/metric_registry.h>

#include <util/random/random.h>

using namespace NHistDb::NStockpile;
using namespace NMonitoring;

namespace {
    struct TClusterData {
        TVector<THostAndClusterName> Hosts;
        TStockpilePort Port;
    };

    TClusterData ExtractSolomonDataByCluster(const TString& jsonData, const TMaybe<TString>& clusterName, TLog& log) {
        NJson::TJsonValue json;
        NJson::ReadJsonTree(jsonData, &json);
        const auto& jsonMap = json.GetMapSafe();

        TClusterData result;

        result.Port = jsonMap.at(TStringBuf("ports")).GetMapSafe().at(TStringBuf("grpc")).GetUIntegerSafe();

        const auto& hosts = jsonMap.at(TStringBuf("hosts")).GetArraySafe();
        for (const auto& host : hosts) {
            try {
                const auto& hostInfo = host.GetMapSafe();
                const auto& curClusterName = hostInfo.at(TStringBuf("cluster")).GetStringSafe();
                if (!clusterName.Defined() || curClusterName == clusterName.GetRef()) {
                    result.Hosts.emplace_back(hostInfo.at(TStringBuf("fqdn")).GetStringSafe(), curClusterName);
                }
            } catch (const NJson::TJsonException& e) {
                log << TLOG_ERR << "Failed to parse host info: " << e.what();
                throw;
            }
        }

        return result;
    }
}

std::pair<TString, bool> NImpl::TFetcher::operator()(const TString& url, TDuration timeout) const {
    auto options = NHttp::TFetchOptions().SetTimeout(timeout);
    auto reply = NHttp::Fetch(NHttp::TFetchQuery(url, options));
    std::pair<TString, bool> result;
    if (reply->Success()) {
        result.first = reply->Data;
        result.second = true;
    } else {
        result.first = TStringBuilder() << "[" << reply->Code << "]: " << reply->Data;
        result.second = false;
    }
    return result;
}

TDuration TClusterProvider::GetUpdateJitter() {
    return TDuration::FromValue(RandomNumber(CLUSTER_UPDATE_JITTER.GetValue()));
}

TString TClusterProvider::GetClusterTypeName(EStockpileClusterType clusterType) {
    switch (clusterType) {
        case EStockpileClusterType::Production: {
            return TString("PRODUCTION");
        }
        case EStockpileClusterType::Prestable: {
            return TString("PRESTABLE");
        }
        case EStockpileClusterType::Testing: {
            return TString("TESTING");
        }
    }
    Y_UNREACHABLE();
}

TString TClusterProvider::GetDatabaseName(EStockpileDatabase database) {
    switch (database) {
        case EStockpileDatabase::Stockpile: {
            return TString("stockpile");
        }
        case EStockpileDatabase::Metabase: {
            return TString("metabase");
        }
        case EStockpileDatabase::DataProxy: {
            return TString("dataproxy");
        }
    }
    Y_UNREACHABLE();
}

TClusterProvider::TClusterProvider(EStockpileDatabase database, const TClusterInfo& clusterInfo, TLog& log)
    : Database(database)
    , ClusterType(clusterInfo.ClusterType)
    , ClusterName(clusterInfo.ClusterName)
    , GrpcRemoteHosts()
    , Log(log)
{
}

TClusterProvider::TClusterProvider(EStockpileDatabase database, EStockpileClusterType clusterType, TLog& log)
    : Database(database)
    , ClusterType(clusterType)
    , GrpcRemoteHosts()
    , Log(log)
{
}

TVector<TAtomicSharedPtr<TGrpcRemoteHost>> TClusterProvider::GetHosts() const {
    TLightReadGuard guard(Lock);
    TVector<TAtomicSharedPtr<TGrpcRemoteHost>> result(Reserve(GrpcRemoteHosts.size()));
    for (const auto& remoteHostEntry: GrpcRemoteHosts) {
        result.emplace_back(remoteHostEntry.second);
    }
    return result;
};

TVector<TVector<TAtomicSharedPtr<TGrpcRemoteHost>>> TClusterProvider::GetHostsGroupedByCluster() const {
    TLightReadGuard guard(Lock);
    THashMap<TString, TVector<TAtomicSharedPtr<TGrpcRemoteHost>>> clusterToHosts;
    for (const auto& [hostAndClusterName, host]: GrpcRemoteHosts) {
        auto& curHosts = clusterToHosts[hostAndClusterName.second];
        curHosts.reserve(GrpcRemoteHosts.size());
        curHosts.push_back(host);
    }

    TVector<TVector<TAtomicSharedPtr<TGrpcRemoteHost>>> result(Reserve(clusterToHosts.size()));
    for (auto& clusterHosts: clusterToHosts) {
        result.push_back(std::move(clusterHosts.second));
    }
    return result;
}

TAtomicSharedPtr<TGrpcRemoteHost> TClusterProvider::MakeGrpcRemoteHost(const TString& grpcRemoteHostName, TStockpilePort port) {
    TString grpcUrl = TStringBuilder() << grpcRemoteHostName << ":" << port;
    const TGrpcSettings& grpcSettings = TGrpcSettings::Get();
    std::vector<NSolomon::TInterceptorFactoryPtr> interceptors;
    interceptors.emplace_back(NSolomon::CreateCounterInterceptorFactory(*TMetricRegistry::Instance(), grpcSettings.GetSolomonClientId()));
    auto channel = grpc::experimental::CreateCustomChannelWithInterceptors(
        grpcUrl,
        grpc::InsecureChannelCredentials(),
        grpcSettings.GetChannelArguments(),
        std::move(interceptors));

    switch (Database) {
        case EStockpileDatabase::Metabase: {
            return MakeAtomicShared<TMetabaseRemoteHost>(grpcRemoteHostName, port, channel);
        }
        case EStockpileDatabase::Stockpile: {
            return MakeAtomicShared<TStockpileRemoteHost>(grpcRemoteHostName, port, channel);
        }
        case EStockpileDatabase::DataProxy: {
            return MakeAtomicShared<TDataProxyRemoteHost>(grpcRemoteHostName, port, channel);
        }
    }
}

bool TClusterProvider::IsMultiClusterProvider() const {
    return !ClusterName.Defined();
}

TAtomicSharedPtr<TGrpcRemoteHost> TClusterProvider::GetHost(const TString& hostName) const {
    if (IsMultiClusterProvider()) {
        ythrow yexception() << "TClusterProvider::GetHost: Specify cluster name";
    }
    return GetHost(hostName, ClusterName.GetRef());
}

TAtomicSharedPtr<TGrpcRemoteHost> TClusterProvider::GetHost(const TString& hostName, const TString& clusterName) const {
    TLightReadGuard guard(Lock);
    TAtomicSharedPtr<TGrpcRemoteHost> result;
    auto it = GrpcRemoteHosts.find(THostAndClusterName(hostName, clusterName));
    if (it != GrpcRemoteHosts.end()) {
        result = it->second;
    }
    return result;
}

bool TClusterProvider::UpdateWithStockpileResponse(const TString& jsonData) {
    TClusterData solomonData;
    try {
        solomonData = ExtractSolomonDataByCluster(jsonData, ClusterName, Log);
    } catch (...) {
        auto metricName = NMetrics::TStockpileStatsInitializer::MakeFailMetricName(NMetrics::STOCKPILE_DISCOVERY);
        TUnistat::Instance().PushSignalUnsafe(metricName, 1);
        return false;
    }
    if (solomonData.Hosts.empty()) {
        Log << TLOG_ERR << "No hosts in " << GetClusterTypeName(ClusterType)
            << " for " << GetDatabaseName(Database);
        return false;
    }

    THostNameToGrpcHostMap newRemoteHosts(solomonData.Hosts.size());
    {
        // most likely nothing has changed. try to check that with a read lock
        TLightReadGuard guard(Lock);
        if (ClusterPort == solomonData.Port) {
            bool haveNewHosts = false;
            for (const auto& hostAndClusterName : solomonData.Hosts) {
                auto it = GrpcRemoteHosts.find(hostAndClusterName);
                if (it == GrpcRemoteHosts.end()) {
                    haveNewHosts = true;
                } else {
                    newRemoteHosts[hostAndClusterName] = it->second;
                }
            }
            if (!haveNewHosts && newRemoteHosts.size() == GrpcRemoteHosts.size()) {
                // no changes
                return true;
            }
        }
    }
    // something has changed
    {
        TLightWriteGuard guard(Lock);
        for (const auto& hostAndClusterName : solomonData.Hosts) {
            THostNameToGrpcHostMap::insert_ctx insertCtx;
            if (!newRemoteHosts.contains(hostAndClusterName, insertCtx)) {
                TAtomicSharedPtr<TGrpcRemoteHost> hostToInsert;
                if (ClusterPort == solomonData.Port) {
                    // need to look for hostName in current map once again as it might have changed between the two locks
                    auto it = GrpcRemoteHosts.find(hostAndClusterName);
                    if (it != GrpcRemoteHosts.end()) {
                        hostToInsert = it->second;
                    }
                }
                if (!hostToInsert) {
                    hostToInsert = MakeGrpcRemoteHost(hostAndClusterName.first, solomonData.Port);
                }
                newRemoteHosts.emplace_direct(insertCtx, hostAndClusterName, std::move(hostToInsert));
            }
        }
        ClusterPort = solomonData.Port;
        GrpcRemoteHosts.swap(newRemoteHosts);
    }
    return true;
}
