#pragma once

#include <infra/yasm/stockpile_client/common/base_types.h>
#include "rpc.h"

#include <library/cpp/http/client/client.h>
#include <library/cpp/logger/log.h>

#include <util/string/split.h>
#include <util/generic/vector.h>

namespace NHistDb {
    namespace NStockpile {
        static constexpr TDuration CLUSTER_UPDATE_INTERVAL = TDuration::Seconds(40);
        static constexpr TDuration CLUSTER_UPDATE_JITTER = TDuration::Seconds(20);

        using TStockpilePort = unsigned short;
        using THostAndClusterName = std::pair<TString, TString>; // host name first, cluster name second

        namespace NImpl {
            class TFetcher {
            public:
                /**
                 * Fetches data from specified url.
                 * @param url - full url to get
                 * @param timeout - fetch timeout
                 * @return (<reply data>, true) in case of success
                 *         (<error message>, false) in case of a failure
                 */
                std::pair<TString, bool> operator()(const TString& url, TDuration timeout) const;
            };
        }

        class TClusterProvider {
        public:
            static constexpr TStringBuf STOCKPILE_DISCOVERY_URL = "https://solomon.yandex.net/discovery/";

            // construct single cluster host provider
            TClusterProvider(EStockpileDatabase database, const TClusterInfo& clusterInfo, TLog& log);

            // construct multicluster host provider for all clusters of a specified type
            TClusterProvider(EStockpileDatabase database, EStockpileClusterType clusterType, TLog& log);

            bool IsMultiClusterProvider() const;

            TVector<TAtomicSharedPtr<TGrpcRemoteHost>> GetHosts() const;
            TVector<TVector<TAtomicSharedPtr<TGrpcRemoteHost>>> GetHostsGroupedByCluster() const;

            /** Get remote host for the specifed host name */
            TAtomicSharedPtr<TGrpcRemoteHost> GetHost(const TString& hostName, const TString& clusterName) const;

            /** Get remote host for the specifed host name. Will raise an exception if this provider is multicluster provider */
            TAtomicSharedPtr<TGrpcRemoteHost> GetHost(const TString& hostName) const;

            /** Try to update stockpile cluster host list. Discovery handle is queried using the specified fetcher. */
            template <class TFetcher = NImpl::TFetcher>
            bool TryToUpdate(TFetcher fetcher = TFetcher(), bool forceUpdate = false) {
                bool success = true;
                if (forceUpdate || TInstant::Now() - LastUpdate >= CLUSTER_UPDATE_INTERVAL) {
                    success = Update(fetcher);
                    LastUpdate = TInstant::Now() - GetUpdateJitter();
                }
                return success;
            }

        private:
            using THostNameToGrpcHostMap = THashMap<THostAndClusterName, TAtomicSharedPtr<TGrpcRemoteHost>>;
            static TDuration GetUpdateJitter();
            static TString GetClusterTypeName(EStockpileClusterType clusterType);
            static TString GetDatabaseName(EStockpileDatabase database);

            template <class TFetcher>
            bool Update(TFetcher fetcher) {
                TString url = TString::Join(STOCKPILE_DISCOVERY_URL,
                    GetClusterTypeName(ClusterType), "/",
                    GetDatabaseName(Database), ".json");

                auto startTime = TInstant::Now();
                auto [reply, success] = fetcher(url, TDuration::Seconds(5));
                auto duration = TInstant::Now() - startTime;
                TUnistat::Instance().PushSignalUnsafe(NMetrics::STOCKPILE_DISCOVERY, duration.Seconds());

                if (!success) {
                    Log << TLOG_ERR << "Couldn't request solomon " << url << ": " << reply;
                    return false;
                }
                return UpdateWithStockpileResponse(reply);
            }

            bool UpdateWithStockpileResponse(const TString& jsonData);

            TAtomicSharedPtr<TGrpcRemoteHost> MakeGrpcRemoteHost(const TString& grpcRemoteHostName, TStockpilePort port);

            TMaybe<TStockpilePort> ClusterPort;
            const EStockpileDatabase Database;
            const EStockpileClusterType ClusterType;
            const TMaybe<TString> ClusterName;

            THostNameToGrpcHostMap GrpcRemoteHosts;

            TInstant LastUpdate;
            TLightRWLock Lock;
            TLog& Log;
        };
    }
}
