#include "cluster.h"
#include "state_ut.h"

#include <library/cpp/testing/unittest/registar.h>

using namespace NHistDb::NStockpile;

namespace {
    struct TTestFetcher {

        TString Response;
        bool Success;

        std::pair<TString, bool> operator()(const TString&, TDuration) const {
            return std::make_pair(Response, Success);
        }
    };

    inline TClusterInfo CreateClusterInfo() {
        return TClusterInfo({EStockpileClusterType::Testing, "sas"});
    }
}

Y_UNIT_TEST_SUITE(TClusterProviderTest) {
    Y_UNIT_TEST(DoNotFailOnFailedRequest) {
        auto clusterInfo = CreateClusterInfo();
        TLog log;
        TTestFetcher fetcher{"", false};

        TClusterProvider provider(EStockpileDatabase::Metabase, clusterInfo, log);

        UNIT_ASSERT(!provider.TryToUpdate(fetcher));
        UNIT_ASSERT(provider.GetHosts().empty());
    }

    Y_UNIT_TEST(DoNotFailOnIncorrectResponse) {
        auto clusterInfo = CreateClusterInfo();
        TLog log;
        TTestFetcher fetcher{"something that is not a json", true};

        TClusterProvider provider(EStockpileDatabase::Metabase, clusterInfo, log);

        UNIT_ASSERT(!provider.TryToUpdate(fetcher));
        UNIT_ASSERT(provider.GetHosts().empty());
    }

    Y_UNIT_TEST(IgnoreIncorrectResponse) {
        auto clusterInfo = CreateClusterInfo();
        TLog log;

        const auto grpcPort = 8796;
        const auto httpPort = 4500;
        TString singleEntryResponse = TStringBuilder() <<
            "{"
            "    \"envType\": \"TESTING\","
            "    \"hosts\": ["
            "        {"
            "            \"cluster\": \"sas\","
            "            \"fqdn\": \"solomon-test-sas-01.search.yandex.net\""
            "        }"
            "    ],"
            "    \"ports\": {"
            "        \"grpc\": " << grpcPort << ","
            "        \"http\": " << httpPort <<
            "    },"
            "    \"serviceName\": \"metabase\""
            "}";

        TTestFetcher fetcher{singleEntryResponse, true};

        TClusterProvider provider(EStockpileDatabase::Metabase, clusterInfo, log);

        UNIT_ASSERT(provider.TryToUpdate(fetcher));
        UNIT_ASSERT_VALUES_EQUAL(1, provider.GetHosts().size());

        TString corruptedResponse = TStringBuilder() <<
            "{"
            "    \"envType\": \"TESTING\","
            "    \"hosts\": ["
            "        {"
            "            \"cluster\": null,"
            "            \"fqdn\": \"solomon-test-sas-01.search.yandex.net\""
            "        },"
            "        {"
            "            \"cluster\": \"sas\","
            "            \"fqdn\": \"solomon-test-sas-02.search.yandex.net\""
            "        },"
            "        {"
            "            \"cluster\": \"sas\","
            "            \"fqdn\": \"solomon-test-sas-03.search.yandex.net\""
            "        }"
            "    ],"
            "    \"ports\": {"
            "        \"grpc\": " << grpcPort << ","
            "        \"http\": " << httpPort <<
            "    },"
            "    \"serviceName\": \"metabase\""
            "}";

        TTestFetcher fetcher2{corruptedResponse, true};
        UNIT_ASSERT(!provider.TryToUpdate(fetcher2, true));
        UNIT_ASSERT_VALUES_EQUAL(1, provider.GetHosts().size()); // hosts from previous successful update
    }

    Y_UNIT_TEST(ReusesHostOnUpdate) {
        auto clusterInfo = CreateClusterInfo();
        TLog log;

        const auto grpcPort = 8796;
        const auto httpPort = 4500;
        TString singleEntryResponse = TStringBuilder() <<
           "{"
           "    \"envType\": \"TESTING\","
           "    \"hosts\": ["
           "        {"
           "            \"cluster\": \"sas\","
           "            \"fqdn\": \"solomon-test-sas-01.search.yandex.net\""
           "        }"
           "    ],"
           "    \"ports\": {"
           "        \"grpc\": " << grpcPort << ","
           "        \"http\": " << httpPort <<
           "    },"
           "    \"serviceName\": \"metabase\""
           "}";

        TClusterProvider provider(EStockpileDatabase::Metabase, clusterInfo, log);

        UNIT_ASSERT(provider.TryToUpdate(TTestFetcher{singleEntryResponse, true}));
        UNIT_ASSERT_VALUES_EQUAL(1, provider.GetHosts().size());

        auto remoteHost = provider.GetHost("solomon-test-sas-01.search.yandex.net");

        TString multipleEntriesResponse = TStringBuilder() <<
            "{"
            "    \"envType\": \"TESTING\","
            "    \"hosts\": ["
            "        {"
            "            \"cluster\": \"sas\","
            "            \"fqdn\": \"solomon-test-sas-01.search.yandex.net\""
            "        },"
            "        {"
            "            \"cluster\": \"sas\","
            "            \"fqdn\": \"solomon-test-sas-02.search.yandex.net\""
            "        },"
            "        {"
            "            \"cluster\": \"sas\","
            "            \"fqdn\": \"solomon-test-sas-03.search.yandex.net\""
            "        }"
            "    ],"
            "    \"ports\": {"
            "        \"grpc\": " << grpcPort << ","
            "        \"http\": " << httpPort <<
            "    },"
            "    \"serviceName\": \"metabase\""
            "}";

        UNIT_ASSERT(provider.TryToUpdate(TTestFetcher{multipleEntriesResponse, true}, true));
        UNIT_ASSERT_VALUES_EQUAL(3, provider.GetHosts().size());

        // host wasn't changed after update
        UNIT_ASSERT_EQUAL(remoteHost, provider.GetHost("solomon-test-sas-01.search.yandex.net"));
    }

    Y_UNIT_TEST(UpdatesPort) {
        auto clusterInfo = CreateClusterInfo();
        TLog log;

        const auto grpcPort1 = 8796;
        const auto httpPort = 4500;
        TString response1 = TStringBuilder() <<
            "{"
            "    \"envType\": \"TESTING\","
            "    \"hosts\": ["
            "        {"
            "            \"cluster\": \"sas\","
            "            \"fqdn\": \"solomon-test-sas-01.search.yandex.net\""
            "        }"
            "    ],"
            "    \"ports\": {"
            "        \"grpc\": " << grpcPort1 << ","
            "        \"http\": " << httpPort <<
            "    },"
            "    \"serviceName\": \"metabase\""
            "}";

        TClusterProvider provider(EStockpileDatabase::Metabase, clusterInfo, log);

        UNIT_ASSERT(provider.TryToUpdate(TTestFetcher{response1, true}));
        UNIT_ASSERT_VALUES_EQUAL(1, provider.GetHosts().size());

        auto remoteHost1 = provider.GetHost("solomon-test-sas-01.search.yandex.net");
        UNIT_ASSERT_VALUES_EQUAL(grpcPort1, remoteHost1->GetPort());

        const auto grpcPort2 = 8797;
        TString response2 = TStringBuilder() <<
            "{"
            "    \"envType\": \"TESTING\","
            "    \"hosts\": ["
            "        {"
            "            \"cluster\": \"sas\","
            "            \"fqdn\": \"solomon-test-sas-01.search.yandex.net\""
            "        }"
            "    ],"
            "    \"ports\": {"
            "        \"grpc\": " << grpcPort2 << ","
            "        \"http\": " << httpPort <<
            "    },"
            "    \"serviceName\": \"metabase\""
            "}";

        UNIT_ASSERT(provider.TryToUpdate(TTestFetcher{response2, true}, true));
        UNIT_ASSERT_VALUES_EQUAL(1, provider.GetHosts().size());

        auto remoteHost2 = provider.GetHost("solomon-test-sas-01.search.yandex.net");
        UNIT_ASSERT_VALUES_EQUAL(grpcPort1, remoteHost1->GetPort());
        UNIT_ASSERT_VALUES_EQUAL(grpcPort2, remoteHost2->GetPort());
    }
}
