#include "base_types.h"

#include <library/cpp/yaml/as/tstring.h>
#include <library/cpp/consistent_hashing/consistent_hashing.h>

#include <util/string/split.h>
#include <limits>


using namespace NHistDb::NStockpile;

namespace {
    const size_t CLUSTER_LABEL_ALLOC_SIZE =
        Max(STOCKPILE_GROUP_CLUSTER_LABEL.size(), STOCKPILE_HOST_CLUSTER_LABEL.size())
        + std::numeric_limits<size_t>::digits
        + 1;

    const size_t PROJECT_LABEL_ALLOC_SIZE =  STOCKPILE_YASM_PROJECTS_PREFIX.size() + 32;

    const TString& KeyFromLabel(const TLabel& label) {
        return label.Key;
    }

    const TString& KeyFromLabel(const yandex::solomon::model::Label& label) {
        return label.key();
    }

    const TStringBuf& KeyFromLabel(const std::pair<TStringBuf, TStringBuf>& label) {
        return label.first;
    }

    const TString& ValueFromLabel(const TLabel& label) {
        return label.Value;
    }

    const TString& ValueFromLabel(const yandex::solomon::model::Label& label) {
        return label.value();
    }

    const TStringBuf& ValueFromLabel(const std::pair<TStringBuf, TStringBuf>& label) {
        return label.second;
    }

    template <class T>
    TString CreateLabelString(const T& labels) {
        TVector<TString> labelsString;
        for (const auto& label : labels) {
            labelsString.emplace_back(TString::Join(KeyFromLabel(label), '=', ValueFromLabel(label)));
        }
        return JoinRange(",", labelsString.begin(), labelsString.end());
    }

    template <class T>
    TMetabaseShardKey FromLabels(const T& labels) {
        TString projectId{};
        TString cluster{};
        TString service{};
        for (const auto& label : labels) {
            const auto& key = KeyFromLabel(label);
            const auto& value = ValueFromLabel(label);
            if (key == PROJECT_LABEL_KEY) {
                projectId = value;
            } else if (key == CLUSTER_LABEL_KEY) {
                cluster = value;
            } else if (key == SERVICE_LABEL_KEY) {
                service = value;
            }
        }
        if (projectId.empty() || cluster.empty() || service.empty()) {
            ythrow yexception() << "Incomplete shard info: " << CreateLabelString(labels);
        }
        return TMetabaseShardKey(
            std::move(projectId),
            std::move(cluster),
            std::move(service)
        );
    }

    class TMetabaseShardKeyBuilder {
    public:
        TMetabaseShardKeyBuilder() {
            Cluster.reserve(CLUSTER_LABEL_ALLOC_SIZE);
        }

        void SetProject(const TStringBuf project) {
            Project = project;
            ProjectId = MakeItypeProjectId(project);
        }

        void SetCluster(
            const NZoom::NHost::THostName host,
            const NZoom::NSignal::TSignalName signalName,
            const TShardIndex shardIndexCount
        ) {
            // NOTE(rocco66): we should use hash based on value, not address of interned strings
            ui64 hashValue{};
            if (host.Empty() || host.IsGroup()) {
                // this is a group or something else
                hashValue = CityHash64(signalName.GetName());
                Cluster += STOCKPILE_GROUP_CLUSTER_LABEL;
            } else {
                // this is a host
                hashValue = CombineHashes(CityHash64(host.GetName()), CityHash64(signalName.GetName()));
                Cluster += STOCKPILE_HOST_CLUSTER_LABEL;
            }

            Cluster += '_';
            const auto n = Cluster.size();
            Cluster.resize(CLUSTER_LABEL_ALLOC_SIZE);
            const auto strLen = ToString(
                ConsistentHashing(hashValue, shardIndexCount),
                Cluster.begin() + n, Cluster.capacity() - (n + 1)
            );

            Cluster.resize(n + strLen);
        }

        void SetService(const TStringBuf service) {
            Service = service;
        }

        TMetabaseShardKey Create() {
            return TMetabaseShardKey(
                std::move(ProjectId),
                std::move(Cluster),
                std::move(Service),
                std::move(Project)
            );
        }

    private:
        TString ProjectId;
        TString Project;
        TString Cluster;
        TString Service;
        TString Id;
    };

    template <class T>
    TSeriesKey CreateSeriesKey(const T& labels) {
        TStringBuf itype;
        TVector<std::pair<TStringBuf, TStringBuf>> tags;
        TVector<TStringBuf> aggregated;
        TStringBuf signalName;
        for (const auto& label : labels) {
            const auto& key = KeyFromLabel(label);
            const auto& value = ValueFromLabel(label);
            if (key == CLUSTER_LABEL_KEY || key == SERVICE_LABEL_KEY) {
                // skip metabase shard key
            } else if (key == PROJECT_LABEL_KEY) {
                itype = value;
                Y_VERIFY(itype.StartsWith(STOCKPILE_YASM_PROJECTS_PREFIX));
                itype.NextTokAt(STOCKPILE_YASM_PROJECTS_PREFIX.size());
            } else if (key == SIGNAL_LABEL_KEY) {
                signalName = value;
            } else if (value == AGGREGATED_MARKER) {
                aggregated.emplace_back(key);
            } else {
                tags.emplace_back(key, value);
            }
        }
        if (itype.empty() || signalName.empty()) {
            ythrow yexception() << "Incomplete series labels: " << CreateLabelString(labels);
        }
        return TSeriesKey::Make(
            NTags::TInstanceKey::FromParts(itype, tags, aggregated),
            NZoom::NSignal::TSignalName(signalName)
        );
    }

    struct TRequestKeyVisitor final: public NTags::IRequestKeyVisitor {
        TRequestKeyVisitor(TLabelSelectors& selectors)
            : Selectors(selectors)
            , Flat(*this)
            , Nested(*this)
            , SelectedVisitor(&Flat)
            , CurrentSelector(nullptr)
        {
        }

        void OnItype(TStringBuf itype) override {
            Y_UNUSED(itype);
            // NOTE(rocco66): itype lives in ProjectId, see https://st.yandex-team.ru/GOLOVAN-6591
        }

        void OnKey(TStringBuf key) override {
            CurrentSelector = &Selectors.emplace_back();
            CurrentSelector->set_key(TString(key));
        }

        void OnGroupStart(size_t count) override {
            SelectedVisitor = &Nested;
            SelectedVisitor->OnGroupStart(count);
        }

        void OnGroupEnd() override {
            SelectedVisitor->OnGroupEnd();
            SelectedVisitor = &Flat;
        }

        void OnEqualValues(TStringBuf value) override {
            SelectedVisitor->OnEqualValues(value);
        }

        void OnRegexpValues(TStringBuf value) override {
            SelectedVisitor->OnRegexpValues(value);
        }

        struct TFlatVisitor final: public NTags::IRequestKeyVisitor {
            TFlatVisitor(TRequestKeyVisitor& parent)
                : Parent(parent)
            {
            }

            void OnEqualValues(TStringBuf value) override {
                Parent.CurrentSelector->set_pattern(TString(value));
                Parent.CurrentSelector->set_match_type(yandex::solomon::model::MatchType::EXACT);
            }

            void OnRegexpValues(TStringBuf value) override {
                Parent.CurrentSelector->set_pattern(TString(value));
                Parent.CurrentSelector->set_match_type(yandex::solomon::model::MatchType::GLOB);
            }

            TRequestKeyVisitor& Parent;
        };

        struct TNestedVisitor final: public NTags::IRequestKeyVisitor {
            TNestedVisitor(TRequestKeyVisitor& parent)
                : Parent(parent)
            {
            }

            void OnGroupStart(size_t count) override {
                Values.clear();
                Values.reserve(count);
                Parent.CurrentSelector->set_match_type(yandex::solomon::model::MatchType::GLOB);
            }

            void OnGroupEnd() override {
                Parent.CurrentSelector->set_pattern(JoinSeq("|", Values));
            }

            void OnEqualValues(TStringBuf value) override {
                Values.emplace_back(value);
            }

            void OnRegexpValues(TStringBuf value) override {
                Values.emplace_back(value);
            }

            TRequestKeyVisitor& Parent;
            TVector<TStringBuf> Values;
        };

        TLabelSelectors& Selectors;
        TFlatVisitor Flat;
        TNestedVisitor Nested;
        NTags::IRequestKeyVisitor* SelectedVisitor;
        yandex::solomon::model::Selector* CurrentSelector;
    };
}

TSensorId::TSensorId(TShardId shardId, TLocalId localId, yandex::solomon::model::MetricType type)
    : ShardId(shardId)
    , LocalId(localId)
    , Type(type)
{}

TSensorId::TSensorId(const yandex::solomon::metabase::Metric& metric)
    : ShardId(metric.metric_id().shard_id())
    , LocalId(metric.metric_id().local_id())
    , Type(metric.type())
{}

TSensorId::TSensorId(const yandex::solomon::stockpile::MetricData& metric)
    : ShardId(metric.GetShardId())
    , LocalId(metric.GetLocalId())
    , Type(metric.GetType())
{}


TMetabaseConfig::TMetabaseConfig(const TStringBuf rawConfig)
    : Config{}
{
    auto yamlConfig {YAML::Load(static_cast<std::string>(rawConfig))};
    for (YAML::Node part: yamlConfig["shards"]) {
        Config.emplace(part["itype"].as<TString>(), part["shards_count"].as<size_t>());
    }
}

TShardIndex TMetabaseConfig::GetShardsCount(const TStringBuf itype) const {
    auto res = Config.find(itype);
    if (res != Config.end()) {
        return res->second;
    } else {
        return 1;
    }
}

TString NHistDb::NStockpile::MakeItypeProjectId(TStringBuf itype) {
    TString result;
    result.reserve(STOCKPILE_YASM_PROJECTS_PREFIX.size() + itype.size());
    result += STOCKPILE_YASM_PROJECTS_PREFIX;
    result += itype;
    return result;
}

TMetabaseShardKey TMetabaseShardKey::Make(const TLabels& labels) {
    return FromLabels(labels);
}

TMetabaseShardKey TMetabaseShardKey::Make(const google::protobuf::RepeatedPtrField<yandex::solomon::model::Label>& labels) {
    return FromLabels(labels);
}

TMetabaseShardKey TMetabaseShardKey::Make(
    TStringBuf instanceType,
    const NZoom::NHost::THostName& host,
    const NZoom::NSignal::TSignalName& signalName,
    const TShardIndex shardIndexCount
) {
    TMetabaseShardKeyBuilder builder;
    builder.SetProject(instanceType);
    builder.SetCluster(host, signalName, shardIndexCount);
    builder.SetService(STOCKPILE_YASM_SERVICE);
    return builder.Create();
}

TMetabaseShardKey TMetabaseShardKey::Make(
    const NTags::TInstanceKey instanceKey,
    const NZoom::NSignal::TSignalName signalName,
    const TShardIndex shardIndexCount
) {
    TMetabaseShardKeyBuilder builder;
    builder.SetProject(instanceKey.GetItype());
    builder.SetCluster(instanceKey.GetHostName(), signalName, shardIndexCount);
    builder.SetService(STOCKPILE_YASM_SERVICE);
    return builder.Create();
}

TMetabaseShardKey TMetabaseShardKey::Make(
    const NZoom::NHost::THostName host,
    const NTags::TRequestKey& requestKey,
    const NZoom::NSignal::TSignalName signalName,
    const TShardIndex shardIndexCount
) {
    TMetabaseShardKeyBuilder builder;
    builder.SetProject(requestKey.GetItype());
    builder.SetCluster(host, signalName, shardIndexCount);
    builder.SetService(STOCKPILE_YASM_SERVICE);
    return builder.Create();
}

void TMetabaseShardKey::FillLabels(TLabels& labels) const noexcept {
    auto& projectLabel = labels.emplace_back();
    projectLabel.Key = PROJECT_LABEL_KEY;
    projectLabel.Value = GetProjectId();

    auto& clusterLabel = labels.emplace_back();
    clusterLabel.Key = CLUSTER_LABEL_KEY;
    clusterLabel.Value = GetCluster();

    auto& serviceLabel = labels.emplace_back();
    serviceLabel.Key = SERVICE_LABEL_KEY;
    serviceLabel.Value = GetService();
}

void TMetabaseShardKey::FillSelectors(TLabelSelectors& selectors) const noexcept {
    auto& projectSelector = selectors.emplace_back();
    projectSelector.set_key(PROJECT_LABEL_KEY);
    projectSelector.set_pattern(GetProjectId());
    projectSelector.set_match_type(yandex::solomon::model::MatchType::EXACT);

    auto& clusterSelector = selectors.emplace_back();
    clusterSelector.set_key(CLUSTER_LABEL_KEY);
    clusterSelector.set_pattern(GetCluster());
    clusterSelector.set_match_type(yandex::solomon::model::MatchType::EXACT);

    auto& serviceSelector = selectors.emplace_back();
    serviceSelector.set_key(SERVICE_LABEL_KEY);
    serviceSelector.set_pattern(GetService());
    serviceSelector.set_match_type(yandex::solomon::model::MatchType::EXACT);
}

TSeriesKey TSeriesKey::Make(NTags::TInstanceKey instanceKey, NZoom::NSignal::TSignalName signalName) {
    return {
        .InstanceKey = instanceKey,
        .SignalName = signalName
    };
}

TSeriesKey TSeriesKey::Make(const TLabels& labels) {
    return CreateSeriesKey(labels);
}

TSeriesKey TSeriesKey::Make(const google::protobuf::RepeatedPtrField<yandex::solomon::model::Label>& labels) {
    return CreateSeriesKey(labels);
}

TSeriesKey TSeriesKey::Make(const TVector<std::pair<TStringBuf, TStringBuf>>& labels) {
    return CreateSeriesKey(labels);
}

void TSeriesKey::FillLabels(TLabels& labels) const noexcept {
    auto& signalLabel = labels.emplace_back();
    signalLabel.Key = SIGNAL_LABEL_KEY;
    signalLabel.Value = SignalName.GetName();

    for (const auto& tagPair : InstanceKey.GetTags()) {
        auto& label = labels.emplace_back();
        label.Key = tagPair.Name;
        label.Value = tagPair.Value;
    }

    for (const auto& tag : InstanceKey.GetAggregated()) {
        auto& aggregatedLabel = labels.emplace_back();
        aggregatedLabel.Key = tag;
        aggregatedLabel.Value = AGGREGATED_MARKER;
    }
}

void TSelectorsBuilder::FromHostName(const NZoom::NHost::THostName& hostName) {
    auto& hostSelector(Selectors.emplace_back());
    hostSelector.set_key(hostName.IsGroup() ? GROUP_LABEL_KEY : HOST_LABEL_KEY);
    hostSelector.set_pattern(hostName.GetName());
    hostSelector.set_match_type(yandex::solomon::model::MatchType::EXACT);
}

void TSelectorsBuilder::FromMultipleHostNames(const TVector<NZoom::NHost::THostName>& hostNames) {
    if (hostNames.empty()) {
        return;
    } else if (hostNames.size() == 1) {
        return FromHostName(hostNames.back());
    }

    auto& someSelector(Selectors.emplace_back());
    someSelector.set_match_type(yandex::solomon::model::MatchType::GLOB);

    bool isGroup = AnyOf(hostNames.begin(), hostNames.end(), [](const NZoom::NHost::THostName& hostName) {
        return hostName.IsGroup();
    });
    someSelector.set_key(isGroup ? GROUP_LABEL_KEY : HOST_LABEL_KEY);

    TVector<TStringBuf> names(Reserve(hostNames.size()));
    for (const auto& hostName : hostNames) {
        names.emplace_back(hostName.GetName());
    }
    someSelector.set_pattern(JoinSeq("|", names));
}

void TSelectorsBuilder::FromRequestKey(const NTags::TRequestKey& requestKey) {
    TRequestKeyVisitor visitor(Selectors);
    requestKey.Visit(visitor);
}

void TSelectorsBuilder::FromInstanceKey(const NTags::TInstanceKey& instanceKey) {
    for (const auto& tag : instanceKey.GetTags()) {
        auto& signalSelector(Selectors.emplace_back());
        signalSelector.set_key(TString{tag.Name});
        signalSelector.set_pattern(TString{tag.Value});
        signalSelector.set_match_type(yandex::solomon::model::MatchType::EXACT);
    }
}

void TSelectorsBuilder::FromSignalNames(const TVector<NZoom::NSignal::TSignalName>& signalNames) {
    TVector<TStringBuf> names(Reserve(signalNames.size()));
    for (const auto& signal : signalNames) {
        names.emplace_back(signal.GetName());
    }

    auto& signalSelector(Selectors.emplace_back());
    signalSelector.set_key(SIGNAL_LABEL_KEY);
    signalSelector.set_pattern(JoinSeq("|", names));
    signalSelector.set_match_type(yandex::solomon::model::MatchType::GLOB);
}

template <>
void Out<TMetabaseShardKey>(IOutputStream& stream,
                            TTypeTraits<TMetabaseShardKey>::TFuncParam key) {
    stream << "TMetabaseShardKey{.ProjectId=\"" << key.GetProjectId() << "\", .ClusterType=\"" << key.GetCluster() << "\", .Service=\"" << key.GetService() << "\"}";
}

template <>
void Out<TSeriesKey>(IOutputStream& stream,
                     TTypeTraits<TSeriesKey>::TFuncParam key) {
    stream << "TSeriesKey{.InstanceKey=\"" << key.InstanceKey << "\", .SignalName=\"" << key.SignalName << "\"}";
}

template <>
void Out<TClusterInfo>(IOutputStream& stream,
                       TTypeTraits<TClusterInfo>::TFuncParam info) {
    stream << "TClusterInfo{.ClusterType=\"" << info.ClusterType << "\", .ClusterName=\"" << info.ClusterName << "}";
}

template <>
void Out<EStockpileClusterType>(IOutputStream& stream,
                                TTypeTraits<EStockpileClusterType>::TFuncParam clusterType) {
    switch (clusterType) {
        case EStockpileClusterType::Testing: {
            stream << "testing";
            break;
        };
        case EStockpileClusterType::Prestable: {
            stream << "prestable";
            break;
        };
        case EStockpileClusterType::Production: {
            stream << "production";
            break;
        };
    }
}
