#pragma once

#include <infra/yasm/stockpile_client/settings.h>

#include <infra/yasm/common/labels/signal/signal_name.h>
#include <infra/yasm/common/labels/host/host.h>
#include <infra/yasm/common/labels/tags/instance_key.h>
#include <infra/yasm/common/labels/tags/request_key.h>

#include <solomon/protos/metabase/metric.pb.h>
#include <solomon/protos/model/label.pb.h>
#include <solomon/protos/model/metric.pb.h>
#include <solomon/protos/model/metric_id.pb.h>
#include <solomon/protos/model/selector.pb.h>
#include <solomon/protos/stockpile/metric_data.pb.h>

#include <util/datetime/base.h>
#include <util/digest/city.h>
#include <util/generic/hash.h>
#include <util/generic/string.h>
#include <util/string/builder.h>
#include <util/string/join.h>
#include <util/string/hex.h>

namespace NHistDb::NStockpile {
    using TShardId = ui32;
    using TLocalId = ui64;
    using TStockpileShardId = TShardId;

    class TShardNotFoundError : public yexception {};

    struct TSensorId {
        TShardId ShardId;
        TLocalId LocalId;
        yandex::solomon::model::MetricType Type;

        TSensorId(TShardId shardId, TLocalId localId, yandex::solomon::model::MetricType type);
        TSensorId(const yandex::solomon::metabase::Metric& metric);
        TSensorId(const yandex::solomon::stockpile::MetricData& metric);

        bool operator==(const TSensorId& other) const {
            return ShardId == other.ShardId && LocalId == other.LocalId && Type == other.Type;
        }
    };

    struct TLabel {
        TString Key;
        TString Value;
    };

    using TLabels = TVector<TLabel>;

    using TLabelSelectors = TVector<yandex::solomon::model::Selector>;

    static const TString YASM_USER_AGENT{"yasm"};

    enum class EStockpileDatabase {
        Stockpile,
        Metabase,
        DataProxy
    };

    static const TString STOCKPILE_YASM_PROJECTS_PREFIX {"yasm_"};
    static const TString STOCKPILE_YASM_SERVICE {"yasm"};

    static const TString HOST_LABEL_KEY{"host"};
    static const TString GROUP_LABEL_KEY{"group"};
    static const TString SIGNAL_LABEL_KEY{"signal"};
    static const TString PROJECT_LABEL_KEY{"project"};
    static const TString CLUSTER_LABEL_KEY{"cluster"};
    static const TString SERVICE_LABEL_KEY{"service"};
    static const TString AGGREGATED_MARKER{"/SELF"};
    static const TString POLICY_5_MIN_AFTER_8_DAYS_NAME{"POLICY_5_MIN_AFTER_8_DAYS"};

    static const TString STOCKPILE_GROUP_CLUSTER_LABEL{"group"};
    static const TString STOCKPILE_HOST_CLUSTER_LABEL{"host"};

    /** Construct solomon storage project id for a specified itype */
    TString MakeItypeProjectId(TStringBuf itype);

    class TMetabaseShardKey {
    public:
        TMetabaseShardKey()
        {
        }

        // NOTE(rocco66): ProjectId, ClusterId, ServiceId used in JSON API for create shards
        // ProjectId, Cluster, Service used in GRPC shards Labels
        // Project is just common project name property in JSON API

        TMetabaseShardKey(
            TString projectId,
            TString cluster,
            TString service,
            TMaybe<TString> project = Nothing()
        )
            : ProjectId(std::move(projectId))
            , Cluster(std::move(cluster))
            , Service(std::move(service))
            , Project(std::move(project))
        {
        }

        static TMetabaseShardKey Make(const TLabels& labels);
        static TMetabaseShardKey Make(const google::protobuf::RepeatedPtrField<yandex::solomon::model::Label>& labels);

        static TMetabaseShardKey Make(
            const NTags::TInstanceKey instanceKey,
            const NZoom::NSignal::TSignalName signalName,
            const TShardIndex shardIndexCount
        );
        static TMetabaseShardKey Make(
            const NZoom::NHost::THostName host,
            const NTags::TRequestKey& requestKey,
            const NZoom::NSignal::TSignalName signalName,
            const TShardIndex shardIndexCount
        );

        static TMetabaseShardKey Make(
            TStringBuf instanceType,
            const NZoom::NHost::THostName& host,
            const NZoom::NSignal::TSignalName& signalName,
            const TShardIndex shardIndexCount
        );

        const TString& GetProjectId() const noexcept {
            return ProjectId;
        }

        const TString& GetProject() const noexcept {
            return Project.GetRef();
        }

        const TString& GetCluster() const noexcept {
            return Cluster;
        }

        const TString GetClusterId() const noexcept {
            // NOTE(rocco66): limited uniq ID
            return AddProjectIdPrefix(Cluster);
        }

        const TString& GetService() const noexcept {
            return Service;
        }

        const TString GetServiceId() const noexcept {
            // NOTE(rocco66): limited uniq ID
            return ProjectId;
        }

        const TString AddProjectIdPrefix(const TStringBuf string_buf) const noexcept {
            return JoinSeq("_", {GetProjectId(), string_buf});
        }

        const TString GetShardId() const noexcept {
            // NOTE(rocco66): limited uniq ID
            return JoinSeq("_", {GetProjectId(), Cluster});
        }

        bool IsGroupShard() const {
            return Cluster.StartsWith(STOCKPILE_GROUP_CLUSTER_LABEL + "_");
        }

        bool IsYasmShard() const noexcept {
            return ProjectId.StartsWith(STOCKPILE_YASM_PROJECTS_PREFIX) && Service.equal(STOCKPILE_YASM_SERVICE);
        }

        size_t Hash() const noexcept {
            return MultiHash(ProjectId, Cluster);
        }

        bool operator==(const TMetabaseShardKey& other) const noexcept {
            return ProjectId == other.ProjectId && Cluster == other.Cluster;
        }

        void FillLabels(TLabels& labels) const noexcept;
        void FillSelectors(TLabelSelectors& selectors) const noexcept;

    private:
        TString ProjectId; // NOTE(rocco66): limited uniq ID
        TString Cluster;  // NOTE(rocco66): unlimited human friendly
        TString Service;
        TMaybe<TString> Project;  // NOTE(rocco66): unlimited human friendly, can't get from metabase shards labels
    };

    struct TSeriesKey {
        NTags::TInstanceKey InstanceKey;
        NZoom::NSignal::TSignalName SignalName;

        static TSeriesKey Make(NTags::TInstanceKey instanceKey, NZoom::NSignal::TSignalName signalName);
        static TSeriesKey Make(const TLabels& labels);
        static TSeriesKey Make(const google::protobuf::RepeatedPtrField<yandex::solomon::model::Label>& labels);
        static TSeriesKey Make(const TVector<std::pair<TStringBuf, TStringBuf>>& labels);

        inline bool operator==(const TSeriesKey& other) const noexcept {
            return InstanceKey == other.InstanceKey && SignalName == other.SignalName;
        }

        inline size_t Hash() const noexcept {
            return MultiHash(InstanceKey, SignalName);
        }

        void FillLabels(TLabels& labels) const noexcept;
    };

    struct TSelectorsBuilder {
        TSelectorsBuilder(TLabelSelectors& selectors)
            : Selectors(selectors)
        {
        }

        void FromHostName(const NZoom::NHost::THostName& hostName);
        void FromMultipleHostNames(const TVector<NZoom::NHost::THostName>& hostNames);
        void FromRequestKey(const NTags::TRequestKey& requestKey);
        void FromInstanceKey(const NTags::TInstanceKey& instanceKey);
        void FromSignalNames(const TVector<NZoom::NSignal::TSignalName>& signalNames);

        TLabelSelectors& Selectors;
    };
}

template <>
struct THash<NHistDb::NStockpile::TMetabaseShardKey> {
    inline size_t operator()(const NHistDb::NStockpile::TMetabaseShardKey& obj) const noexcept {
        return obj.Hash();
    }
};

template <>
struct THash<NHistDb::NStockpile::TSeriesKey> {
    inline size_t operator()(const NHistDb::NStockpile::TSeriesKey& obj) const noexcept {
        return obj.Hash();
    }
};
