#include "base_types_ut.h"

#include <library/cpp/testing/unittest/registar.h>

#include <util/generic/xrange.h>

using namespace NHistDb::NStockpile;
using namespace NTags;
using namespace NZoom::NHost;
using namespace NZoom::NSignal;
using namespace NHistDb::NStockpile::NTest;

struct TLabelsBuilder {
    void Add(const TString& key, const TString& value) {
        auto& label(Labels.emplace_back());
        label.Key = key;
        label.Value = value;
    }

    TLabels Labels;
};

static const TString LONG_ITYPE{"fooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooor_test"};

Y_UNIT_TEST_SUITE(TMetabaseShardKeyTest) {
    Y_UNIT_TEST(ShardKeyFromStockpileStructuresShouldStaysUnchanged) {
        TLabelsBuilder builder;
        builder.Add(PROJECT_LABEL_KEY, "first");
        builder.Add(CLUSTER_LABEL_KEY, "second");
        builder.Add(SERVICE_LABEL_KEY, "third");
        const auto key(TMetabaseShardKey::Make(builder.Labels));
        UNIT_ASSERT_VALUES_EQUAL(key.GetProjectId(), "first");
        UNIT_ASSERT_VALUES_EQUAL(key.GetCluster(), "second");
        UNIT_ASSERT_VALUES_EQUAL(key.GetService(), "third");

        builder.Add("other", "fourth");
        const auto other(TMetabaseShardKey::Make(builder.Labels));
        UNIT_ASSERT_VALUES_EQUAL(key, other);
        UNIT_ASSERT_VALUES_EQUAL(key.Hash(), other.Hash());

        TLabels labels;
        other.FillLabels(labels);
        UNIT_ASSERT_VALUES_EQUAL(labels.size(), 3);
        UNIT_ASSERT_VALUES_EQUAL(labels[0].Key, PROJECT_LABEL_KEY);
        UNIT_ASSERT_VALUES_EQUAL(labels[0].Value, "first");
        UNIT_ASSERT_VALUES_EQUAL(labels[1].Key, CLUSTER_LABEL_KEY);
        UNIT_ASSERT_VALUES_EQUAL(labels[1].Value, "second");
        UNIT_ASSERT_VALUES_EQUAL(labels[2].Key, SERVICE_LABEL_KEY);
        UNIT_ASSERT_VALUES_EQUAL(labels[2].Value, "third");
    }

    Y_UNIT_TEST(ServiceBeforeAndAfterMetabaseShouldNotCheched) {
        const auto shardKeyBeforeMetabase(MakeMetabaseShardKey(
            NTags::TInstanceKey::FromNamed(LONG_ITYPE)
        ));

        TLabelsBuilder builder;
        builder.Add(PROJECT_LABEL_KEY, shardKeyBeforeMetabase.GetProjectId());
        builder.Add(CLUSTER_LABEL_KEY, shardKeyBeforeMetabase.GetCluster());
        builder.Add(SERVICE_LABEL_KEY, shardKeyBeforeMetabase.GetService());
        const auto shardKeyAfterMetabase(TMetabaseShardKey::Make(builder.Labels));
        UNIT_ASSERT_VALUES_EQUAL(shardKeyBeforeMetabase.GetProjectId(), shardKeyAfterMetabase.GetProjectId());
        UNIT_ASSERT_VALUES_EQUAL(shardKeyBeforeMetabase.GetCluster(), shardKeyAfterMetabase.GetCluster());
        UNIT_ASSERT_VALUES_EQUAL(shardKeyBeforeMetabase.GetService(), shardKeyAfterMetabase.GetService());
    }

    Y_UNIT_TEST(ShardsIsGroup) {
        const auto groupKey(MakeMetabaseShardKey(
                NTags::TInstanceKey::FromNamed("base|group=SOME_GROUP;ctype=prod|host,geo")
        ));
        UNIT_ASSERT(groupKey.IsGroupShard());

        const auto hostKey(MakeMetabaseShardKey(
                NTags::TInstanceKey::FromNamed("common|group=SOME_GROUP;host=some_host")
        ));
        UNIT_ASSERT(!hostKey.IsGroupShard());
    }

    Y_UNIT_TEST(IdsShouldHasYasmPrefix) {
        const auto groupKey(MakeMetabaseShardKey(
            NTags::TInstanceKey::FromNamed("base|group=SOME_GROUP;ctype=prod|host,geo")
        ));

        UNIT_ASSERT(groupKey.GetProjectId().StartsWith(STOCKPILE_YASM_PROJECTS_PREFIX));
        UNIT_ASSERT(groupKey.GetClusterId().StartsWith(groupKey.GetProjectId()));
        UNIT_ASSERT(
            groupKey.GetClusterId().EndsWith(JoinSeq("_", {STOCKPILE_GROUP_CLUSTER_LABEL, "0"}))
        );
        UNIT_ASSERT(groupKey.GetServiceId().StartsWith(groupKey.GetProjectId()));

        const auto hostKey(MakeMetabaseShardKey(
            NTags::TInstanceKey::FromNamed("common|group=SOME_GROUP;host=some_host")
        ));

        UNIT_ASSERT(hostKey.GetProjectId().StartsWith(STOCKPILE_YASM_PROJECTS_PREFIX));
        UNIT_ASSERT(hostKey.GetClusterId().StartsWith(hostKey.GetProjectId()));
        UNIT_ASSERT(
            hostKey.GetClusterId().EndsWith(JoinSeq("_", {STOCKPILE_HOST_CLUSTER_LABEL, "0"}))
        );
        UNIT_ASSERT(hostKey.GetServiceId().StartsWith(hostKey.GetProjectId()));
    }

    Y_UNIT_TEST(Reading) {
        const auto groupKey(MakeMetabaseShardKey(
            THostName(TStringBuf("SOME_GROUP")),
            TRequestKey::FromString("itype=base;prj=something")
        ));
        UNIT_ASSERT_VALUES_EQUAL(groupKey.GetProject(), "base");
        UNIT_ASSERT_VALUES_EQUAL(
            groupKey.GetCluster(),
            JoinSeq("_", {STOCKPILE_GROUP_CLUSTER_LABEL, "0"})
        );
        UNIT_ASSERT_VALUES_EQUAL(groupKey.GetService(), STOCKPILE_YASM_SERVICE);
    }

    Y_UNIT_TEST(TestEqual) {
        NZoom::NHost::THostName group(TStringBuf("SAS.000"));
        NZoom::NHost::THostName host(TStringBuf("man1-3578.search.yandex.net"));
        TVector<TMetabaseShardKey> keys;
        keys.emplace_back(MakeMetabaseShardKey(host, TRequestKey::FromString("itype=common")));
        keys.emplace_back(MakeMetabaseShardKey(host, TRequestKey::FromString("itype=common")));
        keys.emplace_back(MakeMetabaseShardKey(host, TRequestKey::FromString("itype=common")));
        keys.emplace_back(MakeMetabaseShardKey(host, TRequestKey::FromString("itype=common;geo=sas")));
        keys.emplace_back(MakeMetabaseShardKey(NTags::TInstanceKey::FromNamed("common").AddGroupAndHost(group, host)));
        TLabels labels;
        keys.front().FillLabels(labels);
        keys.emplace_back(TMetabaseShardKey::Make(labels));
        for (const auto& key1 : keys) {
            for (const auto& key2 : keys) {
                UNIT_ASSERT_VALUES_EQUAL(key1, key2);
            }
        }
        UNIT_ASSERT_VALUES_EQUAL(keys.front().GetShardId(), "yasm_common_host_0");
    }

    Y_UNIT_TEST(TestUnEqual) {
        NZoom::NHost::THostName host1(TStringBuf("man1-1111.search.yandex.net"));
        NZoom::NHost::THostName group(TStringBuf("SAS.000"));
        TVector<TMetabaseShardKey> keys;
        keys.emplace_back(MakeMetabaseShardKey(host1, TRequestKey::FromString("itype=common")));
        keys.emplace_back(MakeMetabaseShardKey(host1, TRequestKey::FromString("itype=base")));
        keys.emplace_back(MakeMetabaseShardKey(group, TRequestKey::FromString("itype=common")));
        for (size_t i : xrange(keys.size() - 1)) {
            for (size_t j : xrange(i + 1, keys.size())) {
                const auto &key1 = keys[i];
                const auto &key2 = keys[j];
                UNIT_ASSERT_VALUES_UNEQUAL(i, j);
                UNIT_ASSERT_VALUES_UNEQUAL(key1, key2);
            }
        }
    }
}

Y_UNIT_TEST_SUITE(TMetabaseShardConfig) {

    Y_UNIT_TEST(TestHostsSharding) {
        NZoom::NHost::THostName group(TStringBuf("MAN"));
        NZoom::NHost::THostName host1(TStringBuf("man1-1111.search.yandex.net"));
        NZoom::NHost::THostName host2(TStringBuf("man1-2222.search.yandex.net"));
        TString signalNameString {"signal_xxxx"};
        NZoom::NSignal::TSignalName signalName {signalNameString};
        auto instanceKey = TInstanceKey::FromNamed("itype");
        const TShardIndex shardsCount {10};

        // NOTE(rocco66): different hosts
        auto key1 = TMetabaseShardKey::Make(instanceKey.AddGroupAndHost(group, host1), signalName, shardsCount);
        auto key2 = TMetabaseShardKey::Make(instanceKey.AddGroupAndHost(group, host2), signalName, shardsCount);
        UNIT_ASSERT_VALUES_UNEQUAL(key1, key2);

        // NOTE(rocco66): different signals
        TString otherSignalNameString {"other_signal_xxxx"};
        NZoom::NSignal::TSignalName otherSignalName {otherSignalNameString};
        auto key3 = TMetabaseShardKey::Make(instanceKey.AddGroupAndHost(group, host1), otherSignalName, shardsCount);
        UNIT_ASSERT_VALUES_UNEQUAL(key1, key3);

        // NOTE(rocco66): different itype
        auto otherItypeInstanceKey = TInstanceKey::FromNamed("common");
        auto key4 = TMetabaseShardKey::Make(otherItypeInstanceKey.AddGroupAndHost(group, host1), signalName, shardsCount);
        UNIT_ASSERT_VALUES_UNEQUAL(key1, key4);

        // NOTE(rocco66): same tags (not itype)
        auto otherInstanceKey = TInstanceKey::FromNamed("itype|geo=man");
        auto key5 = TMetabaseShardKey::Make(otherInstanceKey.AddGroupAndHost(group, host1), signalName, shardsCount);
        UNIT_ASSERT_VALUES_EQUAL(key1, key5);
    }

    Y_UNIT_TEST(TestGroupsSharding) {
        TString signalNameString {"signal_xxxx"};
        NZoom::NSignal::TSignalName signalName {signalNameString};
        auto instanceKey1 = TInstanceKey::FromNamed("itype|group=MAN.001");
        auto instanceKey2 = TInstanceKey::FromNamed("itype|group=MAN.002");
        const TShardIndex shardsCount {10};

        // NOTE(rocco66): different groups
        auto key1 = TMetabaseShardKey::Make(instanceKey1, signalName, shardsCount);
        auto key2 = TMetabaseShardKey::Make(instanceKey2, signalName, shardsCount);
        UNIT_ASSERT_VALUES_EQUAL(key1, key2);

        // NOTE(rocco66): different signals
        TString otherSignalNameString {"other_signal_xxxx"};
        NZoom::NSignal::TSignalName otherSignalName {otherSignalNameString};
        auto key3 = TMetabaseShardKey::Make(instanceKey1, otherSignalName, shardsCount);
        UNIT_ASSERT_VALUES_UNEQUAL(key1, key3);

        // NOTE(rocco66): different itype
        auto otherItypeInstanceKey = TInstanceKey::FromNamed("common|group=MAN.001");
        auto key4 = TMetabaseShardKey::Make(otherItypeInstanceKey, signalName, shardsCount);
        UNIT_ASSERT_VALUES_UNEQUAL(key1, key4);

        // NOTE(rocco66): same tags (not itype)
        auto otherInstanceKey = TInstanceKey::FromNamed("itype|geo=man;group=MAN.001");
        auto key5 = TMetabaseShardKey::Make(otherInstanceKey, signalName, shardsCount);
        UNIT_ASSERT_VALUES_EQUAL(key1, key5);
    }
}

Y_UNIT_TEST_SUITE(TSeriesKeyTest) {
    Y_UNIT_TEST(Conversion) {
        THostName groupName(TStringBuf("SAS.000"));
        THostName hostName(TStringBuf("base_host"));
        TInstanceKey instanceKey(TInstanceKey::FromNamed("addrsupper|ctype=prestable;geo=sas;prj=addrs|tier").AddGroupAndHost(groupName, hostName));
        TSignalName signalName(TStringBuf("something_summ"));

        auto key(TSeriesKey::Make(instanceKey, signalName));
        TLabels labels;
        key.FillLabels(labels);
        auto& projectLabel = labels.emplace_back();
        projectLabel.Key = PROJECT_LABEL_KEY;
        projectLabel.Value = "yasm_addrsupper";

        auto other(TSeriesKey::Make(labels));
        UNIT_ASSERT_VALUES_EQUAL(key, other);
        UNIT_ASSERT_VALUES_EQUAL(key.Hash(), other.Hash());
        UNIT_ASSERT_VALUES_EQUAL(other.InstanceKey, instanceKey);
        UNIT_ASSERT_VALUES_EQUAL(other.SignalName, signalName);

        labels.pop_back();  // NOTE(rocco66): we need project only for for TSeriesKey creation
        THashMap<TString, TString> labelMap;
        for (const auto& label : labels) {
            labelMap[label.Key] = label.Value;
        }

        THashMap<TString, TString> expectedLabelMap;
        expectedLabelMap.emplace(SIGNAL_LABEL_KEY, signalName.GetName());
        for (const auto& tag : instanceKey.GetTags()) {
            expectedLabelMap.emplace(tag.Name, tag.Value);
        }
        for (const auto& tag : instanceKey.GetAggregated()) {
            expectedLabelMap.emplace(tag, AGGREGATED_MARKER);
        }
        UNIT_ASSERT_VALUES_EQUAL(labelMap, expectedLabelMap);
    }
}

Y_UNIT_TEST_SUITE(TSelectorsBuilder) {
    Y_UNIT_TEST(HostName) {
        THostName hostName(TStringBuf("base_host"));
        TLabelSelectors selectors;
        TSelectorsBuilder builder(selectors);
        builder.FromHostName(hostName);
        UNIT_ASSERT_VALUES_EQUAL(selectors.size(), 1);
        UNIT_ASSERT_VALUES_EQUAL(selectors.back().key(), HOST_LABEL_KEY);
        UNIT_ASSERT_VALUES_EQUAL(selectors.back().pattern(), hostName.GetName());
        UNIT_ASSERT_VALUES_EQUAL(selectors.back().match_type(), yandex::solomon::model::MatchType::EXACT);
    }

    Y_UNIT_TEST(GroupName) {
        THostName hostName(TStringBuf("SAS.000"));
        TLabelSelectors selectors;
        TSelectorsBuilder builder(selectors);
        builder.FromHostName(hostName);
        UNIT_ASSERT_VALUES_EQUAL(selectors.size(), 1);
        UNIT_ASSERT_VALUES_EQUAL(selectors.back().key(), GROUP_LABEL_KEY);
        UNIT_ASSERT_VALUES_EQUAL(selectors.back().pattern(), hostName.GetName());
        UNIT_ASSERT_VALUES_EQUAL(selectors.back().match_type(), yandex::solomon::model::MatchType::EXACT);
    }

    Y_UNIT_TEST(Signals) {
        TSignalName firstSignal(TStringBuf("something_summ"));
        TSignalName secondSignal(TStringBuf("other_summ"));
        TVector<TSignalName> signals{firstSignal, secondSignal};
        TLabelSelectors selectors;
        TSelectorsBuilder builder(selectors);
        builder.FromSignalNames(signals);
        UNIT_ASSERT_VALUES_EQUAL(selectors.size(), 1);
        UNIT_ASSERT_VALUES_EQUAL(selectors.back().key(), SIGNAL_LABEL_KEY);
        UNIT_ASSERT_VALUES_EQUAL(selectors.back().pattern(), TStringBuf("something_summ|other_summ"));
        UNIT_ASSERT_VALUES_EQUAL(selectors.back().match_type(), yandex::solomon::model::MatchType::GLOB);
    }

    Y_UNIT_TEST(RequestKey) {
        TRequestKey key(TRequestKey::FromString("itype=upper;ctype=prod,prestable;prj=p*"));
        TLabelSelectors selectors;
        TSelectorsBuilder builder(selectors);
        builder.FromRequestKey(key);
        UNIT_ASSERT_VALUES_EQUAL(selectors.size(), 2);
        UNIT_ASSERT_VALUES_EQUAL(selectors[0].key(), "ctype");
        UNIT_ASSERT_VALUES_EQUAL(selectors[0].pattern(), "prestable|prod");
        UNIT_ASSERT_VALUES_EQUAL(selectors[0].match_type(), yandex::solomon::model::MatchType::GLOB);
        UNIT_ASSERT_VALUES_EQUAL(selectors[1].key(), "prj");
        UNIT_ASSERT_VALUES_EQUAL(selectors[1].pattern(), "p*");
        UNIT_ASSERT_VALUES_EQUAL(selectors[1].match_type(), yandex::solomon::model::MatchType::GLOB);
    }
}
