#include "dataproxy_client.h"
#include "points.h"
#include "metrics.h"

#include <infra/monitoring/common/perf.h>

namespace NHistDb::NStockpile::NDataProxyClient {
    void FillLabelKeysRequest(const TString& projectId, const TString& selectors, TInstant from, TInstant to,
                              TLabelKeysCallState::TRequestType& requestToFill) {
        requestToFill.Setproject_id(projectId);
        requestToFill.Setselectors(selectors);
        requestToFill.Setfrom_millis(from.MilliSeconds());
        requestToFill.Setto_millis(to.MilliSeconds());
    }

    void FillLabelValuesRequest(const TString& projectId, const TString& selectors, const TSet<TString>& keys,
                                TInstant from, TInstant to, size_t limit,
                                TLabelValuesCallState::TRequestType& requestToFill) {
        requestToFill.Setproject_id(projectId);
        requestToFill.Setselectors(selectors);
        for (const auto& key : keys) {
            requestToFill.Addkeys(key);
        }
        requestToFill.Setfrom_millis(from.MilliSeconds());
        requestToFill.Setto_millis(to.MilliSeconds());
        requestToFill.Setlimit(limit);
    }

    void FillUniqueLabelsRequest(const TString& projectId, const TString& selectors, const TSet<TString>& keys,
                                 TUniqueLabelsCallState::TRequestType& requestToFill) {
        requestToFill.Setproject_id(projectId);
        requestToFill.Setselectors(selectors);
        for (const auto& key : keys) {
            requestToFill.Addkeys(key);
        }
    }

    TDataProxyHostSelector::TDataProxyHostSelector(TVector<TVector<TAtomicSharedPtr<NHistDb::NStockpile::TGrpcRemoteHost>>> hosts,
                                                   size_t clustersToRequest, size_t attemptsPerCluster)
        : Hosts(std::move(hosts))
        , AttemptsPerCluster(attemptsPerCluster) {
        ReshuffleClusters(clustersToRequest);
        InitDefaultTargetSequence();
    }

    TAtomicSharedPtr<NHistDb::NStockpile::TGrpcRemoteHost>
    TDataProxyHostSelector::GetHost(const TDataProxyHostSelector::TClusterAndHostIndices& hostIndex) const {
        return Hosts[hostIndex.first][hostIndex.second];
    }

    const TDataProxyHostSelector::TAttemptTargetSequence& TDataProxyHostSelector::GetDefaultTargetSequence() const {
        return DefaultTargetSequence;
    }

    void TDataProxyHostSelector::ReshuffleClusters(size_t clustersToKeep) {
        Shuffle(Hosts.begin(), Hosts.end());
        while (Hosts.size() > clustersToKeep) {
            Hosts.pop_back();
        }
    }

    void TDataProxyHostSelector::InitDefaultTargetSequence() {
        DefaultTargetSequence.reserve(Hosts.size() * AttemptsPerCluster);
        TVector<TVector<size_t>> perClusterHostIndices;
        for (size_t clusterIndex = 0; clusterIndex < Hosts.size(); ++clusterIndex) {
            perClusterHostIndices.emplace_back();
            auto& hostIndices = perClusterHostIndices.back();
            hostIndices.reserve(Hosts[clusterIndex].size());
            for (size_t i = 0; i < Hosts[clusterIndex].size(); ++i) {
                hostIndices.push_back(i);
            }
            Shuffle(hostIndices.begin(), hostIndices.end());
        }

        for (size_t inClusterIndex = 0; inClusterIndex < AttemptsPerCluster; ++inClusterIndex) {
            for (size_t clusterIndex = 0; clusterIndex < Hosts.size(); ++clusterIndex) {
                if (inClusterIndex < perClusterHostIndices[clusterIndex].size()) {
                    DefaultTargetSequence.emplace_back(clusterIndex, perClusterHostIndices[clusterIndex][inClusterIndex]);
                }
            }
        }

        if (DefaultTargetSequence.empty()) {
            throw yexception() << "No DataProxy hosts to request";
        }
    }
}
