#include "dataproxy_client.h"

#include <library/cpp/testing/unittest/registar.h>
#include <library/cpp/testing/unittest/tests_data.h>

using namespace NHistDb::NStockpile;
using namespace NHistDb::NStockpile::NDataProxyClient;

Y_UNIT_TEST_SUITE(TDataProxyHostSelectorTest) {
    Y_UNIT_TEST(TargetSequenceSizeIsCorrect) {
        TPortManager portManager;
        TStringStream url;
        url << "localhost:" << portManager.GetPort();
        auto dummyChannel = grpc::CreateChannel(url.Str(), grpc::InsecureChannelCredentials());

        TVector<TAtomicSharedPtr<TGrpcRemoteHost>> hosts1;
        hosts1.emplace_back(new TDataProxyRemoteHost(TString("host1.cluster1"), 1, dummyChannel));
        hosts1.emplace_back(new TDataProxyRemoteHost(TString("host2.cluster1"), 1, dummyChannel));
        hosts1.emplace_back(new TDataProxyRemoteHost(TString("host3.cluster1"), 1, dummyChannel));
        hosts1.emplace_back(new TDataProxyRemoteHost(TString("host4.cluster1"), 1, dummyChannel));
        hosts1.emplace_back(new TDataProxyRemoteHost(TString("host5.cluster1"), 1, dummyChannel));

        TVector<TAtomicSharedPtr<TGrpcRemoteHost>> hosts2;
        hosts2.emplace_back(new TDataProxyRemoteHost(TString("host1.cluster2"), 1, dummyChannel));
        hosts2.emplace_back(new TDataProxyRemoteHost(TString("host2.cluster2"), 1, dummyChannel));
        hosts2.emplace_back(new TDataProxyRemoteHost(TString("host3.cluster2"), 1, dummyChannel));

        TDataProxyHostSelector::TClusterHosts clusterHosts({hosts1, hosts2});

        TVector<size_t> attemptsPerClusterToTest({1, 2, 3});
        for (auto attemptsPerCluster: attemptsPerClusterToTest) {
            TDataProxyHostSelector hostSelector(clusterHosts, 2, attemptsPerCluster);
            THashSet<TString> seenHosts;
            for (auto hostId: hostSelector.GetDefaultTargetSequence()) {
                seenHosts.insert(hostSelector.GetHost(hostId)->GetHost());
            }
            UNIT_ASSERT_VALUES_EQUAL(attemptsPerCluster * clusterHosts.size(), seenHosts.size());
        }

        THashSet<TString> totalSeenHosts;
        size_t maxCycles = 100; // multiple cases to see that different hosts are selected
        size_t attemptsPerCluster = 3;
        for (size_t i = 0; i < maxCycles && (totalSeenHosts.size() <= attemptsPerCluster * clusterHosts.size()); ++i) {
            TDataProxyHostSelector hostSelector(clusterHosts, 2, attemptsPerCluster);
            for (auto hostId: hostSelector.GetDefaultTargetSequence()) {
                totalSeenHosts.insert(hostSelector.GetHost(hostId)->GetHost());
            }
        }
        UNIT_ASSERT(totalSeenHosts.size() > attemptsPerCluster * clusterHosts.size());
    }

    Y_UNIT_TEST(AttemptsNumberBiggerThanNumberOfHosts) {
        TPortManager portManager;
        TStringStream url;
        url << "localhost:" << portManager.GetPort();
        auto dummyChannel = grpc::CreateChannel(url.Str(), grpc::InsecureChannelCredentials());

        TVector<TAtomicSharedPtr<TGrpcRemoteHost>> hosts1;
        hosts1.emplace_back(new TDataProxyRemoteHost(TString("host1.cluster1"), 1, dummyChannel));

        TVector<TAtomicSharedPtr<TGrpcRemoteHost>> hosts2;
        hosts2.emplace_back(new TDataProxyRemoteHost(TString("host1.cluster2"), 1, dummyChannel));
        hosts2.emplace_back(new TDataProxyRemoteHost(TString("host2.cluster2"), 1, dummyChannel));

        TDataProxyHostSelector::TClusterHosts clusterHosts({hosts1, hosts2});

        TDataProxyHostSelector hostSelector(clusterHosts, 2, 2);
        THashSet<TString> seenHosts;
        auto& defaultSequence = hostSelector.GetDefaultTargetSequence();
        UNIT_ASSERT_VALUES_EQUAL(3, defaultSequence.size());
    }

    Y_UNIT_TEST(EmptyCluster) {
        TPortManager portManager;
        TStringStream url;
        url << "localhost:" << portManager.GetPort();
        auto dummyChannel = grpc::CreateChannel(url.Str(), grpc::InsecureChannelCredentials());

        TVector<TAtomicSharedPtr<TGrpcRemoteHost>> hosts1;

        TVector<TAtomicSharedPtr<TGrpcRemoteHost>> hosts2;
        hosts2.emplace_back(new TDataProxyRemoteHost(TString("host1.cluster2"), 1, dummyChannel));
        hosts2.emplace_back(new TDataProxyRemoteHost(TString("host2.cluster2"), 1, dummyChannel));

        TDataProxyHostSelector::TClusterHosts clusterHosts({hosts1, hosts2});

        TDataProxyHostSelector hostSelector(clusterHosts, 2, 2);
        THashSet<TString> seenHosts;
        auto& defaultSequence = hostSelector.GetDefaultTargetSequence();
        UNIT_ASSERT_VALUES_EQUAL(2, defaultSequence.size());
    }

    Y_UNIT_TEST(EmptyClusters) {
        TPortManager portManager;
        TStringStream url;
        url << "localhost:" << portManager.GetPort();
        auto dummyChannel = grpc::CreateChannel(url.Str(), grpc::InsecureChannelCredentials());

        TVector<TAtomicSharedPtr<TGrpcRemoteHost>> hosts1;
        TVector<TAtomicSharedPtr<TGrpcRemoteHost>> hosts2;
        TDataProxyHostSelector::TClusterHosts clusterHosts({hosts1, hosts2});

        UNIT_ASSERT_EXCEPTION(TDataProxyHostSelector(clusterHosts, 2, 2), yexception);
    }
}

namespace {
    struct TCallInfo {
        TVector<TString> HostsCalled;
        size_t FailFirstHosts;
    };

    struct TTestGrpcHandler {
        using TCall = NHistDb::NStockpile::TGrpcAsyncCallState<decltype(&TDataProxyService::Stub::PrepareAsyncLabelKeys)>;
        TTestGrpcHandler(TCallInfo& info, NMonitoring::TRequestLog&)
            : Info(info) {

        }

        size_t GetCallsInFlight() const {
            return 0;
        }

        bool WaitAsync(TDuration) {
            return false;
        }

        void Wait() {
        }

        TCall& Execute(TCall& call, TGrpcState& state) {
            auto host = call.GetRemoteHost().GetHost();
            Info.HostsCalled.push_back(host);
            if (Info.HostsCalled.size() <= Info.FailFirstHosts) {
                state.MarkAs(TGrpcState::RETRIABLE_FAILURE);
            } else {
                state.MarkAs(TGrpcState::FINISHED);
            }
            return call;
        }

        TCallInfo& Info;
    };

    struct TTestGrpcHandlerFactory {
        using THandlerType = TTestGrpcHandler;
        THandlerType operator()(NMonitoring::TRequestLog& logger) {
            return THandlerType(*Info, logger);
        }
        TAtomicSharedPtr<TCallInfo> Info;
    };
}

Y_UNIT_TEST_SUITE(TDataProxyRequesterTest) {
    Y_UNIT_TEST(RetriesRequest) {
        TPortManager portManager;
        TStringStream url;
        url << "localhost:" << portManager.GetPort();
        auto dummyChannel = grpc::CreateChannel(url.Str(), grpc::InsecureChannelCredentials());

        TVector<TAtomicSharedPtr<TGrpcRemoteHost>> hosts;
        hosts.emplace_back(new TDataProxyRemoteHost(TString("host1.cluster1"), 1, dummyChannel));
        hosts.emplace_back(new TDataProxyRemoteHost(TString("host2.cluster1"), 1, dummyChannel));
        TDataProxyHostSelector::TClusterHosts clusterHosts({hosts});

        TLog dummyLogger;
        NMonitoring::TRequestLog requestLog(dummyLogger, "");
        TTestGrpcHandlerFactory factory;
        factory.Info.Reset(new TCallInfo());
        factory.Info->FailFirstHosts = 1; // fail first request to trigger second one

        auto requester = TDataProxyRequester(
            &TDataProxyService::Stub::PrepareAsyncLabelKeys,
            clusterHosts,
            "",
            1,
            2,
            requestLog,
            factory);

        const TString projectId = "RequestRetryTestProjectId";
        size_t correctProjectIdCount = 0;
        auto& requestProto = requester.PrepareCall([&projectId, &correctProjectIdCount](auto& callState, const auto*){
            if (callState.GetCall().GetRequest().Getproject_id() == projectId) {
                ++correctProjectIdCount;
            }
        });
        requestProto.Setproject_id(projectId);

        auto callsFailed = requester.Request();
        UNIT_ASSERT_VALUES_EQUAL(0, callsFailed);
        UNIT_ASSERT_VALUES_EQUAL(1, correctProjectIdCount); // callback is called only once, user-filled proto is not lost
        UNIT_ASSERT_VALUES_EQUAL(2, factory.Info->HostsCalled.size());

        factory.Info.Reset(new TCallInfo());
        factory.Info->FailFirstHosts = 2;

        auto requester2 = TDataProxyRequester(
            &TDataProxyService::Stub::PrepareAsyncLabelKeys,
            clusterHosts,
            "",
            1,
            2,
            requestLog,
            factory);
        const TString projectId2 = "RequestRetryTestProjectId2";
        size_t correctProjectIdCount2 = 0;
        auto& requestProto2 = requester2.PrepareCall([&projectId2, &correctProjectIdCount2](auto& callState, const auto*){
            if (callState.GetCall().GetRequest().Getproject_id() == projectId2) {
                ++correctProjectIdCount2;
            }
        });
        requestProto2.Setproject_id(projectId2);
        auto callsFailed2 = requester2.Request();
        UNIT_ASSERT_VALUES_EQUAL(1, callsFailed2);
        UNIT_ASSERT_VALUES_EQUAL(1, correctProjectIdCount2); // callback is called only once, user-filled proto is not lost
        UNIT_ASSERT_VALUES_EQUAL(2, factory.Info->HostsCalled.size());
    }
}
