#include "metabase_client.h"
#include "points.h"
#include "metrics.h"

#include <infra/monitoring/common/perf.h>

using namespace NHistDb::NStockpile;

namespace {
    static constexpr TDuration METABASE_TIMEOUT = TDuration::Minutes(2);

    template <typename TMethod>
    auto MetabaseCallState(TMethod method, TAtomicSharedPtr<TGrpcRemoteHost> host) -> TGrpcAsyncCallState<decltype(method)> {
        auto state = GrpcAsyncCallState(method, host, EGrpcRetryModeFlags::ON_DEADLINE);
        state.SetDeadline(METABASE_TIMEOUT);
        return state;
    }
}

TAtomicSharedPtr<TGrpcRemoteHost> TStockpileMetabaseClient::GetHost() const {
    return Host;
}

TStockpileMetabaseClient::TResolveManyCallState TStockpileMetabaseClient::ResolveManySensor(
    const TLabels& commonLabels, const TVector<TLabels>& sensorLabels) {
    auto state(MetabaseCallState(&TMetabaseService::Stub::PrepareAsyncResolveMany, Host));

    auto& request(state.GetRequest());
    CopyLabels(commonLabels, request.MutableCommonLabels());
    request.MutableListLabels()->Reserve(sensorLabels.size());
    for (const auto& labels : sensorLabels) {
        CopyLabels(labels, request.MutableListLabels()->Add()->mutable_labels());
    }
    request.SetDeadlineMillis(ToDeadlineMillis(METABASE_TIMEOUT));

    return state;
}

TStockpileMetabaseClient::TCreateManyCallState TStockpileMetabaseClient::CreateManySensor(
    const TLabels& commonLabels,
    const TVector<TLabels>& sensorLabels,
    const TVector<yandex::solomon::model::MetricType>& sensorTypes) {
    auto state(MetabaseCallState(&TMetabaseService::Stub::PrepareAsyncCreateMany, Host));

    auto& request(state.GetRequest());
    CopyLabels(commonLabels, request.MutableCommonLabels());

    Y_VERIFY(sensorLabels.size() == sensorTypes.size());
    auto labelsIt(sensorLabels.begin());
    auto typesIt(sensorTypes.begin());
    request.mutable_metrics()->Reserve(sensorLabels.size());
    for (; labelsIt != sensorLabels.end() && typesIt != sensorTypes.end(); ++labelsIt, ++typesIt) {
        CopySensor(*labelsIt, *typesIt, request.mutable_metrics()->Add());
    }

    request.SetCreatedAtMillis(TInstant::Now().MilliSeconds());
    request.SetDeadlineMillis(ToDeadlineMillis(METABASE_TIMEOUT));

    return state;
}

TStockpileMetabaseClient::TFindCallState TStockpileMetabaseClient::FindSensor(const TLabelSelectors& selectors, size_t offset, long limit) {
    auto state(MetabaseCallState(&TMetabaseService::Stub::PrepareAsyncFind, Host));

    auto& request(state.GetRequest());
    for (const auto& selector : selectors) {
        request.AddSelectors()->CopyFrom(selector);
    }
    request.MutableSliceOptions()->SetOffset(offset);
    request.MutableSliceOptions()->SetLimit(limit);
    request.SetDeadlineMillis(ToDeadlineMillis(METABASE_TIMEOUT));

    return state;
}

TStockpileMetabaseClient::TServerStatusCallState TStockpileMetabaseClient::ServerStatus() {
    TServerStatusCallState state(MetabaseCallState(&TMetabaseService::Stub::PrepareAsyncServerStatus, Host));
    state.GetRequest().SetDeadlineMillis(ToDeadlineMillis(METABASE_TIMEOUT));
    state.GetContext().set_compression_algorithm(GRPC_COMPRESS_NONE);
    return state;
}

void TStockpileMetabaseClient::CopyLabels(const TLabels& source, ::google::protobuf::RepeatedPtrField<yandex::solomon::model::Label>* dest) {
    dest->Reserve(source.size());
    for (const auto& item : source) {
        auto* label(dest->Add());
        label->set_key(item.Key);
        label->set_value(item.Value);
    }
}

void TStockpileMetabaseClient::CopySensor(const TLabels& labels, yandex::solomon::model::MetricType type, yandex::solomon::metabase::Metric* metric) {
    if (type == yandex::solomon::model::MetricType::METRIC_TYPE_UNSPECIFIED) {
        ythrow TTypeNotSupportedException() << "unsupported metric type: " << type;
    }
    metric->set_type(type);
    CopyLabels(labels, metric->mutable_labels());
}

TMetabaseStatusState::TMetabaseStatusState(TGrpcCompletionQueue& queue, TAtomicSharedPtr<TGrpcRemoteHost> host, TLog& log)
    : Client(std::move(host), log)
    , Log(log)
    , CallState(Client.ServerStatus())
{
    queue.Execute(CallState, *this);
}

void TMetabaseStatusState::Handle() {
    const auto& statusResponse(CallState.GetResponse());
    if (statusResponse.GetStatus() != yandex::solomon::metabase::EMetabaseStatusCode::OK) {
        ythrow yexception() << "Invalid Metabase " << *Client.GetHost() << " ServerStatus response: \""
                            << statusResponse.GetStatusMessage() << "\" (" << statusResponse.GetStatus() << ")";
    }

    Shards.reserve(statusResponse.GetPartitionStatus().size());
    for (const auto& shardStatus : statusResponse.GetPartitionStatus()) {
        try {
            auto shardKey = TMetabaseShardKey::Make(shardStatus.GetLabels());
            if (shardKey.IsYasmShard()) {
                Shards.emplace_back(TShardStatus{
                    .Key = shardKey,
                    .NumId = shardStatus.GetNumId(),
                    .Ready = shardStatus.GetReady(),
                    .ReadyWrite = shardStatus.GetReadyWrite() && shardStatus.Getmetric_count() < shardStatus.Getmetric_limit()
                });
            }
        } catch (...) {
            Log << TLOG_ERR << CurrentExceptionMessage();
        }
    }
}

TVector<TMetabaseStatusState::TShardStatus> TMetabaseStatusState::GetResult() const {
    CallState.Check();
    return Shards;
}

TAtomicSharedPtr<TGrpcRemoteHost> TMetabaseStatusState::GetHost() const {
    return Client.GetHost();
}
