#include "metabase_shard.h"

#include <cmath>

#include <infra/yasm/stockpile_client/metrics.h>
#include <infra/monitoring/common/perf.h>

using namespace NHistDb::NStockpile;
using namespace yandex::solomon;

namespace {
    void ThrowFailureException(metabase::EMetabaseStatusCode status, const TString& message) {
        using metabase::EMetabaseStatusCode;
        switch (status) {
            case EMetabaseStatusCode::DEADLINE_EXCEEDED:
            case EMetabaseStatusCode::SHARD_NOT_READY:
            case EMetabaseStatusCode::INTERNAL_ERROR:
                ythrow TRetriableFailure() << message;
            default:
                ythrow yexception() << message;
        }
    }
}

const TMetabaseShardKey& TMetabaseShard::GetShardKey() const {
    return ShardKey;
}

TNumId TMetabaseShard::GetShardNumId() const {
    return ShardNumId;
}

TMaybe<TSensorId> TMetabaseShard::FindSensor(const TSeriesKey& seriesKey) {
    TLightReadGuard rg(SensorCacheLock);
    return FindUnsafe(seriesKey);
}

TVector<TMaybe<TSensorId>> TMetabaseShard::FindSensors(const TVector<TSeriesKey>& seriesKeys) {
    TLightReadGuard rg(SensorCacheLock);
    TVector<TMaybe<TSensorId>> result(Reserve(seriesKeys.size()));
    for (const auto& key : seriesKeys) {
        result.push_back(FindUnsafe(key));
    }
    return result;
}

TMaybe<TSensorId> TMetabaseShard::FindSensorWithTypeCheck(const TTypedSeriesKey& typedSeriesKey,
                                                          const TSensorTypeMap& conversionsToExclude) {
    TLightReadGuard rg(SensorCacheLock);
    return FindUnsafeWithTypeCheck(typedSeriesKey, conversionsToExclude);
}

TVector<TMaybe<TSensorId>> TMetabaseShard::FindSensorsWithTypeCheck(const TVector<TTypedSeriesKey>& typedSeriesKeys,
                                                                    const TSensorTypeMap& conversionsToExclude) {
    TLightReadGuard rg(SensorCacheLock);
    TVector<TMaybe<TSensorId>> result(Reserve(typedSeriesKeys.size()));
    for (const auto& key : typedSeriesKeys) {
        result.push_back(FindUnsafeWithTypeCheck(key, conversionsToExclude));
    }
    return result;
}

bool TMetabaseShard::ContainsInRejection(const TSeriesKey& seriesKey, TInstant now) {
    TLightReadGuard rg(SensorCacheLock);
    const auto it = SensorRejectionCache.find(seriesKey);
    if (it == SensorRejectionCache.end()) {
        return false;
    }

    return it->second > now;
}

void TMetabaseShard::SaveSensors(const TVector<std::pair<TSeriesKey, TSensorId>>& seriesToSensors) {
    TLightWriteGuard wg(SensorCacheLock);
    for (const auto & [ key, sensorId ] : seriesToSensors) {
        auto& record = SensorCache.emplace(key, key).first->second;
        record.SensorId = sensorId; // could be different or undefined, so assign it
        record.Unlink();
        SensorCacheRecordsList.PushBack(&record);
    }
}

void TMetabaseShard::SaveRejectedSensors(const TVector<std::pair<TSeriesKey, TInstant>>& seriesToRejectTime) {
    TLightWriteGuard wg(SensorCacheLock);
    for (const auto & [ key, time ] : seriesToRejectTime) {
        SensorRejectionCache[key] = time;
    }
}

void TMetabaseShard::ClearSensorCache(TDuration fullRefreshInterval, TDuration minTimeSinceLastCleanup, TMaybe<TInstant> now) {
    TLightWriteGuard wg(SensorCacheLock);
    if (!now.Defined()) {
        now = TInstant::Now();
    }
    if (LastCleanupAt + minTimeSinceLastCleanup <= now.GetRef() and fullRefreshInterval > TDuration::Zero()) {
        double fractionToClear = (now.GetRef() - LastCleanupAt) / fullRefreshInterval;
        auto toClear = static_cast<size_t>(std::ceil(fractionToClear * static_cast<double>(SensorCache.size())));

        while (toClear && !SensorCacheRecordsList.Empty()) {
            auto key = SensorCacheRecordsList.Front()->Key;
            SensorCacheRecordsList.PopFront();
            SensorCache.erase(key);
            --toClear;
        }

        for (auto it = SensorRejectionCache.begin(); it != SensorRejectionCache.end();) {
            auto prev = it++;
            if (now.GetRef() >= prev->second) {
                SensorRejectionCache.erase(prev);
            }
        }

        LastCleanupAt = now.GetRef();
    }
}

void TMetabaseShard::ClearRejectedSensors() {
    TLightWriteGuard wg(SensorCacheLock);
    SensorRejectionCache.clear();
}

TMaybe<TSensorId> TMetabaseShard::FindUnsafe(const TSeriesKey& cacheKey) const {
    const auto it = SensorCache.find(cacheKey);
    if (it != SensorCache.end()) {
        return it->second.SensorId;
    } else {
        return Nothing();
    }
}

TMaybe<TSensorId> TMetabaseShard::FindUnsafeWithTypeCheck(const TTypedSeriesKey& typedSeriesKey,
                                                          const TSensorTypeMap& conversionsToExclude) const {
    auto result = FindUnsafe(typedSeriesKey.SeriesKey);
    if (result && typedSeriesKey.Type != result->Type && typedSeriesKey.Type != model::MetricType::METRIC_TYPE_UNSPECIFIED) {
        // type conversion would be required to use in-cache sensor for the specified series key
        auto it = conversionsToExclude.find(result->Type);
        if (it != conversionsToExclude.end() && it->second == typedSeriesKey.Type) {
            // conversion from result->Type to typedSeriesKey.Type is specified in conversionsToExclude, so ignore such sensor
            result.Clear();
        }
    }
    return result;
}

TMetabaseResolveSensor::TMetabaseResolveSensor(
    const TVector<TTypedSeriesKey>& seriesKeys,
    const TAtomicSharedPtr<TMetabaseShard>& shard,
    const TAtomicSharedPtr<TGrpcRemoteHost>& metabaseHost,
    TGrpcCompletionQueue& queue,
    TLog& log,
    bool readOnly,
    const TSensorTypeMap* typeMismatchFixTable)
    : Shard(shard)
    , CompletionQueue(queue)
    , Client(metabaseHost, log)
    , Log(log)
    , ReadOnly(readOnly)
    , TypeMismatchFixTable(typeMismatchFixTable)
    , Step(RESOLVING)
    , QuotaExceeded(false)
    , SensorTypeChanges(0)
    , TypeMismatchCount(0)
{
    SeriesKeys.reserve(seriesKeys.size());
    SeriesTypes.reserve(seriesKeys.size());
    Result.reserve(seriesKeys.size());
    KeysToResolve.reserve(seriesKeys.size());
    ResolvedSensors.reserve(seriesKeys.size());

    for (const auto& typedKey : seriesKeys) {
        if (typedKey.Type == model::MetricType::METRIC_TYPE_UNSPECIFIED) {
            ythrow yexception() << "key " << typedKey.SeriesKey << " has no type";
        }
        SeriesKeys.emplace_back(typedKey.SeriesKey);
        SeriesTypes.emplace_back(typedKey.Type);
        KeysToResolve[typedKey.SeriesKey].emplace_back(Result.size());
        Result.emplace_back();
    }

    Y_VERIFY(!KeysToResolve.empty());
    Y_VERIFY(seriesKeys.size() >= KeysToResolve.size());

    Handle();

    Y_UNUSED(Log);
}

void TMetabaseResolveSensor::Handle() {
    switch (Step) {
        case RESOLVING: {
            return HandleResolving();
        }
        case CREATING: {
            return HandleCreating();
        }
        case FINISHING: {
            return HandleFinishing();
        }
        case DONE: {
            ythrow yexception() << "can't finish state twice";
        }
    };
}

TVector<TSensorId> TMetabaseResolveSensor::GetResult() const {
    if (Step != DONE) {
        ythrow yexception() << "state not ready";
    }
    if (ReadOnly) {
        ythrow yexception() << "GetResult is called on ReadOnly request";
    }
    TVector<TSensorId> strictResult(Reserve(Result.size()));
    for (const auto& maybeItem: Result) {
        strictResult.emplace_back(maybeItem.GetRef());
    }
    return strictResult;
}

TVector<TMaybe<TSensorId>> TMetabaseResolveSensor::GetRawResult() const {
    return Result;
}

void TMetabaseResolveSensor::PrepareMetabaseRequest() {
    CommonLabels.clear();
    SensorLabels.clear();
    SensorTypes.clear();

    Shard->GetShardKey().FillLabels(CommonLabels);
    SensorLabels.reserve(KeysToResolve.size());
    SensorTypes.reserve(KeysToResolve.size());

    for (const auto & [ key, indices ] : KeysToResolve) {
        key.FillLabels(SensorLabels.emplace_back());
        SensorTypes.emplace_back(SeriesTypes[indices.front()]);
    }
}

TMaybe<model::MetricType> TMetabaseResolveSensor::GetMismatchFixCandidateFor(model::MetricType sensorType) {
    TMaybe<model::MetricType> typeFix;
    if (TypeMismatchFixTable) {
        auto typeIt = TypeMismatchFixTable->find(sensorType);
        if (typeIt != TypeMismatchFixTable->end()) {
            typeFix = typeIt->second;
        }
    }
    return typeFix;
}

bool TMetabaseResolveSensor::HandleTypeChange(const TSeriesKey& key, const TSensorId& sensorId) {
    const auto it = KeysToResolve.find(key);
    if (it == KeysToResolve.end()) {
        return false;
    } else {
        // check if resolved sensor's type is different from the one that we need
        bool typeMismatch = false;
        for (const auto index : it->second) {
            if (SeriesTypes[index] != sensorId.Type) {
                typeMismatch = true;
                ++TypeMismatchCount;
            }
        }
        if (!typeMismatch) {
            return false;
        }

        // check if the type change from returned sensor type to the requested type is permitted
        TMaybe<model::MetricType> typeFix = GetMismatchFixCandidateFor(sensorId.Type);

        if (!typeFix.Defined()) {
            return false;
        }

        for (const auto index : it->second) {
            if (sensorId.Type != SeriesTypes[index] && typeFix != SeriesTypes[index]) {
                return false;  // does not help, do nothing
            }
        }

        ++SensorTypeChanges;
        for (const auto index : it->second) {
            SeriesTypes[index] = *typeFix;
        }
        return true;
    }
}

void TMetabaseResolveSensor::MarkSensorAsResolved(const TSeriesKey& key, const TSensorId& sensorId) {
    const auto it = KeysToResolve.find(key);
    if (it == KeysToResolve.end()) {
        ythrow yexception() << "Metabase return sensor " << key << " we didn't ask";
    }
    for (const auto index : it->second) {
        Result[index] = sensorId;
    }
    ResolvedSensors.emplace_back(it->first, sensorId);
    KeysToResolve.erase(it);
}

void TMetabaseResolveSensor::MarkSensorAsRejected() {
    auto rejectUntil = TInstant::Now() + TDuration::Hours(5);
    TVector<std::pair<TSeriesKey, TInstant>> rejectedSensors;
    rejectedSensors.reserve(KeysToResolve.size());
    for (const auto & [ key, _ ] : KeysToResolve) {
        rejectedSensors.push_back({key, rejectUntil});
    }
    Shard->SaveRejectedSensors(rejectedSensors);
    KeysToResolve.clear();
}

void TMetabaseResolveSensor::HandleResolving() {
    PrepareMetabaseRequest();

    ResolveCallState.ConstructInPlace(Client.ResolveManySensor(CommonLabels, SensorLabels));
    ResolveCallState->GetContext().set_compression_algorithm(GRPC_COMPRESS_NONE);
    CompletionQueue.Execute(*ResolveCallState, *this);

    Step = CREATING;
}

void TMetabaseResolveSensor::HandleCreating() {
    const auto& response = ResolveCallState->GetResponse();
    if (response.GetStatus() != metabase::EMetabaseStatusCode::OK) {
        TString message = TStringBuilder() << "Invalid ResolveManySensor response: \""
                                           << response.GetStatusMessage() << "\" (" << response.GetStatus() << ") "
                                           << " ShardId=" << Shard->GetShardKey();
        ThrowFailureException(response.GetStatus(), message);
    }

    for (const auto& metric : response.metrics()) {
        const auto seriesKey = TSeriesKey::Make(metric.labels());
        const auto sensorId = TSensorId(metric);
        if (!HandleTypeChange(seriesKey, sensorId)) {
            MarkSensorAsResolved(seriesKey, sensorId);
        }
    }

    if (!ResolvedSensors.empty()) {
        Shard->SaveSensors(ResolvedSensors);
        ResolvedSensors.clear();
    }

    if (KeysToResolve.empty()) {
        Step = DONE;
    } else if (ReadOnly) {
        // skip creation step
        MarkSensorAsRejected();
        Step = DONE;
    } else {
        PrepareMetabaseRequest();

        CreateCallState.ConstructInPlace(Client.CreateManySensor(CommonLabels, SensorLabels, SensorTypes));
        CreateCallState->GetContext().set_compression_algorithm(GRPC_COMPRESS_NONE);
        CompletionQueue.Execute(*CreateCallState, *this);

        Step = FINISHING;
    }
}

void TMetabaseResolveSensor::HandleFinishing() {
    const auto& response(CreateCallState->GetResponse());
    if (response.GetStatus() != metabase::EMetabaseStatusCode::OK) {
        TString message = TStringBuilder() << "Invalid CreateManySensor response: \""
                                           << response.GetStatusMessage() << "\" (" << response.GetStatus() << ") "
                                           << " ShardId=" << Shard->GetShardKey();
        if (response.GetStatus() == metabase::EMetabaseStatusCode::QUOTA_ERROR) {
            QuotaExceeded = true;
            MarkSensorAsRejected();
        }
        ThrowFailureException(response.GetStatus(), message);
    }

    for (const auto& metric : response.metrics()) {
        MarkSensorAsResolved(TSeriesKey::Make(metric.labels()), TSensorId(metric));
    }

    if (!KeysToResolve.empty()) {
        ythrow yexception() << "Metabase can't create some sensors";
    }

    Shard->SaveSensors(ResolvedSensors);
    Step = DONE;
}

TMetabaseFindSensor::TMetabaseFindSensor(
        const TAtomicSharedPtr<TMetabaseShard>& shard,
        const TAtomicSharedPtr<TGrpcRemoteHost>& metabaseHost,
        TLabelSelectors&& selectors,
        size_t offset,
        long limit,
        TLog& log
        )
        : Shard(shard)
        , MetabaseHost(metabaseHost)
        , Selectors(std::move(selectors))
        , Log(log)
        , Client(metabaseHost, Log)
        , Offset(offset)
        , Limit(limit)
        , ResultSize(0)
{
    CallState.ConstructInPlace(Client.FindSensor(Selectors, offset, limit));
}

THolder<TMetabaseFindSensor> TMetabaseFindSensor::Make(
    const NZoom::NHost::THostName& hostName,
    const NTags::TRequestKey& requestKey,
    const TVector<NZoom::NSignal::TSignalName>& signalNames,
    const TAtomicSharedPtr<TMetabaseShard>& shard,
    const TAtomicSharedPtr<TGrpcRemoteHost>& metabaseHost,
    TLog& log)
{
    return Make(hostName, requestKey, signalNames, shard, metabaseHost, DEFAULT_LIMIT, log);
}

THolder<TMetabaseFindSensor> TMetabaseFindSensor::Make(
    const NZoom::NHost::THostName& hostName,
    const NTags::TRequestKey& requestKey,
    const TVector<NZoom::NSignal::TSignalName>& signalNames,
    const TAtomicSharedPtr<TMetabaseShard>& shard,
    const TAtomicSharedPtr<TGrpcRemoteHost>& metabaseHost,
    long limit,
    TLog& log)
{
    TLabelSelectors selectors;
    shard->GetShardKey().FillSelectors(selectors);

    TSelectorsBuilder builder(selectors);
    builder.FromHostName(hostName);
    builder.FromRequestKey(requestKey);
    builder.FromSignalNames(signalNames);
    return THolder<TMetabaseFindSensor>(new TMetabaseFindSensor(shard, metabaseHost, std::move(selectors), 0, limit, log));
}

THolder<TMetabaseFindSensor> TMetabaseFindSensor::Make(
    const NZoom::NHost::THostName& hostName,
    const TAtomicSharedPtr<TMetabaseShard>& shard,
    const TAtomicSharedPtr<TGrpcRemoteHost>& metabaseHost,
    long limit,
    TLog& log)
{
    TLabelSelectors selectors;
    shard->GetShardKey().FillSelectors(selectors);

    TSelectorsBuilder builder(selectors);
    builder.FromHostName(hostName);

    return THolder<TMetabaseFindSensor>(new TMetabaseFindSensor(shard, metabaseHost, std::move(selectors), 0, limit, log));
}

THolder<TMetabaseFindSensor> TMetabaseFindSensor::Next() {
    THolder<TMetabaseFindSensor> result;
    if (IsSuccess() && Offset + ResultSize < TotalCount) {
        result.Reset(new TMetabaseFindSensor(Shard, MetabaseHost, std::move(Selectors), Offset + ResultSize, Limit, Log));
    }
    return result;
}

THolder<TMetabaseFindSensor> TMetabaseFindSensor::Repeat(const TAtomicSharedPtr<TMetabaseShard>& shard,
                                                         const TAtomicSharedPtr<TGrpcRemoteHost>& metabaseHost) {
    return THolder<TMetabaseFindSensor>(new TMetabaseFindSensor(shard, metabaseHost, std::move(Selectors), Offset, Limit, Log));
}

void TMetabaseFindSensor::Handle() {
    const auto& response(CallState->GetResponse());
    if (response.GetStatus() != metabase::EMetabaseStatusCode::OK) {
        TString message = TStringBuilder() << "Invalid Find response: \"" << response.GetStatusMessage()
                                           << "\" (" << response.GetStatus() << ") "
                                           << " ShardId=" << Shard->GetShardKey();
        ThrowFailureException(response.GetStatus(), message);
    }

    ResultSize = response.metrics_size();
    TotalCount = response.GetTotalCount();
    ResolvedSensors.reserve(ResultSize);
    for (const auto& metric : response.metrics()) {
        const auto key = TSeriesKey::Make(metric.labels());
        ResolvedSensors.emplace_back(key, metric);
    }

    if (!ResolvedSensors.empty()) {
        Shard->SaveSensors(ResolvedSensors);
    }
}

grpc::ClientContext* TMetabaseFindSensor::GetContext() {
    if (CallState) {
        return &CallState->GetContext();
    } else {
        return nullptr;
    }
}

void TMetabaseFindSensor::ScheduleForExecute(TGrpcCompletionQueue& queue) {
    queue.Execute(CallState.GetRef(), *this);
}

TVector<std::pair<TSeriesKey, TSensorId>> TMetabaseFindSensor::GetResult() {
    return std::move(ResolvedSensors);
}

template <>
void Out<TMetabaseShard>(IOutputStream& stream, TTypeTraits<TMetabaseShard>::TFuncParam shard) {
    stream << "TMetabaseShard{.ShardKey=" << shard.GetShardKey() << "}";
}
