#pragma once

#include <infra/yasm/stockpile_client/common/base_types.h>
#include "metabase_client.h"

#include <infra/yasm/common/labels/tags/instance_key.h>
#include <infra/yasm/common/labels/tags/request_key.h>
#include <infra/yasm/common/labels/signal/signal_name.h>
#include <infra/yasm/common/labels/host/host.h>

#include <library/cpp/logger/log.h>
#include <library/cpp/threading/light_rw_lock/lightrwlock.h>
#include <util/generic/queue.h>
#include <util/generic/intrlist.h>
#include <util/random/random.h>

namespace NHistDb::NStockpile {
    struct TTypedSeriesKey {
        TSeriesKey SeriesKey;
        yandex::solomon::model::MetricType Type;
    };

    using TSensorTypeMap = THashMap<yandex::solomon::model::MetricType, yandex::solomon::model::MetricType>;

    class TMetabaseShard {
    public:
        TMetabaseShard(TMetabaseShardKey shardKey, TNumId shardNumId, TInstant creationTime = TInstant::Now())
            : ShardKey(std::move(shardKey))
            , ShardNumId(shardNumId)
            , LastCleanupAt(creationTime) {
        }

        TMetabaseShard(TMetabaseShardKey shardKey, TInstant creationTime = TInstant::Now())
            : TMetabaseShard(std::move(shardKey), 0, creationTime) {
        }

        const TMetabaseShardKey& GetShardKey() const;
        TNumId GetShardNumId() const;

        /** Find sensor in cache. Note, that if sensor was not added previously by SaveSensors, it won't be found here. */
        TMaybe<TSensorId> FindSensor(const TSeriesKey& seriesKey);
        /** Find multiple sensors at once. */
        TVector<TMaybe<TSensorId>> FindSensors(const TVector<TSeriesKey>& seriesKeys);
        /** Find sensor and check it's type. If conversion from the type in cache to the type requested is
         *  specified in conversionsToExclude map then do not return such sensor. */
        TMaybe<TSensorId> FindSensorWithTypeCheck(const TTypedSeriesKey& typedSeriesKey,
                                                  const TSensorTypeMap& conversionsToExclude);
        /** Find multiple sensors at once and check their types. If conversion from the type in cache to the type requested is
         *  specified in conversionsToExclude map then do not return such sensor.*/
        TVector<TMaybe<TSensorId>> FindSensorsWithTypeCheck(const TVector<TTypedSeriesKey>& typedSeriesKeys,
                                                            const TSensorTypeMap& conversionsToExclude);
        /** Check sensor in rejection list */
        bool ContainsInRejection(const TSeriesKey& seriesKey, TInstant now);

        /**
         * Save sensors in cache. If there is an entry for the specified series key, then update it.
         * @param seriesToSensors key-value pairs to insert
         * @param now Current time. Do not pass it unless in tests.
         */
        void SaveSensors(const TVector<std::pair<TSeriesKey, TSensorId>>& seriesToSensors);

        void SaveRejectedSensors(const TVector<std::pair<TSeriesKey, TInstant>>& seriesToRejectTime);

        size_t SensorCacheSize() const {
            TLightReadGuard rg(SensorCacheLock);
            return SensorCache.size() + SensorRejectionCache.size();
        }

        /**
         * Clear oldest records in cache. Record's age is updated on write operation (SaveSensors).
         * Will delete a fraction of all records that is equal to (now - LastCleanupTimestamp) / fullCleanupInterval.
         * @param fullRefreshInterval desired interval during which, the cache will be fully refreshed.
         * @param minTimeSinceLastCleanup minimal time since last cleanup. If not enough time has passed since the last
         *                                cleanup then this function does nothing.
         * @param now current time. Pass nothing to detect current time inside the function.
         */
        void ClearSensorCache(TDuration fullRefreshInterval, TDuration minTimeSinceLastCleanup, TMaybe<TInstant> now = Nothing());

        void ClearRejectedSensors();

        TInstant GetLastCleanupTimestamp() const {
            TLightReadGuard rg(SensorCacheLock);
            return LastCleanupAt;
        }
    private:
        struct TSensorCacheRecord: public TIntrusiveListItem<TSensorCacheRecord> {
            explicit TSensorCacheRecord(TSeriesKey key)
                : Key(std::move(key))
                , SensorId() {
            }
            TSeriesKey Key;
            TMaybe<TSensorId> SensorId; // Undefined only if not inserted into cache.
        };

        using TSensorCacheRecordsList = TIntrusiveList<TSensorCacheRecord>;
        using TSensorIdCache = THashMap<TSeriesKey, TSensorCacheRecord>;
        using TSensorIdRejectionCache = THashMap<TSeriesKey, TInstant>;

        TMaybe<TSensorId> FindUnsafe(const TSeriesKey& key) const;
        TMaybe<TSensorId> FindUnsafeWithTypeCheck(const TTypedSeriesKey& typedSeriesKey,
                                                  const TSensorTypeMap& conversionsToExclude) const;

        TMetabaseShardKey ShardKey;
        TNumId ShardNumId;

        TLightRWLock SensorCacheLock;
        TSensorIdCache SensorCache;
        TSensorIdRejectionCache SensorRejectionCache;
        TSensorCacheRecordsList SensorCacheRecordsList;
        TInstant LastCleanupAt;
    };

    class TMetabaseResolveSensor final: public TGrpcState {
    public:
        TMetabaseResolveSensor(
            const TVector<TTypedSeriesKey>& seriesKeys,
            const TAtomicSharedPtr<TMetabaseShard>& shard,
            const TAtomicSharedPtr<TGrpcRemoteHost>& metabaseHost,
            TGrpcCompletionQueue& queue,
            TLog& log,
            bool readOnly = false,
            const TSensorTypeMap* typeMismatchFixTable = nullptr);

        void Handle() override;

        TVector<TSensorId> GetResult() const; // do not call when request is readOnly
        TVector<TMaybe<TSensorId>> GetRawResult() const; // can be called even if the request have failed on sensor create step
        bool IsReadOnly() const {
            return ReadOnly;
        }
        bool IsQuotaExceeded() const {
            return QuotaExceeded;
        }

        size_t GetSensorTypeChanges() const {
            return SensorTypeChanges;
        }

        size_t GetTypeMismatchCount() const {
            return TypeMismatchCount;
        }

        TStringBuf GetRequestName() const override {
            return NHistDb::NStockpile::NMetrics::STOCKPILE_METABASE_GRPC_SENSOR_RESOLVE;
        }

        void MarkAs(EExecutionStatus status) override {
            if (status == EExecutionStatus::FINISHED) {
                if (Step == DONE) {
                    TGrpcState::MarkAs(status);
                }
            } else {
                TGrpcState::MarkAs(status);
            }
        }

    private:
        enum EStep {
            RESOLVING,
            CREATING,
            FINISHING,
            DONE
        };

        void PrepareMetabaseRequest();
        bool HandleTypeChange(const TSeriesKey& key, const TSensorId& sensorId);
        void MarkSensorAsResolved(const TSeriesKey& key, const TSensorId& sensorId);
        void MarkSensorAsRejected();
        TMaybe<yandex::solomon::model::MetricType> GetMismatchFixCandidateFor(yandex::solomon::model::MetricType sensorType);

        void HandleResolving();
        void HandleCreating();
        void HandleFinishing();

        TVector<TSeriesKey> SeriesKeys;
        TVector<yandex::solomon::model::MetricType> SeriesTypes;
        TAtomicSharedPtr<TMetabaseShard> Shard;

        TGrpcCompletionQueue& CompletionQueue;
        TStockpileMetabaseClient Client;
        TLog& Log;
        const bool ReadOnly;
        const TSensorTypeMap* TypeMismatchFixTable;

        EStep Step;
        TMaybe<TStockpileMetabaseClient::TResolveManyCallState> ResolveCallState;
        TMaybe<TStockpileMetabaseClient::TCreateManyCallState> CreateCallState;

        TVector<TMaybe<TSensorId>> Result;
        TVector<std::pair<TSeriesKey, TSensorId>> ResolvedSensors;

        THashMap<TSeriesKey, TVector<size_t>> KeysToResolve;
        TLabels CommonLabels;
        TVector<TLabels> SensorLabels;
        TVector<yandex::solomon::model::MetricType> SensorTypes;

        bool QuotaExceeded;
        size_t SensorTypeChanges;
        size_t TypeMismatchCount;
    };

    class TMetabaseFindSensor final: public TGrpcState {
    public:
        static constexpr long DEFAULT_LIMIT = -1;

        static THolder<TMetabaseFindSensor> Make(
            const NZoom::NHost::THostName& hostName,
            const NTags::TRequestKey& requestKey,
            const TVector<NZoom::NSignal::TSignalName>& signalNames,
            const TAtomicSharedPtr<TMetabaseShard>& shard,
            const TAtomicSharedPtr<TGrpcRemoteHost>& metabaseHost,
            TLog& log);

        static THolder<TMetabaseFindSensor> Make(
            const NZoom::NHost::THostName& hostName,
            const NTags::TRequestKey& requestKey,
            const TVector<NZoom::NSignal::TSignalName>& signalNames,
            const TAtomicSharedPtr<TMetabaseShard>& shard,
            const TAtomicSharedPtr<TGrpcRemoteHost>& metabaseHost,
            long limit,
            TLog& log);

        static THolder<TMetabaseFindSensor> Make(
            const NZoom::NHost::THostName& hostName,
            const TAtomicSharedPtr<TMetabaseShard>& shard,
            const TAtomicSharedPtr<TGrpcRemoteHost>& metabaseHost,
            long limit,
            TLog& log);

        void Handle() override;

        const TAtomicSharedPtr<TMetabaseShard>& GetShard() const { return Shard; }

        TVector<std::pair<TSeriesKey, TSensorId>> GetResult();

        void ScheduleForExecute(TGrpcCompletionQueue& queue);

        TStringBuf GetRequestName() const override {
            return NHistDb::NStockpile::NMetrics::STOCKPILE_METABASE_GRPC_SENSOR_FIND;
        }

        grpc::ClientContext* GetContext() override;

        THolder<TMetabaseFindSensor> Next();
        THolder<TMetabaseFindSensor> Repeat(const TAtomicSharedPtr<TMetabaseShard>& shard,
                                            const TAtomicSharedPtr<TGrpcRemoteHost>& metabaseHost);

    private:
        TMetabaseFindSensor(
                const TAtomicSharedPtr<TMetabaseShard>& shard,
                const TAtomicSharedPtr<TGrpcRemoteHost>& metabaseHost,
                TLabelSelectors&& Selectors,
                size_t offset,
                long limit,
                TLog& log);
    private:
        TAtomicSharedPtr<TMetabaseShard> Shard;
        TAtomicSharedPtr<TGrpcRemoteHost> MetabaseHost;
        TLabelSelectors Selectors;
        TLog& Log;
        TStockpileMetabaseClient Client;

        TMaybe<TStockpileMetabaseClient::TFindCallState> CallState;
        TVector<std::pair<TSeriesKey, TSensorId>> ResolvedSensors;
        size_t Offset;
        long Limit;
        size_t TotalCount;
        size_t ResultSize;
    };
}
